package ru.yandex.webmaster3.worker.nca;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.core.type.TypeReference;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.nca.CertificateTransparencyLogService;
import ru.yandex.webmaster3.storage.nca.CtlogCertificatesCHDao;
import ru.yandex.webmaster3.storage.nca.DeserializerCertificateTransparencyLogs;
import ru.yandex.webmaster3.storage.nca.data.CertificateIdByDomainEntry;
import ru.yandex.webmaster3.storage.nca.data.ParsedCertificateState;
import ru.yandex.webmaster3.storage.nca.data.ParsedLogEntry;
import ru.yandex.webmaster3.storage.nca.data.UnparsedEntry;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;


/**
 * @author kravchenko99
 * @date 4/28/22
 */


@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class CertificateTransparencyLogParserTask extends PeriodicTask<CertificateTransparencyLogParserTask.TaskState> {

    private static final String WILDCARD_PREFIX = "*.";
    private static final int BATCH_SIZE = 30_000;
    private static final int PAGE_SIZE = 512;
    private static final Set<String> OPERATOR__WITH_NOTIFICATIONS = Set.of("Yandex");
    private final CertificateTransparencyLogService certificateTransparencyLogService;
    private final SettingsService settingsService;
    private final CtlogCertificatesCHDao ctlogCertificatesCHDao;
    private final WMCEventsService wmcEventsService;
    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;


    private static final TypeReference<Map<String, Integer>> typeReference = new TypeReference<>() {
    };

    @Override
    public Result run(UUID runId) throws Exception {
        List<Pair<String, String>> ctlogs = certificateTransparencyLogService.getYandexLogs();
        CommonDataState setting =
                settingsService.getSettingUncached(CommonDataType.LAST_RECORD_FROM_CTLOG);

        Map<String, Integer> lastRecordByCtlog;
        if (setting != null) {
            lastRecordByCtlog = JsonMapping.readValue(setting.getValue(), typeReference);
        } else {
            lastRecordByCtlog = new HashMap<>();
        }

        for (var p : ctlogs) {
            String ctlog = p.getKey();
            String operatorName = p.getValue();
            int logSize = certificateTransparencyLogService.getLogSize(ctlog);
            int lastRecord = lastRecordByCtlog.getOrDefault(ctlog, 0);
            if (lastRecord < logSize) {

                process(ctlog, lastRecord, logSize, OPERATOR__WITH_NOTIFICATIONS.contains(operatorName), lastRecordByCtlog);
            }
        }

        return new Result(TaskResult.SUCCESS);
    }

    private void process(String ctlog, int start, int end, boolean sendNotifications,
                         Map<String, Integer> lastRecordByCtlog) {

        List<CertificateIdByDomainEntry> dataToInsert = new ArrayList<>(end - start);
        int leftBorder = start;
        while (leftBorder < end) {
            int rightBorder = Math.min(leftBorder + PAGE_SIZE, end) - 1;
            // обе границы включены
            log.info("start read ctlog - {}, from - {}", ctlog, leftBorder);
            List<UnparsedEntry> entries = certificateTransparencyLogService.getEntries(ctlog, leftBorder, rightBorder);
            log.info("finish read ctlog - {}, and wsa get - {} records", ctlog, entries.size());

            int i = leftBorder - 1;
            for (UnparsedEntry entry : entries) {
                i++;
                ParsedLogEntry parsedLogEntry = DeserializerCertificateTransparencyLogs.parseLogEntry(entry);
                ParsedCertificateState parsedCertificateState =
                        DeserializerCertificateTransparencyLogs.parseCertificate(parsedLogEntry.getLogEntryType()
                                , parsedLogEntry.getCertificateX509());
                if (parsedCertificateState == null) {
                    continue;
                }
                dataToInsert.add(new CertificateIdByDomainEntry(
                        parsedCertificateState.getDomains(),
                        ctlog,
                        i,
                        parsedCertificateState.getNotBefore().getTime(),
                        parsedCertificateState.getSerialNumber(),
                        parsedLogEntry.getCertificateX509(),
                        parsedLogEntry.getLogEntryType()
                ));
            }
            //маловероятно что с нашими логами хоть когда-то зайдем сюда
            if (dataToInsert.size() > BATCH_SIZE) {
                processBatch(ctlog, leftBorder + entries.size(),
                        dataToInsert, sendNotifications, lastRecordByCtlog);
                dataToInsert.clear();
            }

            // некоторые логи не дают entries больше определенного кол-ва
            leftBorder += entries.size();
        }

        processBatch(ctlog, end, dataToInsert, sendNotifications, lastRecordByCtlog);
    }

    private void processBatch(String ctlog,
                              int processedCtlogLen,
                              List<CertificateIdByDomainEntry> dataToInsert,
                              boolean sendNotifications,
                              Map<String, Integer> lastRecordByCtlog) {
        ctlogCertificatesCHDao.insert(dataToInsert);
        lastRecordByCtlog.put(ctlog, processedCtlogLen);
        settingsService.update(CommonDataType.LAST_RECORD_FROM_CTLOG,
                JsonMapping.writeValueAsString(lastRecordByCtlog));
        if (!sendNotifications) {
            return;
        }
        var domainsWithCtlogsInfo = dataToInsert.stream()
                .flatMap(x -> x.getDomains().stream().map(domain -> Triple.of(domain, x.getCtlog(), x.getCtlogRow())))
                .toList();

        Set<Triple<WebmasterHostId, String, Integer>> hostsForNotification = new HashSet<>();
        allVerifiedHostsCacheService.foreachHost(hostId -> {
            for (var trip : domainsWithCtlogsInfo) {
                String domain = trip.getLeft();
                if (suitableDomain(hostId, domain)) {
                    hostsForNotification.add(Triple.of(hostId, trip.getMiddle(), trip.getRight()));
                }
            }
        });

        List<WMCEvent> wmcEvents = hostsForNotification.stream()
                .map(x -> new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                        x.getLeft(),
                        new MessageContent.NcaCtlogCertificate(x.getLeft(), x.getMiddle(), x.getRight()),
                        NotificationType.NCA_CERTIFICATE,
                        true)
                ))
                .map(WMCEvent::create)
                .toList();

        wmcEventsService.addEvents(wmcEvents);
    }

    static boolean suitableDomain(WebmasterHostId hostId, String domain) {
        if (hostId.getSchema() != WebmasterHostId.Schema.HTTPS){
            return false;
        }
        String verifiedDomain = hostId.getPunycodeHostname();
        if (verifiedDomain.equals(domain)) {
            return true;
        }
        if (domain.endsWith("." + verifiedDomain)) {
            return true;
        }
        if (domain.startsWith(WILDCARD_PREFIX) && verifiedDomain.endsWith(domain.substring(1))) {
            return true;
        }
        return false;
    }

    @Getter
    public static class TaskState implements PeriodicTaskState {

    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.PARSER_CERTIFICATE_TRANSPARENCY_LOGS;
    }

    @Override
    public TaskSchedule getSchedule() {

        return TaskSchedule.startByCron("0 27 3/5 * * *");
    }
}
