package ru.yandex.webmaster3.worker.nca;

import java.util.Map;
import java.util.UUID;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.nca.NcaRulesYDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.util.ydb.YdbYqlService;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.host.CommonDataType.NCA_RULES_LAST_IMPORT;


/**
 * @author kravchenko99
 * @date 3/22/22
 */


@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class ImportDomainsWithNcaCertificatesTask extends PeriodicTask<ImportDomainsWithNcaCertificatesTask.TaskState> {

    @Value("${webmaster3.worker.nca.antispam.domains_list_with_tls}")
    private final YtPath tablePath;
    private final YtService ytService;
    private final SettingsService settingsService;
    private final NcaRulesYDao ncaRulesYDao;
    private final YdbYqlService ydbYqlService;



    public static final String PREFIX_QUERY = "$CutWWWM = Re2::Capture(\"^(www\\\\.|m\\\\.)?(.+)$\");";
    public static final String DATA_SELECT_QUERY = """
            SELECT
                cast(${CURRENT_TIMESTAMP} * 1000 as Timestamp) as `time_insert`,
                cast(mask as Utf8) as `rule`,
                cast(change_ts * 1000000 as Timestamp) as `add_timestamp`,
            FROM ${SOURCE_TABLE}
            """;

    @Override
    public Result run(UUID runId) throws Exception {
        ytService.inTransaction(tablePath).execute(cypressService -> {
            YtNode node = cypressService.getNode(tablePath);
            if (node == null) {
                return false;
            }
            final DateTime modificationTime = new DateTime(node.getNodeMeta()
                    .get("modification_time")
                    .asText());
            CommonDataState settings = settingsService.getSettingUncached(NCA_RULES_LAST_IMPORT);
            if (settings != null && new DateTime(settings.getValue()).equals(modificationTime)) {
                return false;
            }
            StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                    "CURRENT_TIMESTAMP", String.valueOf(modificationTime.getMillis()),
                    "SOURCE_TABLE", tablePath.toYqlPath()
            ));
            ydbYqlService.importToYdb(ncaRulesYDao.getTablePath(), substitutor.replace(DATA_SELECT_QUERY), PREFIX_QUERY);
            ncaRulesYDao.deleteBefore(modificationTime);
            settingsService.update(NCA_RULES_LAST_IMPORT, modificationTime.toString());
            return true;
        });
        return new Result(TaskResult.SUCCESS);
    }

    @Getter
    public static class TaskState implements PeriodicTaskState {

    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_DOMAINS_WITH_NCA_CERTIFICATE;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }
}
