package ru.yandex.webmaster3.worker.nca;

import java.io.IOException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Range;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;


/**
 * @author kravchenko99
 * @date 3/15/22
 */


@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class SendNcaCertificateMessagesTask extends PeriodicTask<SendNcaCertificateMessagesTask.TaskState> {

    @Value("${webmaster3.worker.nca.antispam.domains_list_with_tls}")
    private final YtPath ncaDomains;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.arnold.export.archive.verifiedHosts.latest.link.path}")
//    @Value("${external.yt.service.arnold}://home/webmaster/prod/export/archive/webmaster-verified-hosts-latest")
    private final YtPath verifiedWmcHosts;
    @Value("${external.yt.service.arnold.root.default}/tmp/nca_certificate/new_wmc_domains_with_nca_certificate")
    private final YtPath newWmcDomainsWithNcaCertificate;
    @Value("${external.yt.service.arnold.root.default}/tmp/nca_certificate/old_wmc_domains_with_nca_certificate")
    private final YtPath oldWmcDomainsWithNcaCertificate;
    @Value("${external.yt.service.arnold.root.default}/tmp/nca_certificate/diff_wmc_domains_with_nca_certificate")
    private final YtPath diffWmcDomainsWithNcaCertificate;

    String yqlQuery = """
            use ${CLUSTER};
            PRAGMA yt.InferSchema = '1';

            $old_hosts = select domain from ${OLD_TEMP_TABLE} ;

            insert into ${NEW_TEMP_TABLE} with truncate
            select distinct String::ReplaceAll(mask, "*.", "") as domain
            from ${NYC_FILTER};
            COMMIT;

            insert into ${DIFF_TEMP_TABLE} with truncate
            select domain from ${NEW_TEMP_TABLE} where domain not in $old_hosts
            """;

    private final YqlService yqlService;
    private final YtService ytService;
    private final SettingsService settingsService;
    private final WMCEventsService wmcEventsService;

    @Override
    public Result run(UUID runId) throws Exception {
        CommonDataState lastImport =
                settingsService.getSettingUncached(CommonDataType.NCA_CERTIFICATE_LAST_IMPORT);
        DateTime modificationTime =
                ytService.inTransaction(ncaDomains).query(cypressService -> {
                    YtNode node = cypressService.getNode(ncaDomains);
                    if (node == null) {
                        return null;
                    }
                    return new DateTime(node.getNodeMeta()
                            .get("modification_time")
                            .asText());
                });

        if (modificationTime == null) {
            log.error("table with nyc domains certificate doesnt exists path: {}", ncaDomains);
            return new Result(TaskResult.FAIL);
        }
        if (lastImport != null && new DateTime(lastImport.getValue()).equals(modificationTime)) {
            return new Result(TaskResult.SUCCESS);
        }

        StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                "WMC_HOSTS", verifiedWmcHosts.toYqlPath(),
                "NYC_FILTER", ncaDomains.toYqlPath(),
                "NEW_TEMP_TABLE", newWmcDomainsWithNcaCertificate.toYqlPath(),
                "DIFF_TEMP_TABLE", diffWmcDomainsWithNcaCertificate.toYqlPath(),
                "OLD_TEMP_TABLE", oldWmcDomainsWithNcaCertificate.toYqlPath(),
                "CLUSTER", ncaDomains.getCluster()
        ));

        yqlService.execute(substitutor.replace(yqlQuery));

        Set<WebmasterHostId> hosts = new HashSet<>();

        ytService.inTransaction(diffWmcDomainsWithNcaCertificate).execute(cypressService -> {
            cypressService.move(newWmcDomainsWithNcaCertificate, oldWmcDomainsWithNcaCertificate,
                    false, true);

            var tableReadDriver = YtTableReadDriver.createYSONDriver(Row.class);
            var tableReader = new AsyncTableReader<>(cypressService, diffWmcDomainsWithNcaCertificate, Range.all(),
                    tableReadDriver)
                    .withRetry(5);

            try (var it = tableReader.read()) {
                while (it.hasNext()) {
                    Row next = it.next();
                    var hostId = IdUtils.urlToHostId("https://" + next.domain);
                    hosts.add(hostId);
                }
            } catch (IOException e) {
                throw new YtException("Unable to read table: " + diffWmcDomainsWithNcaCertificate, e);
            }


            return true;
        });
        // сделанно для того, чтобы не перепослать дважды одно сообщение
        hosts.forEach(hostId -> {
            wmcEventsService.addEvent(WMCEvent.create(new RetranslateToUsersEvent<>(UserHostMessageEvent.create(
                    hostId,
                    new MessageContent.NcaCertificate(hostId),
                    NotificationType.NCA_CERTIFICATE,
                    true)
            )));
        });

        settingsService.update(CommonDataType.NCA_CERTIFICATE_LAST_IMPORT, modificationTime.toString());

        return new Result(TaskResult.SUCCESS);
    }


    private static class Row {
        @JsonProperty("domain")
        String domain;
    }

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> DOMAIN = SCHEMA.addColumn("domain", YtColumn.Type.STRING);
    }

    @Getter
    public static class TaskState implements PeriodicTaskState {

    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.SEND_NCA_CERTIFICATE_NOTIFICATION;
    }

    @Override
    public TaskSchedule getSchedule() {
        //0 */5 * * * *
        return TaskSchedule.never();
    }
}
