package ru.yandex.webmaster3.worker.news;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.news.TrustedMassMediaYDao;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * ishalaru
 * 17.11.2020
 **/
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ExportTrustedMassMediaPeriodicTask extends PeriodicTask<ExportTrustedMassMediaPeriodicTask.State> {
    private static final int BATCH_SIZE = 5000;
    @Value("${external.yt.service.arnold.root.default}/export/news/trusted_mass_media")
    private YtPath path;
    private final YtService ytService;
    private final TrustedMassMediaYDao trustedMassMediaYDao;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new State());

        YtTableData tableData =
                ytService.prepareTableData("export-mass-media", tableWriter -> {
                            final List<Pair<WebmasterHostId, Set<Long>>> list = new ArrayList<>();
                            trustedMassMediaYDao.forEach(e -> {
                                list.add(e);
                                if (list.size() >= BATCH_SIZE) {
                                    writeBatch(list, tableWriter);
                                    list.clear();
                                }

                            });
                            if (list.size() > 0) {
                                writeBatch(list, tableWriter);
                                list.clear();
                            }
                        }

                );
        ytService.inTransaction(path).execute(cService -> {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
            if (cService.exists(path)) {
                cService.remove(path);
            }

            cService.create(path, YtNode.NodeType.TABLE, true, attributes);
            cService.writeTable(path, tableData);

            return true;
        });
        return Result.SUCCESS;
    }

    private void writeBatch(List<Pair<WebmasterHostId, Set<Long>>> batch, TableWriter tableWriter) {
        for (Pair<WebmasterHostId, Set<Long>> rowRecord : batch) {
            F.hostId.set(tableWriter, IdUtils.hostIdToUrl(rowRecord.getLeft()));
            F.trustedMassMediaList.set(tableWriter, rowRecord.getRight());
            tableWriter.rowEnd();
        }
    }


    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> hostId = SCHEMA.addColumn("host_id", YtColumn.Type.STRING);
        YtColumn<Object> trustedMassMediaList = SCHEMA.addColumn("trusted_mass_media_list", YtColumn.Type.ANY);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_APPROVED_MASS_MEDIA;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }

    public class State implements PeriodicTaskState {
    }
}
