package ru.yandex.webmaster3.worker.notifications;

import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.Data;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.notifications.NotificationProgress;
import ru.yandex.webmaster3.storage.notifications.NotificationRecListId;
import ru.yandex.webmaster3.storage.notifications.SearchBaseNotificationInfo;
import ru.yandex.webmaster3.storage.notifications.dao.NotificationProgressCypressDao;
import ru.yandex.webmaster3.storage.notifications.dao.SearchBaseNotificationListCHDao;
import ru.yandex.webmaster3.storage.notifications.dao.SearchBaseNotificationsYDao;
import ru.yandex.webmaster3.storage.searchquery.importing.dao.YtClickhouseDataLoadYDao;
import ru.yandex.webmaster3.storage.searchurl.offline.SearchBaseImportTablesService;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportInfo;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportTaskType;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
@Component("prepareSearchBaseNotificationListTask")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class PrepareSearchBaseNotificationListTask extends PeriodicTask<PrepareSearchBaseNotificationListTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(PrepareSearchBaseNotificationListTask.class);

    private static final Duration NOTIFICATION_TTL = Duration.standardDays(14);

    private final SearchBaseNotificationListCHDao searchBaseNotificationListCHDao;
    private final SearchBaseNotificationsYDao searchBaseNotificationsYDao;
    private final SearchBaseReceiversListService searchBaseReceiversListService;
    @Qualifier("searchBaseImportTablesService")
    private final SearchBaseImportTablesService searchBaseImportTablesService;
    @Qualifier("searchBaseNotificationProgressCypressDao")
    private final NotificationProgressCypressDao notificationProgressCypressDao;
    private final YtClickhouseDataLoadYDao ytClickhouseDataLoadYDao;

    @Override
    public Result run(UUID runId) throws Exception {
        for (SearchBaseNotificationInfo notificationInfo : searchBaseNotificationsYDao.listNotifications()) {
            UUID recListId = notificationInfo.getRecListId();
            if (recListId == null) {
                recListId = UUIDs.timeBased();
                log.info("Building rec list {} for notification {}", recListId, notificationInfo.getNotificationId());
                var pair = searchBaseImportTablesService.getLatestSwitcherInfo(SearchBaseImportTaskType.SEARCH_URL_EMAIL_SAMPLES);
                Instant latestSamplesSwitch = pair.getRight().getBaseSwitchDate();
                SearchBaseImportInfo baseImportInfo = pair.getLeft();

                log.info("Latest base switched at " + latestSamplesSwitch);
                if (!latestSamplesSwitch.equals(notificationInfo.getNotificationDate().toInstant())) {
                    log.error("Ignoring non-actual search base notification {}, its date is {}",
                            notificationInfo.getNotificationId(), notificationInfo.getNotificationDate());
                    continue;
                }

                // Убедимся что структуры тоже доехали
                Instant samplesBaseDate = baseImportInfo.getSearchBaseDate();
                Instant importedBaseDate = new Instant(Long.parseLong(ytClickhouseDataLoadYDao.load(YtClickhouseDataLoadType.LAST_SITE_STRUCTURE).getData().split("_")[1]));
                if (samplesBaseDate.isAfter(importedBaseDate)) {
                    log.error("Structures for {} not imported yet", baseImportInfo);
                    continue;
                }

                NotificationRecListId notificationRecListId = new NotificationRecListId(notificationInfo.getNotificationId(), recListId);
                searchBaseReceiversListService.sendSearchBaseUpdateMessage(
                        notificationRecListId,
                        baseImportInfo,
                        latestSamplesSwitch.toDateTime(),
                        notificationInfo.isSendBaseUpdate()
                );

                notificationInfo = notificationInfo.withRecListId(recListId);
                searchBaseNotificationsYDao.updateNotification(notificationInfo, NOTIFICATION_TTL);
                log.info("Done building rec list {} for notification {}", recListId, notificationInfo.getNotificationId());
                setState(new TaskState(notificationRecListId, baseImportInfo, latestSamplesSwitch.toDateTime()));
            }

            if (!notificationInfo.isSending()) {
                UUID notificationId = notificationInfo.getNotificationId();
                Pair<NotificationProgress, Integer> progressAndVersion = notificationProgressCypressDao.getRecordIfExists(notificationId);
                if (progressAndVersion == null) {
                    int targetsInList = searchBaseNotificationListCHDao.countUniqueTargets(new NotificationRecListId(notificationId, recListId));
                    int chunkSize = Math.min(targetsInList, 1000);
                    notificationProgressCypressDao.createRecord(
                            notificationId,
                            NotificationProgress.createInitial(chunkSize, targetsInList, 0)
                    );
                }
                searchBaseNotificationsYDao.updateNotification(notificationInfo.withSending(true), NOTIFICATION_TTL);
            }
        }
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.PREPARE_SEARCH_BASE_NOTIFICATION_LIST;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 1/2 * * * *");
    }

    @Data
    public static class TaskState implements PeriodicTaskState {
        private final NotificationRecListId recListId;
        private final SearchBaseImportInfo baseImportInfo;
        private final DateTime baseSwitchDate;
    }
}
