package ru.yandex.webmaster3.worker.notifications;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableLong;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.UserWithLogin;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.core.util.concurrent.graph.BlockingBatchConsumer;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphExecution;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphExecutionBuilder;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphOutQueue;
import ru.yandex.webmaster3.storage.clickhouse.replication.MdbClickhouseReplicationManager;
import ru.yandex.webmaster3.storage.iks.IksService;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.NotificationRecListId;
import ru.yandex.webmaster3.storage.notifications.dao.PreparedGlobalMessageInfo;
import ru.yandex.webmaster3.storage.notifications.dao.SearchBaseNotificationListCHDao;
import ru.yandex.webmaster3.storage.searchurl.history.dao.LastSiteStructureCHDao;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlStat;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportInfo;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.SearchUrlEmailSamplesCHDao;
import ru.yandex.webmaster3.storage.searchurl.samples.data.RawSearchUrlEventSample;
import ru.yandex.webmaster3.storage.spam.FastSpamHostFilter;
import ru.yandex.webmaster3.storage.spam.SpamHostsYDao;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationChannelsYDao;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationEmailYDao;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationHostSettingsYDao;
import ru.yandex.webmaster3.storage.user.dao.VerifiedHostsYDao;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationSetting;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.service.UserPersonalInfoService;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;

/**
 * @author avhaliullin
 */
@Service("searchBaseReceiversListService")
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SearchBaseReceiversListService {
    private static final Logger log = LoggerFactory.getLogger(SearchBaseReceiversListService.class);

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));
    private static final Set<NotificationType> IMPORTANT_URL_NOTIFICATION_TYPES = EnumSet.of(
            NotificationType.URL_SEARCH_LAST_ACCESS_CHANGE,
            NotificationType.URL_SEARCH_STATUS_CHANGE,
            NotificationType.URL_TITLE_CHANGE,
            NotificationType.URL_DESCRIPTION_CHANGE,
            NotificationType.URL_REL_CANONICAL_TARGET_CHANGE
    );
    private static final Set<NotificationType> SEARCH_NOTIFICATION_TYPES = EnumSet.copyOf(IMPORTANT_URL_NOTIFICATION_TYPES);

    static {
        SEARCH_NOTIFICATION_TYPES.add(NotificationType.SEARCH_BASE_UPDATE);
    }

    private static final int MAX_HOSTS_IN_EMAIL = 10;

    private final ClickhouseServer clickhouseServer;
    private final FastSpamHostFilter fastSpamHostFilter;
    private final IksService iksService;
    private final LastSiteStructureCHDao lastSiteStructureCHDao;
    private final MdbClickhouseReplicationManager clickhouseReplicationManager;
    private final SearchBaseNotificationListCHDao searchBaseNotificationListCHDao;
    private final SearchUrlEmailSamplesCHDao searchUrlEmailSamplesCHDao;
    private final SpamHostsYDao spamHostsYDao;
    private final UserNotificationChannelsYDao userNotificationChannelsYDao;
    private final UserNotificationEmailYDao userNotificationEmailYDao;
    private final UserNotificationHostSettingsYDao userNotificationHostSettingsYDao;
    private final UserPersonalInfoService userPersonalInfoService;
    private final VerifiedHostsYDao verifiedHostsYDao;

    void sendSearchBaseUpdateMessage(NotificationRecListId recListId, SearchBaseImportInfo samplesImportInfo,
                                     DateTime baseSwitchDate, boolean sendBaseUpdate) {
        log.info("recListId: {}, base switch date: {}, import info: {}", recListId, baseSwitchDate, samplesImportInfo);

        ClickhouseHost initialHost = clickhouseServer.pickRandomAliveHost();
        searchBaseNotificationListCHDao.createTable(recListId, initialHost);
        log.info("Created CH table: {}", SearchBaseNotificationListCHDao.getTableName(recListId));

        // (user,hosts) --> channels-resolver -----------------------> personals-resolver -----------------\
        //                                   \--> emails-resolver -/                     \--> blackbox -/  |
        //                                                                                                 |
        // writer <-- user2hosts-multiplexor <-------------------------------------------------------------/
        //         \--------- base-host-resolver <---------------------------/ /
        //          \-------- base-host-resolver_email------------------------/

        GraphExecutionBuilder builder = executionBuilder(recListId);
        GraphExecutionBuilder.Queue<PreparedGlobalMessageInfo> writer = builder
                .process(() -> searchBaseWriter(recListId, initialHost))
                .name("writer")
                .batchLimit(10000)
                .forceFullBatch()
                .getInput();

        // Добавляет инфу, которую будем показывать в email нотификациях
        GraphExecutionBuilder.Queue<SearchBaseInfoForEmailDistrib> searchBaseHostEmailInfoResolver = builder
                .process(writer, writerQ -> hostInfoResolverForEmail(writerQ, samplesImportInfo, baseSwitchDate))
                .name("base-host-email-resolver")
                .batchLimit(128)
                .concurrency(8)
                .getInput();

        // Добавляет инфу, которую будем показывать в cервисных нотификациях
        GraphExecutionBuilder.Queue<SearchBaseInfoForServiceDistrib> searchBaseHostInfoResolver = builder
                .process(writer, writerQ -> hostInfoResolver(writerQ, samplesImportInfo, baseSwitchDate))
                .name("base-host-resolver")
                .batchLimit(128)
                .concurrency(8)
                .getInput();


        // Для хостов пользователя, о которых будем нотифицировать по email, оставляет top N хостов по ИКСу
        GraphExecutionBuilder.Queue<SearchBaseInfoForEmailDistrib> searchBaseEmailTopHostsSelector = builder
                .process(searchBaseHostEmailInfoResolver, this::topHostsSelectorForEmail)
                .name("search-base-email-top-selector")
                .batchLimit(1000)
                .concurrency(4)
                .getInput();

        GraphExecutionBuilder.Queue<SearchBaseInfoForServiceDistrib> searchBaseSpamFilter = builder
                .process(searchBaseHostInfoResolver, out -> spamHostsFilter(out, info -> info.hostId))
                .name("search-base-spam-filter")
                .batchLimit(1000)
                .concurrency(4)
                .getInput();

        // Разделяет нотификации для SEARCH_BASE_UPDATE на два потока: в сервисе и по email
        GraphExecutionBuilder.Queue<UserHostsPersonalsInfo> hostInfoMultiplexor = builder
                .process(searchBaseSpamFilter, searchBaseEmailTopHostsSelector, (base, baseToEmail) -> searchBaseUser2HostsMultiplexor(base, baseToEmail, sendBaseUpdate))
                .name("hosts-multiplex")
                .batchLimit(10000)
                .concurrency(4)
                .getInput();

        // Ходит в ЧЯ за персональными данными пользователя
        GraphExecutionBuilder.Queue<UserHostsChannelsInfo> bbResolver = builder
                .process(hostInfoMultiplexor, this::searchBaseBlackboxResolver)
                .name("blackbox")
                .batchLimit(100)
                .getInput();

        // Достает из нашего кеша персональные данные пользователя.
        // Если их там нет - добавляет поход в ЧЯ.
        GraphExecutionBuilder.Queue<UserHostsChannelsInfo> personalsResolver = builder
                .process(hostInfoMultiplexor, bbResolver, this::searchBasePersonalsResolver)
                .name("personals")
                .concurrency(4)
                .batchLimit(1000)
                .getInput();

        // Добавляет email пользователя в UserHostsChannelsInfo
        GraphExecutionBuilder.Queue<UserHostsChannelsInfo> emailsResolver = builder
                .process(personalsResolver, this::searchBaseEmailsResolver)
                .name("emails")
                .concurrency(4)
                .batchLimit(200)
                .getInput();

        // Определяет в какие каналы в итоге будем слать нотификации исходя из настроек пользователя.
        // Если будем слать в том числе и по почте, отправляет дальше на обработку в emailsResolver,
        // иначе - в personalsResolver.
        GraphExecutionBuilder.Queue<UserHostsInfo> channelsResolver = builder
                .process(personalsResolver, emailsResolver, this::resolveChannels)
                .name("channels")
                .concurrency(4)
                .batchLimit(200)
                .getInput();

        GraphExecution<UserHostsInfo> graph = builder.build(channelsResolver);

        graph.start();
        try {
            MutableLong curUser = new MutableLong(-1L);
            List<WebmasterHostId> hosts = new ArrayList<>();
            verifiedHostsYDao.forEach(uh -> {
                try {
                    if (!curUser.getValue().equals(uh.getLeft())) {
                        if (!hosts.isEmpty()) {
                            graph.put(new UserHostsInfo(curUser.getValue(), new ArrayList<>(hosts)));
                            hosts.clear();
                        }
                        curUser.setValue(uh.getLeft());
                    }
                    hosts.add(uh.getRight().getWebmasterHostId());
                } catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            });
            if (!hosts.isEmpty()) {
                graph.put(new UserHostsInfo(curUser.getValue(), new ArrayList<>(hosts)));
            }

            graph.doneWritingAndAwaitTermination();

            var command = clickhouseReplicationManager.nonShardedReplication(
                    searchBaseNotificationListCHDao.getDbName(),
                    SearchBaseNotificationListCHDao.getTableName(recListId),
                    searchBaseNotificationListCHDao.buildCreateTableQuery(recListId)
            );
            clickhouseReplicationManager.replicateSynchronously(command);
        } catch (InterruptedException | ExecutionException e) {
            throw new WebmasterException("Execution failed",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null), e);
        } finally {
            graph.terminateAbruptly();
        }
    }

    private BlockingBatchConsumer<PreparedGlobalMessageInfo> searchBaseWriter(NotificationRecListId listId, ClickhouseHost initialHost) {
        return batch -> RetryUtils.execute(RETRY_POLICY, () -> {
            searchBaseNotificationListCHDao.addRecords(listId, batch, initialHost);
        });
    }

    private <T> BlockingBatchConsumer<T> spamHostsFilter(GraphOutQueue<T> out, Function<T, WebmasterHostId> extractHostId) {
        return (List<T> batch) -> {
            List<T> preFiltered = batch.stream().filter(info -> !fastSpamHostFilter.checkHost(extractHostId.apply(info))).collect(Collectors.toList());
            if (preFiltered.isEmpty()) {
                return;
            }
            Set<WebmasterHostId> spamHosts = RetryUtils.query(RETRY_POLICY,
                    () -> spamHostsYDao.getSpamHosts(preFiltered.stream()
                            .map(extractHostId)
                            .collect(Collectors.toSet()))
            );
            int passed = 0;
            for (T item : preFiltered) {
                if (!spamHosts.contains(extractHostId.apply(item))) {
                    out.put(item);
                    passed++;
                }
            }
            // log.info("Filtered out {} spam hosts of {}", (batch.size() - passed), batch.size());
        };
    }

    private void createSearchBaseUpdateNewMessageContent(WebmasterHostId hostId,
                                                         Map<WebmasterHostId, List<RawSearchUrlEventSample>> host2Samples,
                                                         Map<WebmasterHostId, SearchUrlStat> host2Stats,
                                                         DateTime baseSwitchDate,
                                                         Consumer<MessageContent.SearchBaseUpdateNew> consumer) {
        SearchUrlStat stat = host2Stats.get(hostId);
        if (stat == null) {
            log.info("createSearchBaseUpdateNewMessageContent: no stats for {}, skipping", hostId);
            return;
        }

        String hostUrl = IdUtils.hostIdToReadableUrl(hostId);
        List<RawSearchUrlEventSample> samples = host2Samples.getOrDefault(hostId, Collections.emptyList());
        Set<String> newPages = new HashSet<>();
        Set<String> gonePages = new HashSet<>();
        for (RawSearchUrlEventSample sample : samples) {
            switch (sample.getEventType()) {
                case GONE:
                    gonePages.add(hostUrl +
                            URLEncodeUtil.prettifyUrl(sample.getUrl()));
                    break;
                case NEW:
                    newPages.add(hostUrl +
                            URLEncodeUtil.prettifyUrl(sample.getUrl()));
                    break;
                default:
                    throw new RuntimeException("Unknown sample kind " + sample.getEventType());
            }
        }
        List<String> newPagesList = new ArrayList<>(newPages);
        Collections.sort(newPagesList);
        List<String> gonePagesList = new ArrayList<>(gonePages);
        Collections.sort(gonePagesList);
        MessageContent.SearchBaseUpdateNew messageContent = new MessageContent.SearchBaseUpdateNew(
                baseSwitchDate,
                hostId,
                stat.getTotalPages(),
                stat.getNewPages(),
                stat.getGonePages(),
                newPagesList,
                gonePagesList);

        consumer.accept(messageContent);
    }

    /**
     * Добавляет инфу, которую будем показывать в нотификациях
     */
    private BlockingBatchConsumer<SearchBaseInfoForEmailDistrib> hostInfoResolverForEmail(
            GraphOutQueue<PreparedGlobalMessageInfo> out, SearchBaseImportInfo samplesImportInfo, DateTime baseSwitchDate) {
        return batch -> {
            Collection<WebmasterHostId> hostIds = batch.stream().flatMap(hi -> hi.hostIds.stream()).collect(Collectors.toSet());
            Map<WebmasterHostId, SearchUrlStat> host2Stats = RetryUtils.query(RETRY_POLICY,
                    () -> lastSiteStructureCHDao.getSearchUrlStats(hostIds, samplesImportInfo.getSearchBaseDate())
            );

            Map<WebmasterHostId, List<RawSearchUrlEventSample>> host2Samples = RetryUtils.query(RETRY_POLICY,
                    () -> searchUrlEmailSamplesCHDao.getSamples(samplesImportInfo, hostIds)
            );

            for (SearchBaseInfoForEmailDistrib hostInfo : batch) {
                List<MessageContent.SearchBaseUpdateNew> searchBaseUpdates = new ArrayList<>();
                for (WebmasterHostId hostId : hostInfo.hostIds) {
                    createSearchBaseUpdateNewMessageContent(hostId, host2Samples, host2Stats, baseSwitchDate, searchBaseUpdates::add);
                }

                if (!searchBaseUpdates.isEmpty()) {
                    out.put(new PreparedGlobalMessageInfo(
                            hostInfo.userId,
                            hostInfo.email,
                            "",
                            new MessageContent.SearchBaseUpdateNewAllUserHosts(baseSwitchDate, searchBaseUpdates),
                            Collections.singleton(NotificationChannel.EMAIL),
                            hostInfo.personalInfo.getLanguage(),
                            hostInfo.personalInfo.getLogin(),
                            hostInfo.personalInfo.getFio()
                    ));
                }
            }
        };
    }

    private BlockingBatchConsumer<SearchBaseInfoForServiceDistrib> hostInfoResolver(
            GraphOutQueue<PreparedGlobalMessageInfo> out, SearchBaseImportInfo samplesImportInfo, DateTime baseSwitchDate) {
        return batch -> {
            Collection<WebmasterHostId> hostIds = batch.stream().map(hi -> hi.hostId).collect(Collectors.toSet());
            Map<WebmasterHostId, SearchUrlStat> host2Stats = RetryUtils.query(RETRY_POLICY,
                    () -> lastSiteStructureCHDao.getSearchUrlStats(hostIds, samplesImportInfo.getSearchBaseDate())
            );

            Map<WebmasterHostId, List<RawSearchUrlEventSample>> host2Samples = RetryUtils.query(RETRY_POLICY,
                    () -> searchUrlEmailSamplesCHDao.getSamples(samplesImportInfo, hostIds)
            );

            for (SearchBaseInfoForServiceDistrib hostInfo : batch) {
                List<MessageContent.SearchBaseUpdateNew> searchBaseUpdates = new ArrayList<>();
                createSearchBaseUpdateNewMessageContent(hostInfo.hostId, host2Samples, host2Stats, baseSwitchDate, searchBaseUpdates::add);
                if (!searchBaseUpdates.isEmpty()) {
                    out.put(new PreparedGlobalMessageInfo(
                            hostInfo.userId,
                            "",
                            "",
                            searchBaseUpdates.get(0),
                            Collections.singleton(NotificationChannel.SERVICE),
                            hostInfo.personalInfo.getLanguage(),
                            hostInfo.personalInfo.getLogin(),
                            hostInfo.personalInfo.getFio()
                    ));
                }
            }
        };
    }

    /**
     * Для хостов пользователя, о которых будем нотифицировать по email, оставляет top N хостов по ИКСу
     */
    private BlockingBatchConsumer<SearchBaseInfoForEmailDistrib> topHostsSelectorForEmail(GraphOutQueue<SearchBaseInfoForEmailDistrib> out) {
        return batch -> {
            for (SearchBaseInfoForEmailDistrib info : batch) {
                long userId = info.userId;

                List<WebmasterHostId> allHosts = info.hostIds;
                Map<WebmasterHostId, Integer> iksValues = iksService.getIksValues(info.hostIds);
                allHosts.sort(Comparator.comparingInt(iksValues::get));
                Collections.reverse(allHosts);
                if (!allHosts.isEmpty()) {
                    List<WebmasterHostId> topHosts = new ArrayList<>(allHosts.subList(0, Math.min(MAX_HOSTS_IN_EMAIL, allHosts.size())));

                    out.put(new SearchBaseInfoForEmailDistrib(info.userId, topHosts, info.personalInfo, info.email));
                }
            }
        };
    }

    /**
     * Разделяет нотификации для SEARCH_BASE_UPDATE на два потока: в сервисе и по email
     */
    private BlockingBatchConsumer<UserHostsPersonalsInfo> searchBaseUser2HostsMultiplexor
            (GraphOutQueue<SearchBaseInfoForServiceDistrib> baseUpdateServiceQ, GraphOutQueue<SearchBaseInfoForEmailDistrib> baseUpdateEmailQ, boolean sendBaseUpdate) {
        return batch -> {
            for (UserHostsPersonalsInfo personalInfo : batch) {
                List<WebmasterHostId> hosts = new ArrayList<>();
                long userId = personalInfo.userPersonalInfo.getUserId();
                // для всех хостов, по которым будем слать нотификации
                for (Map.Entry<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> hostEntry : personalInfo.channelsInfo.host2Settings.entrySet()) {
                    WebmasterHostId hostId = hostEntry.getKey();

                    // каналы по которым будем слать нотификации SEARCH_BASE_UPDATE
                    Map<NotificationType, Set<NotificationChannel>> type2Channels = hostEntry.getValue();
                    Set<NotificationChannel> channels = type2Channels.getOrDefault(NotificationType.SEARCH_BASE_UPDATE, Collections.emptySet());
                    // для оповещений в сервисе положим в одну очередь
                    if (sendBaseUpdate && !channels.isEmpty() && channels.contains(NotificationChannel.SERVICE)) {
                        baseUpdateServiceQ.put(new SearchBaseInfoForServiceDistrib(userId, hostId, personalInfo.userPersonalInfo));
                    }

                    //  а для оповещений по email - соберем все хосты вместе
                    if (sendBaseUpdate && !channels.isEmpty() && channels.contains(NotificationChannel.EMAIL)) {
                        hosts.add(hostId);
                    }
                }

                if (!hosts.isEmpty()) {
                    baseUpdateEmailQ.put(new SearchBaseInfoForEmailDistrib(userId, hosts, personalInfo.userPersonalInfo, personalInfo.channelsInfo.email));
                }
            }
        };
    }

    /**
     * Ходит в ЧЯ за персональными данными пользователя
     */
    private BlockingBatchConsumer<UserHostsChannelsInfo> searchBaseBlackboxResolver(GraphOutQueue<UserHostsPersonalsInfo> out) {
        return buffer -> {
            List<Long> unknownUsers = new ArrayList<>();
            List<UserWithLogin> unknownPersonals = new ArrayList<>();
            for (UserHostsChannelsInfo info : buffer) {
                long userId = info.userId;
                if (info.login == null) {
                    unknownUsers.add(userId);
                } else {
                    unknownPersonals.add(new UserWithLogin(userId, info.login));
                }
            }
            Map<Long, UserPersonalInfo> personals = RetryUtils.query(RETRY_POLICY,
                    () -> userPersonalInfoService.resolveCacheMisses(unknownUsers, unknownPersonals)
            );

            for (UserHostsChannelsInfo rawInfo : buffer) {
                long userId = rawInfo.userId;
                UserPersonalInfo personalInfo = personals.get(userId);
                if (personalInfo != null) {
                    out.put(new UserHostsPersonalsInfo(rawInfo, personalInfo));
                }
            }
        };

    }

    /**
     * Достает из нашего кеша в Кассандре персональные данные пользователя.
     */
    private BlockingBatchConsumer<UserHostsChannelsInfo> searchBasePersonalsResolver(GraphOutQueue<UserHostsPersonalsInfo> out, GraphOutQueue<UserHostsChannelsInfo> blackboxQ) {
        return batch -> {
            UserPersonalInfoService.CachedResponse personals = RetryUtils.query(RETRY_POLICY,
                    () -> userPersonalInfoService.getCachedUsersPersonalInfos(batch.stream().map(info -> info.userId).collect(Collectors.toSet()))
            );
            Map<Long, UserWithLogin> unknownPersonals = personals.getUnknownPersonals().stream().collect(Collectors.toMap(UserWithLogin::getUserId, Function.identity()));
            for (UserHostsChannelsInfo rawInfo : batch) {
                long userId = rawInfo.userId;
                UserPersonalInfo personalInfo = personals.getResult().get(userId);
                if (personalInfo == null) {
                    if (personals.getUnknownUsers().contains(userId)) {
                        blackboxQ.put(rawInfo);
                    } else {
                        UserWithLogin userWithLogin = unknownPersonals.get(userId);
                        if (userWithLogin == null) {
                            log.info("searchBasePersonalsResolver: Should not happen: {}", userId);
                            break; // Вроде такого не должно быть
                        }
                        blackboxQ.put(new UserHostsChannelsInfo(userId, rawInfo.host2Settings, rawInfo.email, userWithLogin.getLogin()));
                    }
                } else {
                    out.put(new UserHostsPersonalsInfo(rawInfo, personalInfo));
                }
            }
        };
    }

    /**
     * Добавляет в UserHostsChannelsInfo email пользователя на который нужно слать нотификации.
     * Если email нет, выкидывает из полученных в resolveChannels  каналов пользователя все каналы с email.
     */
    private BlockingBatchConsumer<UserHostsChannelsInfo> searchBaseEmailsResolver(GraphOutQueue<UserHostsChannelsInfo> out) {
        return batch -> {
            Map<Long, String> emails = RetryUtils.query(RETRY_POLICY,
                    () -> userNotificationEmailYDao.getUserEmails(batch.stream().map(info -> info.userId).collect(Collectors.toSet()))
            );
            for (UserHostsChannelsInfo rawInfo : batch) {
                long userId = rawInfo.userId;
                String email = emails.get(userId);
                email = StringUtils.isEmpty(email) ? null : email;

                if (email == null) {
                    // У нас нет email пользователя, нужно выкинуть полученных в resolveChannels каналов все email каналы
                    Map<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> newChannels = new HashMap<>();
                    for (Map.Entry<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> hostEntry : rawInfo.host2Settings.entrySet()) {
                        for (Map.Entry<NotificationType, Set<NotificationChannel>> typeEntry : hostEntry.getValue().entrySet()) {
                            for (NotificationChannel notificationChannel : typeEntry.getValue()) {
                                if (notificationChannel != NotificationChannel.EMAIL) {
                                    newChannels.computeIfAbsent(hostEntry.getKey(), ign -> new EnumMap<>(NotificationType.class))
                                            .computeIfAbsent(typeEntry.getKey(), ign -> EnumSet.noneOf(NotificationChannel.class))
                                            .add(notificationChannel);
                                }
                            }
                        }
                    }

                    if (!newChannels.isEmpty()) {
                        out.put(new UserHostsChannelsInfo(userId, newChannels, null, null));
                    }
                } else {
                    out.put(new UserHostsChannelsInfo(userId, rawInfo.host2Settings, email, null));
                }
            }
        };
    }

    /**
     * Для пользователей/хостов в батче определяет в какие каналы в итоге будем слать нотификации исходя из настроек пользователя
     */
    private BlockingBatchConsumer<UserHostsInfo> resolveChannels(GraphOutQueue<UserHostsChannelsInfo> out, GraphOutQueue<UserHostsChannelsInfo> clarifyEmailsQ) {
        return batch -> {
            // возьмем всех пользователей в батче
            Set<Long> userIds = batch.stream().map(uh -> uh.userId).collect(Collectors.toSet());

            // вытащим для них настройки нотификаций
            Map<Long, Map<NotificationType, Set<NotificationChannel>>> user2Channels = RetryUtils.query(RETRY_POLICY,
                    () -> userNotificationChannelsYDao.getChannelsInfo(userIds)
            );
            Map<Long, List<HostNotificationSetting>> user2HostNotificationSettings =
                    RetryUtils.query(RETRY_POLICY, () -> userNotificationHostSettingsYDao.listSettingsForUsers(userIds));

            // пройдемся по батчу
            for (UserHostsInfo userHostsInfo : batch) {
                long userId = userHostsInfo.userId;

                // по каким каналам слать нотификации этому пользователю
                Map<NotificationType, Set<NotificationChannel>> type2Channels = user2Channels.get(userId);
                if (type2Channels == null) {
                    type2Channels = Collections.emptyMap();
                }

                // нужно ли нам знать email пользователя для рассылок нотификаций
                boolean needEmail = false;

                // в какие каналы в итоге будем слать нотификации исходя из настроек пользователя
                Map<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> host2Channels = new HashMap<>();

                // для всех нотификаций которые мы рассылаем при обновлении базы
                for (NotificationType notificationType : SEARCH_NOTIFICATION_TYPES) {
                    // по какому каналу слать эту нотификацию данному пользователю
                    Set<NotificationChannel> defaultChannels = type2Channels.getOrDefault(notificationType, notificationType.getDefaultChannels());

                    // похостовые настройки нотификаций для данного пользователя
                    Map<WebmasterHostId, Map<NotificationChannel, HostNotificationMode>> host2Settings =
                            user2HostNotificationSettings.getOrDefault(userId, Collections.emptyList())
                                    .stream()
                                    .filter(s -> s.getType() == notificationType)
                                    .collect(Collectors.groupingBy(
                                            HostNotificationSetting::getHostId,
                                            Collectors.toMap(HostNotificationSetting::getChannel, HostNotificationSetting::getMode)
                                    ));

                    // для всех хостов пользователя
                    for (WebmasterHostId hostId : userHostsInfo.hosts) {

                        // настройки нотификаций для данного хоста
                        Map<NotificationChannel, HostNotificationMode> channel2Mode = host2Settings.getOrDefault(hostId, Collections.emptyMap());
                        for (NotificationChannel channel : NotificationChannel.values()) {
                            HostNotificationMode hostMode = channel2Mode.getOrDefault(channel, HostNotificationMode.DEFAULT);

                            // нужно ли нам слать эту нотификацию для пользователя/хоста или у пользователя все выключено
                            boolean needChannel;
                            switch (hostMode) {
                                case ENABLED:
                                    needChannel = true;
                                    break;
                                case DISABLED:
                                    needChannel = false;
                                    break;
                                case DEFAULT:
                                    needChannel = defaultChannels.contains(channel);
                                    break;
                                default:
                                    throw new RuntimeException("Unknown host notification mode " + hostMode);
                            }

                            // слать нужно
                            if (needChannel) {
                                if (channel == NotificationChannel.EMAIL) {
                                    // в том числе и по почте
                                    needEmail = true;
                                }

                                // добавим канал

                                Map<NotificationType, Set<NotificationChannel>> notificationChannels = host2Channels.computeIfAbsent(hostId, ign -> new HashMap<>());
                                notificationChannels.computeIfAbsent(notificationType, ign -> EnumSet.noneOf(NotificationChannel.class))
                                        .add(channel);
                            }
                        }
                    }
                }

                if (!host2Channels.isEmpty()) {
                    UserHostsChannelsInfo msg = new UserHostsChannelsInfo(userId, host2Channels, null, null);
                    if (needEmail) {
                        clarifyEmailsQ.put(msg);
                    } else {
                        out.put(msg);
                    }
                }
            }
        };
    }

    @RequiredArgsConstructor
    private static class SearchBaseInfoForServiceDistrib {
        private final long userId;
        private final WebmasterHostId hostId;
        private final UserPersonalInfo personalInfo;
    }

    @RequiredArgsConstructor
    private static class SearchBaseInfoForEmailDistrib {
        private final long userId;
        private final List<WebmasterHostId> hostIds;
        private final UserPersonalInfo personalInfo;
        private final String email;
    }

    @RequiredArgsConstructor
    private static class UserHostsPersonalsInfo {
        private final UserHostsChannelsInfo channelsInfo;
        private final UserPersonalInfo userPersonalInfo;
    }

    @RequiredArgsConstructor
    private static class UserHostsChannelsInfo {
        private final long userId;
        private final Map<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> host2Settings;
        private final String email;
        private final String login;
    }

    @RequiredArgsConstructor
    private static class UserHostsInfo {
        private final long userId;
        private final List<WebmasterHostId> hosts;
    }

    private GraphExecutionBuilder executionBuilder(NotificationRecListId recListId) {
        return GraphExecutionBuilder.newBuilder(recListId.getListId().toString().substring(0, 8));
    }
}
