package ru.yandex.webmaster3.worker.notifications;

import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

import lombok.AllArgsConstructor;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.blackbox.service.BlackboxUsersService;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserVerifiesHostEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsObserver;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.notifications.EmailTrackingInfo;
import ru.yandex.webmaster3.storage.notifications.dao.ReceivedServiceSubscriptionNotificationYDao;
import ru.yandex.webmaster3.storage.notifications.service.TrackingEmailSenderService;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationEmailYDao;
import ru.yandex.webmaster3.storage.user.service.UserPersonalInfoService;
import ru.yandex.webmaster3.worker.notifications.auto.NotificationsTemplateUtil;

/**
 * WMC-7004
 *
 * @author akhazhoyan 02/2019
 */
@AllArgsConstructor(onConstructor_ = @Autowired)
public final class UserVerifiesHostEventSendNotificationObserver implements WMCEventsObserver {
    private static final Logger log = LoggerFactory.getLogger(UserVerifiesHostEventSendNotificationObserver.class);
    private static final String MESSAGE_TYPE = "SERVICE_SUBSCRIPTION_NOTIFICATION";

    private final BlackboxUsersService blackboxExternalYandexUsersService;
    private final CommonDataStateYDao commonDataStateYDao;
    private final TrackingEmailSenderService trackingEmailSenderService;
    private final EuEmailService euEmailService;
    private final ReceivedServiceSubscriptionNotificationYDao receivedServiceSubscriptionNotificationYDao;
    private final UserNotificationEmailYDao userNotificationEmailYDao;
    private final UserPersonalInfoService userPersonalInfoService;
    private final UserNotificationSettingsService userNotificationSettingsService;

    private boolean isServiceSubscriptionNotificationOn() {
        return Optional.of(commonDataStateYDao)
                .map(c -> c.getValue(CommonDataType.SERVICE_SUBSCRIPTION_NOTIFICATION_SWITCH))
                .map(s -> s.getValue())
                .map(s -> s.isEmpty())
                .orElse(true);
    }

    @Override
    public boolean observe(WMCEvent event) {
        if (!(event.getContent() instanceof UserVerifiesHostEvent)) {
            log.error("Unexpected event type: {}", event.getContent().getType());
            return false;
        }
        if (!isServiceSubscriptionNotificationOn()) {
            log.info("Service subscription notification is off right now, skipping event {}", event.getId());
            return false;
        }

        UserVerifiesHostEvent content = (UserVerifiesHostEvent) event.getContent();
        long userId = content.getUserId();

        boolean isUserAlreadyProcessed = receivedServiceSubscriptionNotificationYDao.contains(userId);
        if (isUserAlreadyProcessed) {
            log.info("User {} is already subscribed, skipping", userId);
            return false;
        }
        receivedServiceSubscriptionNotificationYDao.insert(userId);

        //С большой вероятностью у нас уже будет email в базе, т.к. пользователь успевает
        //кликнуть на подтверждение email до того, как выполнится этот код
        String address = userNotificationEmailYDao.getEmail(userId);
        if (address == null) {
            address = fetchEmailOrNull(userId);
            if (address == null) {
                log.info("Blackbox does not have an email address for user {}", userId);
                return false;
            }
            userNotificationSettingsService.saveEmail(userId, address);
        } else {
            log.info("User {} already has an email!", userId);
        }

        if (euEmailService.isAddressEuropean(address)) {
            log.info("User {} has a european email address: '{}'", userId, address);
            return false;
        }

        sendNotification(userId, address);
        return true;
    }

    private void sendNotification(long userId, String address) {
        UserPersonalInfo info = userPersonalInfoService.getUserPersonalInfo(userId);
        String login = info.getLogin();
        NotificationsTemplateUtil.EmailContent content =
                NotificationsTemplateUtil.renderServiceSubscriptionNotification(info);
        log.info("Sending service subscription verification email to address {}", login);
        EmailTrackingInfo trackingInfo = new EmailTrackingInfo(userId, MESSAGE_TYPE, DateTime.now());
        trackingEmailSenderService.sendEmail(address, login, content.getSubject(), content.getBody(), trackingInfo);
    }

    private String fetchEmailOrNull(long userId) {
        String defaultEmail = blackboxExternalYandexUsersService.getDefaultEmail(userId);
        if (defaultEmail != null) {
            return defaultEmail;
        }

        List<String> emails = blackboxExternalYandexUsersService.getValidatedEmails(userId);
        if (!emails.isEmpty()) {
            return emails.stream().filter(IS_YANDEX_RU).findAny().orElse(emails.get(0));
        }
        return null;
    }

    private static final Predicate<String> IS_YANDEX_RU = email -> email.endsWith("@yandex.ru");
}
