package ru.yandex.webmaster3.worker.notifications.auto;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsObserver;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.UserHostNotificationConfiguration;
import ru.yandex.webmaster3.storage.notifications.UserNotificationConfiguration;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.spam.ISpamHostFilter;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;
import ru.yandex.webmaster3.storage.user.notification.HostNotificationMode;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.user.service.UserPersonalInfoService;
import ru.yandex.webmaster3.worker.notifications.EuEmailService;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class AutoNotificationsObserver implements WMCEventsObserver {
    private static final Set<MessageTypeEnum> YT_SUP_NOTIFICATION_TYPES = Set.of(MessageTypeEnum.IKS_UPDATE);

    private final AutoNotificationsSenderService autoNotificationsSenderService;
    private final EuEmailService euEmailService;
    private final ISpamHostFilter fastSpamHostFilter;
    private final UserNotificationSettingsService userNotificationSettingsService;
    private final UserPersonalInfoService userPersonalInfoService;
    private final UserHostsService userHostsService;
    private final ExtendedInfoService extendedInfoService;
    private NotificationChannel sendByChannel;

    @Override
    public boolean observe(WMCEvent event) {
        WMCEventContent content = event.getContent();
        if (!(content instanceof UserHostMessageEvent)) {
            return false;
        }
        UserHostMessageEvent userHostMessageEvent = (UserHostMessageEvent) content;
        //Т.к. в теукщи момент обновление X просиходит в короткий промежуток времени массово,
        // то PUSH рассылать нельзя, надо делать через YT, а тут мы их динамим
        //TODO: По хорошему надо сделать это отключаемым.
        if (sendByChannel == NotificationChannel.SUP && YT_SUP_NOTIFICATION_TYPES.contains(userHostMessageEvent.getMessageContent().getType())) {
            return false;
        }
        WebmasterHostId hostId = userHostMessageEvent.getHostId();
        NotificationType notificationType = userHostMessageEvent.getNotificationType();
        List<WebmasterHostId> hostIds = userHostMessageEvent.getHostIds();
        log.info("AutoNotificationsObserver. Type={} HostIds={}", notificationType, hostIds);

        if (hostId != null && fastSpamHostFilter.checkHost(hostId)) {
            log.info("Ignoring spam host {}", hostId);
            return false;
        }
        String email = null;
        boolean sendNotification = false;

        Long userId = userHostMessageEvent.getUserId();
        if (UserNotificationSettingsService.PER_HOST_NOTIFICATION_TYPES.contains(notificationType)) {
            if (hostId == null) {
                log.error("HostId is null");
                return false;
            }
            UserHostNotificationConfiguration userNotificationsSettings =
                    userNotificationSettingsService.getUserNotificationsSettings(
                            new WebmasterUser(userId), hostIds);
            if (userNotificationsSettings == null) {
                log.info("No notification settings found for user {} host {}", userId, hostId);
                return false;
            }

            Optional<HostNotificationMode> perHostSetting = Optional.empty();
            if (hostIds.size() > 1) {
                hostIds = userHostsService.filterUserVerifiedHosts(userId, hostIds);
            }
            // идем по всем хостам и ищем включенные уведомления
            for (WebmasterHostId hid : hostIds) {
                perHostSetting = userNotificationsSettings.getPerHostSettingResolved(hid, notificationType, sendByChannel);
                if (perHostSetting.isPresent()) {
                    email = userNotificationsSettings.getEmail();
                    sendNotification = perHostSetting.get() == HostNotificationMode.ENABLED ||
                            isServiceEmailNotification(email, notificationType);
                    if (sendNotification) {
                        hostId = hid;
                        break;
                    }
                }
            }
            if (perHostSetting.isEmpty()) {
                log.info("No notification settings found for user {} host {}", userId, hostId);
                return false;
            }
        } else {
            UserNotificationConfiguration notificationSettings = userNotificationSettingsService
                    .getUserNotificationsSettings(userId, notificationType);

            if (notificationSettings == null) {
                log.info("No notification settings found for user {}", userId);
                return false;
            }
            email = notificationSettings.getEmail();
            sendNotification = notificationSettings.getChannels().contains(sendByChannel) ||
                    isServiceEmailNotification(email, notificationType);
        }

        if (!sendNotification) {
            return false;
        }

        UserPersonalInfo personalInfo = getPersonalInfo(userId);
        if (personalInfo == null) {
            log.warn("Personal info for user {} not found - user may be missing", userId);
            return false;
        }

        if (email == null && sendByChannel == NotificationChannel.EMAIL) {
            log.info("No email found for user {}", userId);
            return false;
        }
        final NotificationInfo notificationInfo = NotificationInfo.builder()
                .id(event.getId())
                .email(email)
                .hostId(hostId)
                .userId(userId)
                .personalInfo(getPersonalInfo(userId))
                .messageContent(userHostMessageEvent.getMessageContent())
                .channel(sendByChannel)
                .critical(userHostMessageEvent.isCritical())
                .extendedInfo(extendedInfoService.convert(userHostMessageEvent.getExperimentInfo(), notificationType.toString()))
                .expType(userHostMessageEvent.getExperimentInfo() == null ? null : userHostMessageEvent.getExperimentInfo().group)
                .build();
        autoNotificationsSenderService.sendMessage(notificationInfo);

        return true;
    }

    private boolean isServiceEmailNotification(String email, NotificationType notificationType) {
        if (sendByChannel == NotificationChannel.SERVICE) {
            return false;
        }
        if (euEmailService.isAddressEuropean(email)) {
            return UserNotificationSettingsService.EUROPEAN_SERVICE_NOTIFICATION.contains(notificationType);
        }
        return UserNotificationSettingsService.SERVICE_NOTIFICATION.contains(notificationType);
    }

    @Override
    public boolean isProcessedLoggingEnabled() {
        return true;
    }

    private UserPersonalInfo getPersonalInfo(long userId) {
        Map<Long, UserPersonalInfo> personalInfos = userPersonalInfoService.getUsersPersonalInfos(
                Collections.singleton(userId)
        );
        return personalInfos.get(userId);
    }

    public void setSendByChannel(NotificationChannel sendByChannel) {
        this.sendByChannel = sendByChannel;
    }

}
