package ru.yandex.webmaster3.worker.notifications.auto;

import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.solomon.metric.SolomonCounter;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricConfiguration;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.sup.SupIntegrationService;
import ru.yandex.webmaster3.storage.abt.AbtService;
import ru.yandex.webmaster3.storage.abt.model.Experiment;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.notifications.EmailTrackingInfo;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.service.TrackingEmailSenderService;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.user.UserInitializationInfo;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.dao.UserMessages3CHDao;
import ru.yandex.webmaster3.storage.user.dao.UserMessages4CHDao;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;
import ru.yandex.webmaster3.storage.user.message.UserMessageInfo;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.service.AllVerifiedHostUsersCacheService;
import ru.yandex.webmaster3.storage.user.service.InitializedUsersService;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class AutoNotificationsSenderService {
    public static final String SOLOMON_LABEL_MESSAGE_TYPE = "message_type";
    public static final String SOLOMON_LABEL_NOTIFICATION_CHANNEL = "notification_channel";

    private static final int CACHE_EXPIRATION_DURATION = 60; // seconds

    private static final Object KEY = new Object();
    private LoadingCache<Object, Map<MessageTypeEnum, Set<SiteProblemTypeEnum>>> blockedMessageTypesCache =
            CacheBuilder.newBuilder().expireAfterWrite(CACHE_EXPIRATION_DURATION, TimeUnit.SECONDS)
                    .build(new BlockedMessageTypesLoader());

    private class BlockedMessageTypesLoader extends CacheLoader<Object, Map<MessageTypeEnum, Set<SiteProblemTypeEnum>>> {

        @Override
        public Map<MessageTypeEnum, Set<SiteProblemTypeEnum>> load(Object key) {
            CommonDataState state = settingsService.getSettingUncached(CommonDataType.BLOCKED_MESSAGE_TYPES);
            if (state == null) {
                return Collections.emptyMap();
            }
            // в строке хранятся MessageTypeEnum через точку с запятой
            // если надо указать SiteProblemTypeEnum - то он отделяется запятой от MessageTypeEnum
            // например
            // SERPLINKS_UPDATE;CHECKLIST_CHANGES,DNS_ERROR,MISSING_FAVICON;URL_SEARCH_LAST_ACCESS_CHANGE
            try {
                Map<MessageTypeEnum, Set<SiteProblemTypeEnum>> result = new EnumMap<>(MessageTypeEnum.class);
                String[] messageTypeParts = state.getValue().split(";");
                for (String messageTypePart : messageTypeParts) {
                    String[] siteProblemTypeParts = messageTypePart.split(",");
                    Set<SiteProblemTypeEnum> set = EnumSet.noneOf(SiteProblemTypeEnum.class);
                    for (int i = 1; i < siteProblemTypeParts.length; i++) {
                        set.add(SiteProblemTypeEnum.valueOf(siteProblemTypeParts[i]));
                    }
                    result.put(MessageTypeEnum.valueOf(siteProblemTypeParts[0]), set);
                }
                return result;
            } catch (Exception e) {
                // если кривота с форматом - не будем ничего ломать
                log.error("Error when loading blocked message types", e);
                return Collections.emptyMap();
            }
        }
    }

    private final UserMessages3CHDao mdbUserMessages3CHDao;
    private final UserMessages4CHDao mdbUserMessages4CHDao;
    private final TrackingEmailSenderService trackingEmailSenderService;
    private final SolomonMetricRegistry solomonMetricRegistry;
    private final SolomonMetricConfiguration solomonMetricConfiguration;
    private final SettingsService settingsService;
    private final SupIntegrationService supIntegrationService;
    private final AbtService abtService;
    private final AllVerifiedHostUsersCacheService allVerifiedHostUsersCacheService;
    private final InitializedUsersService initializedUsersService;

    private Map<Pair<NotificationChannel, String>, SolomonCounter> metrics;

    public void init() {
        metrics = new HashMap<>();
        for (NotificationChannel channel : NotificationChannel.values()) {
            for (MessageTypeEnum messageType : MessageTypeEnum.values()) {
                if (!messageType.isLegacy()) {
                    if (messageType != MessageTypeEnum.CHECKLIST_CHANGES) {
                        getCounter(channel, messageType, null);
                    } else {
                        SiteProblemTypeEnum.ENABLED_PROBLEMS.forEach(p -> getCounter(channel, messageType, p));
                    }
                }
            }
        }
    }

    private SolomonCounter getCounter(NotificationChannel channel, MessageContent messageContent) {
        if (messageContent instanceof MessageContent.ChecklistChanges) {
            return getCounter(channel, messageContent.getType(), ((MessageContent.ChecklistChanges) messageContent).getProblemType());
        } else {
            return getCounter(channel, messageContent.getType(), null);
        }
    }

    private SolomonCounter getCounter(NotificationChannel channel, MessageTypeEnum messageType, SiteProblemTypeEnum problemType) {
        String messageTypeString = getMessageType(messageType, problemType);
        return metrics.computeIfAbsent(
                Pair.of(channel, messageTypeString),
                ign -> solomonMetricRegistry.createCounter(
                        solomonMetricConfiguration,
                        SolomonKey.create(SOLOMON_LABEL_MESSAGE_TYPE, messageTypeString)
                                .withLabel(SOLOMON_LABEL_NOTIFICATION_CHANNEL, channel.name())
                )
        );
    }

    public boolean sendMessage(NotificationInfo notificationInfo) {
        if (notificationInfo.getMessageContent().isRequiresHost() && notificationInfo.getHostId() == null) {
            throw new IllegalArgumentException("Host id is required for message " + notificationInfo.getMessageContent().getType());
        }
        // проверим, не надо ли блокировать данный тип сообщений
        if (isMessageTypeBlocked(notificationInfo.getMessageContent())) {
            log.info("Ignoring message type {} for host {}", getMessageType(notificationInfo.getMessageContent()), notificationInfo.getHostId());
            return true;
        }
        boolean result = false;
        switch (notificationInfo.getChannel()) {
            case EMAIL:
                EmailTrackingInfo trackingInfo = new EmailTrackingInfo(
                        notificationInfo.getHostId(), notificationInfo.getUserId(), getMessageType(notificationInfo.getMessageContent()), DateTime.now());
                result = sendEmail(notificationInfo, trackingInfo);
                break;
            case SERVICE:
                result = sendInService(notificationInfo.getId(), notificationInfo.getHostId(), notificationInfo.getUserId(), notificationInfo.isCritical(), notificationInfo.getMessageContent());
                break;
            case SUP:
                if (abtService.isInExperiment(notificationInfo.getUserId(), Experiment.SUP_NOTIFICATION_SUBSCRIPTION)) {
                    result = sendSupNotification(notificationInfo.getHostId(), notificationInfo.getUserId(), notificationInfo.getPersonalInfo(), notificationInfo.getMessageContent());
                }
                break;
            default:
                throw new IllegalArgumentException("Unknown notification channel " + notificationInfo.getChannel());
        }
        if (result) {
            reportNotificationSent(notificationInfo.getChannel(), notificationInfo.getMessageContent());
        }
        return result;
    }

    private boolean sendSupNotification(WebmasterHostId hostId, long userId, UserPersonalInfo personalInfo, MessageContent messageContent) {
        final SupNotificationTemplateUtil.SupContent nC = SupNotificationTemplateUtil.createNotificationContent(hostId, personalInfo, messageContent);
        boolean result = false;
        if (nC != null) {
            SupIntegrationService.PushData pushData = new SupIntegrationService.PushData(nC.getPushId(), nC.getTitle(), nC.getProjectTitle(), nC.getShortTitle(), nC.getBody(), nC.getLink(), List.of(userId));
            result = supIntegrationService.send(pushData);
        }
        var notificationContent = BellNotificationTemplateUtil.createNotificationContent(hostId, messageContent);
        if (notificationContent != null) {
            supIntegrationService.sendBell(List.of(userId), notificationContent);
        }
        return result;
    }

    private boolean isMessageTypeBlocked(MessageContent messageContent) {
        try {
            Set<SiteProblemTypeEnum> set = blockedMessageTypesCache.get(KEY).get(messageContent.getType());
            if (set == null) {
                return false;
            }
            // для чеклиста проверим и что внутри
            if (messageContent.getType() == MessageTypeEnum.CHECKLIST_CHANGES) {
                return set.isEmpty() || set.contains(((MessageContent.ChecklistChanges) messageContent).getProblemType());
            }
            return true;
        } catch (ExecutionException e) {
            log.error("Error reading value from cache", e);
            return false;
        }
    }

    private boolean sendInService(UUID id, WebmasterHostId hostId, long userId, boolean critical,
                                  MessageContent messageContent) {
        try {
            mdbUserMessages3CHDao.sendMessage(UserMessageInfo.createWithEventUUID(id, userId, hostId, critical, messageContent));
            try {
                UUID userUUID = getUserUUID(userId);
                if (userUUID != null) {
                    mdbUserMessages4CHDao.sendMessage(UserMessageInfo.createWithEventUUID(id, userUUID, hostId, critical, messageContent));
                } else {
                    log.error("No UUID for user {}", userId);
                }
            } catch (Exception e) {
                log.error("NEW_USER_MESSAGES error", e);
            }

            return true;
        } catch (ClickhouseException e) {
            log.error("Failed to send message " + id, e);
            return false;
        }
    }

    @Nullable
    private UUID getUserUUID(long userId) {
        var uuid = allVerifiedHostUsersCacheService.getUserUUIDNoBlock(userId);
        if (uuid == null) {
            initializedUsersService.createUUIDRecordIfNeeded(userId);
            uuid = initializedUsersService.getUUIDById(userId);
        }

        return uuid;
    }

    private boolean sendEmail(NotificationInfo notificationInfo, EmailTrackingInfo trackingInfo) {
        if (notificationInfo.getPersonalInfo().getLanguage() == LanguageEnum.TR) {
            //TODO: we don't have tr neither en localizations, waiting for WMC-1360
            log.error("Will not send email for user " + notificationInfo.getPersonalInfo().getLogin() + ", language tr is not supported yet");
            return true;
        }
        NotificationsTemplateUtil.EmailContent emailContent = NotificationsTemplateUtil.createEmailContent(notificationInfo);
        log.info("Before sending email {} to user {} {}", getMessageType(notificationInfo.getMessageContent()), notificationInfo.getPersonalInfo().getUserId(), notificationInfo.getEmail());
        return trackingEmailSenderService.sendEmail(notificationInfo.getEmail(), notificationInfo.getPersonalInfo().getFio(), emailContent.getSubject(), emailContent.getBody(), trackingInfo);
    }

    private void reportNotificationSent(NotificationChannel channel, MessageContent messageContent) {
        SolomonCounter counter = getCounter(channel, messageContent);
        if (counter == null) {
            log.error("Got report about legacy message type " + messageContent.getType());
        } else {
            counter.update();
        }
    }

    private static String getMessageType(MessageTypeEnum messageType, SiteProblemTypeEnum problemType) {
        if (messageType == MessageTypeEnum.CHECKLIST_CHANGES && problemType != null) {
            return messageType.name() + "-" + problemType.name();
        } else {
            return messageType.name();
        }
    }

    private static String getMessageType(MessageContent messageContent) {
        if (messageContent instanceof MessageContent.ChecklistChanges) {
            return getMessageType(messageContent.getType(), ((MessageContent.ChecklistChanges) messageContent).getProblemType());
        } else {
            return getMessageType(messageContent.getType(), null);
        }
    }
}
