package ru.yandex.webmaster3.worker.notifications.auto;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Triple;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.core.checklist.data.CompanyInfo;
import ru.yandex.webmaster3.core.checklist.data.NotMobileFriendlyStatus;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemSeverityEnum;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.codes.HttpCodeGroup;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.feeds.feed.FeedsErrorSeverity;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.notification.UTMLabels;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedInfo;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.util.URLEncodeUtil;
import ru.yandex.webmaster3.core.util.UrlUtils;
import ru.yandex.webmaster3.core.util.WwwUtil;
import ru.yandex.webmaster3.storage.events.service.MissingDataException;
import ru.yandex.webmaster3.storage.importanturls.data.ImportantUrlChange;
import ru.yandex.webmaster3.storage.notifications.ReceiverUserInfo;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.message.MessageTypeEnum;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.message.iks.IksMessageContent;
import ru.yandex.webmaster3.storage.user.message.iks.IksMessageType;
import ru.yandex.webmaster3.tanker.I18nEmailCommons;
import ru.yandex.webmaster3.tanker.I18nEmailFreeHeader;
import ru.yandex.webmaster3.tanker.I18nEmailNewSeverityStatus;
import ru.yandex.webmaster3.tanker.I18nEmailSearchBaseUpdateGroupedByUser;
import ru.yandex.webmaster3.tanker.I18nEmailSeverityStatus;
import ru.yandex.webmaster3.tanker.I18nEmailTexts;
import ru.yandex.webmaster3.tanker.I18nHttpCodeGroup;
import ru.yandex.webmaster3.tanker.I18nNotMobileFriendlyGroup;
import ru.yandex.webmaster3.tanker.I18nSiteProblemEmail;
import ru.yandex.webmaster3.tanker.I18nSiteProblemEmailBody;
import ru.yandex.webmaster3.tanker.I18nSiteProblemEmailSubject;
import ru.yandex.webmaster3.tanker.I18nTurboSourceTexts;
import ru.yandex.webmaster3.worker.importanturls.I18nUtils;

import static ru.yandex.webmaster3.core.turbo.TurboConstants.PREVIEW_URL_SUFFIX;
import static ru.yandex.webmaster3.core.turbo.TurboConstants.TURBO_BASE_URL;
import static ru.yandex.webmaster3.core.turbo.TurboConstants.TURBO_LISTINGS_SUFFIX;

/**
 * Created by ifilippov5 on 23.03.17.
 */
@Slf4j
public class NotificationsTemplateUtil {
    public static Map<LanguageEnum, String> headerLogo = new EnumMap<>(LanguageEnum.class);
    public static Map<LanguageEnum, String> headerDarkLogo = new EnumMap<>(LanguageEnum.class);
    public static Map<LanguageEnum, String> footerLogo = new EnumMap<>(LanguageEnum.class);

    public static final String HEADER_ICON = "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-wrench-white.png";
    public static final String HEADER_DARK_ICON = "https://avatars.mds.yandex.net/get-adv/56893/2a000001765d0b9737e7b1537769ca2dba1d/orig";
    private static final String DEFAULT_UNSUBSCRIBE_LINK = "https://webmaster.yandex.ru/settings/?";
    public static final String DEFAULT_MAIN_PAGE_LINK = "https://webmaster.yandex.ru/?";
    private static final String IMPORTANT_URLS_UNSUBSCRIBE_LINK_FORMAT = "https://webmaster.yandex.ru/site/%s/indexing/url-tracker/?_showSettings=yes&%s";
    private static final String FORCE_UNSUBSCRIBE_LINK = "https://webmaster.yandex.ru/unsubscribe/";
    private static final String TURBO_FEED_HISTORY_LINK_FORMAT = "https://webmaster.yandex.ru/site/%s/%s/sources/history/?feedUrl=%s&date=%s";
    private static final String ERROR_IN_SITEMAP_SAMPLE_FORMAT = "<li><a href=\"https://webmaster.yandex.ru/site/%s/indexing/sitemap/?sitemapId=%s&parentSitemapId=%s&%s\">%s</a></li>";
    private static final String SEARCHABLE_LINK = "https://webmaster.yandex.ru/site/indexing/searchable";
    public static final String BLOG_LINK = "https://webmaster.yandex.ru/blog/settings/?";
    public static final String ZEN_LINK = "https://zen.yandex.ru/webmaster/?";
    private static final DateTimeFormatter TURBO_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");
    private static final String TAG_BR = "<br />";

    //footer dark logo ru - /get-turbo/2980965/2a00000177252941d82f8a1f9bdb0fee0004/orig
    //footer dark logo en - /get-turbo/2791069/2a00000177252ec527c6e5d02dc2595b1f33/orig
    static {
        headerLogo.put(LanguageEnum.EN, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-white.png");
        headerLogo.put(LanguageEnum.TR, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-white.png");
        headerLogo.put(LanguageEnum.RU, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-white.png");
        headerLogo.put(LanguageEnum.UK, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-white.png");

        footerLogo.put(LanguageEnum.EN, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-full-lang-en-gray.png");
        footerLogo.put(LanguageEnum.TR, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-full-lang-en-gray.png");
        footerLogo.put(LanguageEnum.RU, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-full-lang-ru-gray.png");
        footerLogo.put(LanguageEnum.UK, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-full-lang-ru-gray.png");

        headerDarkLogo.put(LanguageEnum.EN, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-dark.png");
        headerDarkLogo.put(LanguageEnum.TR, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-dark.png");
        headerDarkLogo.put(LanguageEnum.RU, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-dark.png");
        headerDarkLogo.put(LanguageEnum.UK, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-dark.png");

    }

    public static EmailContent createEmailContent(NotificationInfo notificationInfo) {
        LanguageEnum language = notificationInfo.getPersonalInfo().getLanguage();
        WebmasterHostId hostId = notificationInfo.getHostId();
        MessageContent messageContent = notificationInfo.getMessageContent();
        StringBuilder subject = new StringBuilder();

        StringBuilder body = new StringBuilder();

        String hostName = hostId2Name(hostId);
        String hostIdForUrl = hostIdForUrl(hostId);

        MessageTypeEnum messageType = messageContent.getType();
        String utmCampaign = messageType.name().toLowerCase();
        SiteProblemSeverityEnum severity = null;
        String term = null;
        if (messageContent instanceof MessageContent.ChecklistChanges) {
            SiteProblemTypeEnum problemTypeEnum = ((MessageContent.ChecklistChanges) messageContent).getProblemType();
            severity = problemTypeEnum.getSeverity();
            utmCampaign = "checklist_changes_" + problemTypeEnum.name().toLowerCase();
            if (((MessageContent.ChecklistChanges) messageContent).getAttempt() > 0) {
                term = "repeated";
            }
        }
        if (messageContent instanceof MessageContent.IksNewInfo) {
            final IksMessageType iksMessageType = ((MessageContent.IksNewInfo) messageContent).getMessageType();
            utmCampaign = "iks_" + iksMessageType.name().toLowerCase();
        }
        if (messageContent instanceof MessageContent.TurboAutoparsedPagesAppeared) {
            utmCampaign = "autoturbo";
        }
        if (notificationInfo.getExpType() != null) {
            utmCampaign = utmCampaign + "_" + notificationInfo.getExpType();
        }
        // UTM-метки
        UTMLabels utmLabels = UTMLabels.createEmail(utmCampaign, LocalDate.now(TimeUtils.EUROPE_MOSCOW_ZONE), term, notificationInfo.getExtendedInfo());
        String unsubscribeLink;
        if (messageContent instanceof ImportantUrlChange) {
            unsubscribeLink = String.format(IMPORTANT_URLS_UNSUBSCRIBE_LINK_FORMAT, hostIdForUrl, utmLabels.getLabelsInQuery());
        } else if (messageContent instanceof MessageContent.HostAccessDelegatedToNotUser) {
            unsubscribeLink = FORCE_UNSUBSCRIBE_LINK + ((MessageContent.HostAccessDelegatedToNotUser) messageContent).getToken();
        } else {
            unsubscribeLink = DEFAULT_UNSUBSCRIBE_LINK + utmLabels.getLabelsInQuery();
        }
        String mainPageLink = DEFAULT_MAIN_PAGE_LINK + utmLabels.getLabelsInQuery();

        I18nEmailCommons.NEW_HEADER
                .newBuilder(language)
                .utm(utmLabels)
                .headerLogo(NotificationsTemplateUtil.headerLogo.get(language))
                .headerIcon(HEADER_ICON)
                .darkHeaderLogo(NotificationsTemplateUtil.headerDarkLogo.get(language))
                .darkHeaderIcon(HEADER_DARK_ICON)
                .mainPageLink(mainPageLink)
                .renderTo(body);
        if (severity == null) {
            I18nEmailFreeHeader.WITHOUT_STATUS.createRenderableText(language).renderTo(body);
        } else {
            I18nEmailNewSeverityStatus.fromEnum(severity)
                    .createRenderableText(language)
                    .renderTo(body);
        }

        I18nEmailCommons.HELLO
                .newBuilder(language)
                .utm(utmLabels)
                .login(notificationInfo.getPersonalInfo().getLogin())
                .renderTo(body);

        int headerLength = body.length();

        switch (messageType) {
            case CHECKLIST_CHANGES:
                MessageContent.ChecklistChanges checklistMessage = (MessageContent.ChecklistChanges) messageContent;

                I18nSiteProblemEmailSubject.fromEnum(checklistMessage.getProblemType())
                        .newBuilder(language)
                        .host(hostName)
                        .renderTo(subject);
                NotificationsTemplateUtil.buildChecklistChangesMessage(language, body, hostName, hostIdForUrl, utmLabels, checklistMessage, subject);
                break;

            case SERPLINKS_UPDATE:
//                MessageContent.SerpLinksUpdate serpLinksUpdate = (MessageContent.SerpLinksUpdate) messageContent;

                I18nEmailTexts.SERPLINKS_UPDATE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.SERPLINKS_UPDATE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;

            case MAIN_MIRROR_CHANGED:
                MessageContent.MainMirrorChanged mainMirrorChanged = (MessageContent.MainMirrorChanged) messageContent;

                String oldMirrorName = hostId2Name(mainMirrorChanged.getOldMirrorId());
                String newMirrorName = hostId2Name(mainMirrorChanged.getNewMirrorId());

                I18nEmailTexts.MAIN_MIRROR_CHANGED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .oldMirror(oldMirrorName)
                        .newMirror(newMirrorName)
                        .renderTo(subject);

                I18nEmailTexts.MAIN_MIRROR_CHANGED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .oldMirror(oldMirrorName)
                        .newMirror(newMirrorName)
                        .renderTo(body);
                break;
            case AUTO_ADD_MAIN_MIRROR:
                MessageContent.AutoAddMainMirror autoAddMainMirror = (MessageContent.AutoAddMainMirror) messageContent;

                String oldMirror = hostId2Name(autoAddMainMirror.getOldMirrorId());
                String newMirror = hostId2Name(autoAddMainMirror.getNewMirrorId());

                I18nEmailTexts.AUTO_ADD_MAIN_MIRROR_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .oldMirror(oldMirror)
                        .newMirror(newMirror)
                        .renderTo(subject);

                I18nEmailTexts.AUTO_ADD_MAIN_MIRROR_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .oldMirror(oldMirror)
                        .newMirror(newMirror)
                        .newMirrorId(autoAddMainMirror.getNewMirrorId().toString())
                        .renderTo(body);
                break;

            case NO_MAIN_MIRRORS:
                MessageContent.NoMainMirrors noMainMirrors = (MessageContent.NoMainMirrors) messageContent;

                String mainMirror = hostId2Name(noMainMirrors.getMainMirrorId());

                I18nEmailTexts.NO_MAIN_MIRRORS_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(noMainMirrors.getHostId().toString())
                        .mainMirrorId(noMainMirrors.getMainMirrorId().toString())
                        .mainMirror(mainMirror)
                        .renderTo(subject);

                I18nEmailTexts.NO_MAIN_MIRRORS_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(noMainMirrors.getHostId().toString())
                        .mainMirrorId(noMainMirrors.getMainMirrorId().toString())
                        .mainMirror(mainMirror)
                        .renderTo(body);
                break;

            case SEARCH_BASE_UPDATE:
                MessageContent.SearchBaseUpdate searchBaseUpdate = (MessageContent.SearchBaseUpdate) messageContent;

                I18nEmailTexts.SEARCH_BASE_UPDATE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(searchBaseUpdate.getDate())
                        .renderTo(subject);

                I18nEmailTexts.SEARCH_BASE_UPDATE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(searchBaseUpdate.getDate())
                        .renderTo(body);
                break;

            case SEARCH_BASE_UPDATE_ALL_USER_HOSTS:
                MessageContent.SearchBaseUpdateNewAllUserHosts searchBaseUpdateNew = (MessageContent.SearchBaseUpdateNewAllUserHosts) messageContent;

                I18nEmailSearchBaseUpdateGroupedByUser.SEARCH_BASE_UPDATE_BATCH_subject
                        .newBuilder(language)
                        .date(searchBaseUpdateNew.getDate())
                        .renderTo(subject);

                if (searchBaseUpdateNew.getHostsInfo().size() == 1) {
                    MessageContent.SearchBaseUpdateNew singleHostInfo = searchBaseUpdateNew.getHostsInfo().get(0);
                    hostId = singleHostInfo.getHostId();
                    hostIdForUrl = hostIdForUrl(hostId);
                    hostName = hostId2Name(hostId);
                    I18nEmailTexts.SEARCH_BASE_UPDATE_NEW_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .date(searchBaseUpdateNew.getDate())
                            .host(hostName)
                            .hostId(hostIdForUrl)
                            .newPages(String.valueOf(singleHostInfo.getNewPages()))
                            .gonePages(String.valueOf(singleHostInfo.getGonePages()))
                            .totalPages(String.valueOf(singleHostInfo.getTotalPages()))
                            .newUrlExamplesHtml(singleHostInfo.getNewUrlExamplesHtml().isEmpty() ? null :
                                    singleHostInfo.getNewUrlExamplesHtml())
                            .goneUrlExamplesHtml(singleHostInfo.getGoneUrlExamplesHtml().isEmpty() ? null :
                                    singleHostInfo.getGoneUrlExamplesHtml())
                            .renderTo(body);
                } else {

                    I18nEmailSearchBaseUpdateGroupedByUser.SEARCH_BASE_UPDATE_body
                            .newBuilder(language)
                            .date(searchBaseUpdateNew.getDate())
                            .renderTo(body);

                    for (MessageContent.SearchBaseUpdateNew info : searchBaseUpdateNew.getHostsInfo()) {
                        String currentHostName = MessageContent.SearchBaseUpdateNewAllUserHosts.linkToHtml(hostId2Name(info.getHostId()));
                        if (info.hasUpdates()) {
                            I18nEmailSearchBaseUpdateGroupedByUser.SEARCH_BASE_UPDATE_HOST_WITH_UPDATE_body
                                    .newBuilder(language)
                                    .host(currentHostName)
                                    .totalPages(String.valueOf(info.getTotalPages()))
                                    .newPages(String.valueOf(info.getNewPages()))
                                    .gonePages(String.valueOf(info.getGonePages()))
                                    .renderTo(body);
                            body.append("<br/>");
                        }
                    }
                    String withoutUpdate = MessageContent.SearchBaseUpdateNewAllUserHosts.getHostsHtml(searchBaseUpdateNew.getHostsInfo());
                    if (!withoutUpdate.isEmpty()) {
                        I18nEmailSearchBaseUpdateGroupedByUser.SEARCH_BASE_UPDATE_HOSTS_WITHOUT_UPDATE_body
                                .newBuilder(language)
                                .hostsList(withoutUpdate)
                                .renderTo(body);
                    }
                    if (withoutUpdate.isEmpty()) {
                        body.append("<br/>");
                    }

                    I18nEmailSearchBaseUpdateGroupedByUser.SEARCH_BASE_UPDATE_link
                            .newBuilder(language)
                            .link(SEARCHABLE_LINK)
                            .hostId(searchBaseUpdateNew.getHostsInfo().get(0).getHostId().toStringId())
                            .utm(utmLabels)
                            .renderTo(body);
                }
                break;
            case URL_TITLE_CHANGE:
                ImportantUrlChange.TitleChange titleChange = (ImportantUrlChange.TitleChange) messageContent;

                I18nEmailTexts.URL_TITLE_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(titleChange.getUrl())
                        .newTitle(titleChange.getNewTitle())
                        .renderTo(subject);

                I18nEmailTexts.URL_TITLE_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(titleChange.getUrl())
                        .oldTitle(titleChange.getOldTitle())
                        .newTitle(titleChange.getNewTitle())
                        .hostId(titleChange.getHostId().toStringId())
                        .relativeUrl(titleChange.getRelativeUrl())
                        .renderTo(body);
                break;
            case URL_DESCRIPTION_CHANGE: {
                ImportantUrlChange.DescriptionChange descriptionChange =
                        (ImportantUrlChange.DescriptionChange) messageContent;
                I18nEmailTexts.URL_DESCRIPTION_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(descriptionChange.getUrl())
                        .renderTo(subject);

                I18nEmailTexts.URL_DESCRIPTION_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(descriptionChange.getUrl())
                        .oldDescription(descriptionChange.getOldDescription())
                        .newDescription(descriptionChange.getNewDescription())
                        .hostId(descriptionChange.getHostId().toStringId())
                        .relativeUrl(descriptionChange.getRelativeUrl())
                        .renderTo(body);
                break;
            }
            case URL_REL_CANONICAL_TARGET_CHANGE: {
                ImportantUrlChange.RelCanonicalTargetChange relCanonicalTargetChange =
                        (ImportantUrlChange.RelCanonicalTargetChange) messageContent;
                I18nEmailTexts.URL_REL_CANONICAL_TARGET_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(relCanonicalTargetChange.getUrl())
                        .renderTo(subject);

                I18nEmailTexts.URL_REL_CANONICAL_TARGET_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(relCanonicalTargetChange.getUrl())
                        .oldRelCanonicalTarget(relCanonicalTargetChange.getOldRelCanonicalTarget())
                        .newRelCanonicalTarget(relCanonicalTargetChange.getNewRelCanonicalTarget())
                        .hostId(relCanonicalTargetChange.getHostId().toStringId())
                        .relativeUrl(relCanonicalTargetChange.getRelativeUrl())
                        .renderTo(body);
                break;
            }
            case URL_INDEXING_LAST_ACCESS_CHANGE:
                ImportantUrlChange.IndexingLastAccessChange lastAccessChange =
                        (ImportantUrlChange.IndexingLastAccessChange) messageContent;

                I18nEmailTexts.URL_INDEXING_LAST_ACCESS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(lastAccessChange.getUrl())
                        .newLastAccess(lastAccessChange.getNewLastAccess())
                        .renderTo(subject);

                I18nEmailTexts.URL_INDEXING_LAST_ACCESS_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(lastAccessChange.getUrl())
                        .oldLastAccess(lastAccessChange.getOldLastAccess())
                        .newLastAccess(lastAccessChange.getNewLastAccess())
                        .hostId(lastAccessChange.getHostId().toStringId())
                        .relativeUrl(lastAccessChange.getRelativeUrl())
                        .renderTo(body);
                break;

            case URL_INDEXING_STATUS_CHANGE:
                ImportantUrlChange.HttpCodeChange indexingStatusChange =
                        (ImportantUrlChange.HttpCodeChange) messageContent;

                I18nEmailTexts.URL_INDEXING_STATUS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(indexingStatusChange.getUrl())
                        .newHttpCode(I18nUtils.getHttpCodeGroupName(indexingStatusChange.getNewHttpCode(), language))
                        .renderTo(subject);

                I18nEmailTexts.URL_INDEXING_STATUS_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(indexingStatusChange.getUrl())
                        .oldHttpCode(I18nUtils.getHttpCodeGroupName(indexingStatusChange.getOldHttpCode(), language))
                        .newHttpCode(I18nUtils.getHttpCodeGroupName(indexingStatusChange.getNewHttpCode(), language))
                        .hostId(indexingStatusChange.getHostId().toStringId())
                        .relativeUrl(indexingStatusChange.getRelativeUrl())
                        .renderTo(body);
                break;

            case URL_SEARCH_LAST_ACCESS_CHANGE:
                ImportantUrlChange.SearchLastAccessChange searchLastAccessChange =
                        (ImportantUrlChange.SearchLastAccessChange) messageContent;

                I18nEmailTexts.URL_SEARCH_LAST_ACCESS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(searchLastAccessChange.getUrl())
                        .newLastAccess(searchLastAccessChange.getNewLastAccess())
                        .renderTo(subject);

                I18nEmailTexts.URL_SEARCH_LAST_ACCESS_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(searchLastAccessChange.getUrl())
                        .oldLastAccess(searchLastAccessChange.getOldLastAccess())
                        .newLastAccess(searchLastAccessChange.getNewLastAccess())
                        .hostId(searchLastAccessChange.getHostId().toStringId())
                        .relativeUrl(searchLastAccessChange.getRelativeUrl())
                        .renderTo(body);
                break;

            case URL_SEARCH_STATUS_CHANGE:
                ImportantUrlChange.SearchUrlStatusChange searchUrlStatusChange =
                        (ImportantUrlChange.SearchUrlStatusChange) messageContent;

                I18nEmailTexts.URL_SEARCH_STATUS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(searchUrlStatusChange.getUrl())
                        .newSearchUrlStatus(I18nUtils.getUrlStatusTitle(
                                searchUrlStatusChange.getNewSearchUrlStatus(), language))
                        .renderTo(subject);

                String extendedDescription = I18nUtils.extendedDescription(
                        hostId, searchUrlStatusChange.getRelativeUrl(),
                        searchUrlStatusChange.getNewSearchUrlStatus(), language
                );

                I18nEmailTexts.URL_SEARCH_STATUS_CHANGE_body_extended
                        .newBuilder(language)
                        .utm(utmLabels)
                        .url(searchUrlStatusChange.getUrl())
                        .oldSearchUrlStatus(I18nUtils.getUrlStatusTitle(
                                searchUrlStatusChange.getOldSearchUrlStatus(), language))
                        .newSearchUrlStatus(I18nUtils.getUrlStatusTitle(
                                searchUrlStatusChange.getNewSearchUrlStatus(), language))
                        .hostId(searchUrlStatusChange.getHostId().toStringId())
                        .relativeUrl(searchUrlStatusChange.getRelativeUrl())
                        .description(extendedDescription)
                        .renderTo(body);
                break;
            case HOST_ACCESS_DELEGATED_SOMEONE_ELSE:
                I18nEmailTexts.HOST_ACCESS_DELEGATED_SOMEONE_ELSE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_ACCESS_DELEGATED_SOMEONE_ELSE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_ACCESS_DELEGATED_TO_NOT_USER:
            case HOST_ACCESS_DELEGATED_TO_USER:
                I18nEmailTexts.HOST_ACCESS_DELEGATED_TO_USER_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .renderTo(subject);

                I18nEmailTexts.HOST_ACCESS_DELEGATED_TO_USER_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .renderTo(body);
                break;
            case HOST_ACCESS_LOST:
                I18nEmailTexts.HOST_ACCESS_LOST_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_ACCESS_LOST_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_TREND_OPENED:
                I18nEmailTexts.HOST_TREND_OPENED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_TREND_OPENED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_RECOMMENDED_OPENED:
                I18nEmailTexts.HOST_RECOMMENDED_OPENED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_RECOMMENDED_OPENED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_RECOMMENDED_CLOSED:
                I18nEmailTexts.HOST_RECOMMENDED_CLOSED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.HostRecommendedClosed hostRecommendedClosed =
                        (MessageContent.HostRecommendedClosed) messageContent;

                I18nEmailTexts.HOST_RECOMMENDED_CLOSED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .date(hostRecommendedClosed.getDate())
                        .renderTo(body);
                break;
            case HOST_DISPLAY_NAME_CHANGED:
                I18nEmailTexts.HOST_DISPLAY_NAME_CHANGED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_DISPLAY_NAME_CHANGED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_DISPLAY_NAME_CHANGE_CANCELLED:
                I18nEmailTexts.HOST_DISPLAY_NAME_CHANGE_CANCELLED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_DISPLAY_NAME_CHANGE_CANCELLED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_METRIKA_COUNTER_BINDING_REQUEST:
                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_REQUEST_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.HostMetrikaCounterBindingRequest hostMetrikaCounterBindingRequest =
                        (MessageContent.HostMetrikaCounterBindingRequest) messageContent;

                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_REQUEST_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .counterId(String.valueOf(hostMetrikaCounterBindingRequest.getCounterId()))
                        .renderTo(body);
                break;
            case HOST_METRIKA_COUNTER_BINDING_REQUEST_CANCELLED:
                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_REQUEST_CANCELLED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.HostMetrikaCounterBindingRequestCancelled hostMetrikaCounterBindingRequestCancelled =
                        (MessageContent.HostMetrikaCounterBindingRequestCancelled) messageContent;

                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_REQUEST_CANCELLED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .counterId(String.valueOf(hostMetrikaCounterBindingRequestCancelled.getCounterId()))
                        .renderTo(body);
                break;
            case HOST_METRIKA_COUNTER_BINDING_DELETED:
                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_DELETED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.HostMetrikaCounterBindingDeleted hostMetrikaCounterBindingDeleted =
                        (MessageContent.HostMetrikaCounterBindingDeleted) messageContent;

                I18nEmailTexts.HOST_METRIKA_COUNTER_BINDING_DELETED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .counterId(String.valueOf(hostMetrikaCounterBindingDeleted.getCounterId()))
                        .user(hostMetrikaCounterBindingDeleted.getUser())
                        .renderTo(body);
                break;
            case HOST_METRIKA_COUNTER_CRAWL_SUSPENDED:
                var hostMetrikaCounterCrawlSuspended = (MessageContent.HostMetrikaCounterCrawlSuspended) messageContent;

                I18nEmailTexts.HOST_METRIKA_COUNTER_CRAWL_SUSPENDED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .counterId(String.valueOf(hostMetrikaCounterCrawlSuspended.getCounterId()))
                        .renderTo(subject);

                I18nEmailTexts.HOST_METRIKA_COUNTER_CRAWL_SUSPENDED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .counterId(String.valueOf(hostMetrikaCounterCrawlSuspended.getCounterId()))
                        .renderTo(body);
                break;
            case HOST_METRIKA_COUNTER_CRAWL_SAMPLES_UPDATED:
                var hostMetrikaCounterCrawlSamplesUpdated = (MessageContent.HostMetrikaCounterCrawlSamplesUpdated) messageContent;

                I18nEmailTexts.HOST_METRIKA_COUNTER_CRAWL_SAMPLES_UPDATED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .counterId(String.valueOf(hostMetrikaCounterCrawlSamplesUpdated.getCounterId()))
                        .renderTo(subject);

                I18nEmailTexts.HOST_METRIKA_COUNTER_CRAWL_SAMPLES_UPDATED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .counterId(String.valueOf(hostMetrikaCounterCrawlSamplesUpdated.getCounterId()))
                        .renderTo(body);
                break;
            case HOST_REGION_CHANGED:
                I18nEmailTexts.HOST_REGION_CHANGED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_REGION_CHANGED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case HOST_REGION_CHANGE_CANCELLED:
                I18nEmailTexts.HOST_REGION_CHANGE_CANCELLED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.HOST_REGION_CHANGE_CANCELLED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case TURBO_AUTOPARSED_PAGES_APPEARED:
                I18nEmailTexts.TURBO_AUTOPARSED_PAGES_APPEARED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_AUTOPARSED_PAGES_APPEARED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case TURBO_LISTINGS_AVAILABLE:
                I18nEmailTexts.TURBO_LISTINGS_AVAILABLE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_LISTINGS_AVAILABLE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .turboListingsPreviewUrl(TURBO_BASE_URL + WwwUtil.cutWWWAndM(hostId) + TURBO_LISTINGS_SUFFIX + PREVIEW_URL_SUFFIX)
                        .renderTo(body);
                break;
            case TURBO_ADV_SETTINGS_CHANGE: {
                MessageContent.TurboAdvSettingsChanged asc = (MessageContent.TurboAdvSettingsChanged) messageContent;
                I18nEmailTexts.TURBO_ADV_SETTINGS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_ADV_SETTINGS_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .platform(asc.getPlatform().name())
                        .placement(asc.getPlacement().name())
                        .renderTo(body);
                break;
            }
            case TURBO_COMMERCE_SETTINGS_CHANGE:
                I18nEmailTexts.TURBO_COMMERCE_SETTINGS_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_COMMERCE_SETTINGS_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case URL_TITLE_CHANGE_ALL_HOST_PAGES:
            case URL_DESCRIPTION_CHANGE_ALL_HOST_PAGES:
            case URL_REL_CANONICAL_TARGET_CHANGE_ALL_HOST_PAGES:
            case URL_INDEXING_LAST_ACCESS_CHANGE_ALL_HOST_PAGES:
            case URL_INDEXING_STATUS_CHANGE_ALL_HOST_PAGES:
            case URL_SEARCH_LAST_ACCESS_CHANGE_ALL_HOST_PAGES:
            case URL_SEARCH_STATUS_CHANGE_ALL_HOST_PAGES:
                NotificationsImportantPagesTemplateUtil.createEmailContent(language, utmLabels, hostName,
                        hostId, subject, body, messageContent, messageType);
                break;
            case ROBOTS_TXT_CHANGE: {
                I18nEmailTexts.ROBOTS_TXT_CHANGE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.RobotsTxtChange robotsTxtChange = (MessageContent.RobotsTxtChange) messageContent;

                I18nEmailTexts.ROBOTS_TXT_CHANGE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .date(robotsTxtChange.getDate())
                        .renderTo(body);
                break;
            }
            case NEW_DOMAINS_NOTIFICATION: {
                MessageContent.NewDomainsNotification newDomainsNotification = (MessageContent.NewDomainsNotification)
                        messageContent;
                I18nEmailTexts.NEW_DOMAINS_NOTIFICATION_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.NEW_DOMAINS_NOTIFICATION_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .newDomainsHtml(newDomainsNotification.renderNewDomainsHtml())
                        .renderTo(body);
                break;
            }
            case IMPORTANT_URLS_AUTO_ADD: {
                MessageContent.AutoAddedImportantUrls autoAddedImportantUrls = (MessageContent.AutoAddedImportantUrls)
                        messageContent;
                I18nEmailTexts.IMPORTANT_URLS_AUTO_ADD_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.IMPORTANT_URLS_AUTO_ADD_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .addedUrlsHtml(autoAddedImportantUrls.renderAddedUrlsHtml())
                        .renderTo(body);
                break;

            }
            case UNVERIFIED_HOST_REMOVAL_AFTER_WEEK: {
                I18nEmailTexts.UNVERIFIED_HOST_REMOVAL_AFTER_WEEK_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.UNVERIFIED_HOST_REMOVAL_AFTER_WEEK_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(body);
                break;

            }
            case UNVERIFIED_HOST_REMOVAL_AFTER_THREE_WEEK: {
                I18nEmailTexts.UNVERIFIED_HOST_REMOVAL_AFTER_THREE_WEEK_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.UNVERIFIED_HOST_REMOVAL_AFTER_THREE_WEEK_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(body);
                break;

            }
            case NEW_REVIEW: {
                MessageContent.NewReviewAvailable newReviewAvailable = (MessageContent.NewReviewAvailable) messageContent;
                I18nEmailTexts.NEW_REVIEW_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);
                NewReviewNotificationTemplate.renderTypeTwo(body, language, utmLabels, hostName, hostId, newReviewAvailable);
                break;
            }
            case IKS_UPDATE: {
                buildIksUpdateNotification(language, body, subject, hostName, hostIdForUrl, utmLabels, (MessageContent.IksNewInfo) messageContent);
                break;
            }
            case TURBO_SCC_BANNED:
                I18nEmailTexts.TURBO_SCC_BANNED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_SCC_BANNED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case TURBO_SCC_FAILED:
                I18nEmailTexts.TURBO_SCC_FAILED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_SCC_FAILED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case TURBO_SCC_PASS:
                I18nEmailTexts.TURBO_SCC_PASS_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_SCC_PASS_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case TURBO_SCC_UNBANNED:
                I18nEmailTexts.TURBO_SCC_UNBANNED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.TURBO_SCC_UNBANNED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case VIDEOHOST_OFFER_CONFIRMED:
                I18nEmailTexts.VIDEOHOST_OFFER_CONFIRMED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                MessageContent.VideohostOfferConfirmed videohostOfferConfirmed = (MessageContent.VideohostOfferConfirmed) messageContent;

                I18nEmailTexts.VIDEOHOST_OFFER_CONFIRMED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .offerUrl(videohostOfferConfirmed.getOfferUrl())
                        .renderTo(body);
                break;
            case VIDEOHOST_OFFER_HAS_PAPER:
                I18nEmailTexts.VIDEOHOST_OFFER_HAS_PAPER_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.VIDEOHOST_OFFER_HAS_PAPER_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;
            case VIDEOHOST_OFFER_REDUNDANT:
                I18nEmailTexts.VIDEOHOST_OFFER_REDUNDANT_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .renderTo(subject);

                I18nEmailTexts.VIDEOHOST_OFFER_REDUNDANT_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(body);
                break;

            case FEEDS_SCC_BANNED:
                I18nEmailTexts.FEEDS_SCC_BANNED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(subject);

                MessageContent.FeedsSccBanned feedsSccBanned = (MessageContent.FeedsSccBanned) messageContent;

                I18nEmailTexts.FEEDS_SCC_BANNED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .feedUrl(feedsSccBanned.getFeedUrls().get(0))
                        .renderTo(body);
                break;
            case FEEDS_PINGER_ENABLE:
                I18nEmailTexts.FEEDS_PINGER_ENABLE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(subject);

                MessageContent.FeedsPingerEnable feedsPingerEnable = (MessageContent.FeedsPingerEnable) messageContent;

                I18nEmailTexts.FEEDS_PINGER_ENABLE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .feedUrl(feedsPingerEnable.getFeedUrl())
                        .renderTo(body);
                break;
            case FEEDS_PINGER_DISABLE:
                I18nEmailTexts.FEEDS_PINGER_DISABLE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(subject);

                MessageContent.FeedsPingerDisable feedsPingerDisable = (MessageContent.FeedsPingerDisable) messageContent;

                I18nEmailTexts.FEEDS_PINGER_DISABLE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .feedUrl(feedsPingerDisable.getFeedUrl())
                        .renderTo(body);
                break;
            case FEEDS_SCC_FAILED:
                MessageContent.FeedsSccFailed feedsSccFailed = (MessageContent.FeedsSccFailed) messageContent;
                if (feedsSccFailed.getAttempt() > 0) {
                    I18nEmailTexts.FEEDS_SCC_FAILED_REPEATED_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_SCC_FAILED_REPEATED_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .feedUrls(feedsSccFailed.getFeedUrls().stream().map(f -> "<li>" + f + "</li>").limit(5).collect(Collectors.joining()))
                            .renderTo(body);
                } else {
                    I18nEmailTexts.FEEDS_SCC_FAILED_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_SCC_FAILED_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .feedUrl(feedsSccFailed.getFeedUrls().get(0))
                            .renderTo(body);
                }
                break;
            case FEEDS_VALIDATION_FAILED:
                MessageContent.FeedsValidationFailed feedsValidationFailed = (MessageContent.FeedsValidationFailed) messageContent;
                String feedUrlsJoined = feedsValidationFailed.getFeedUrls().stream().map(f -> "<li>" + f + "</li>").limit(5).collect(Collectors.joining());
                if (feedsValidationFailed.getAttempt() > 0) {
                    I18nEmailTexts.FEEDS_VALIDATION_FAILED_REPEATED_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_VALIDATION_FAILED_REPEATED_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .feedUrls(feedUrlsJoined)
                            .renderTo(body);
                } else {
                    I18nEmailTexts.FEEDS_VALIDATION_FAILED_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_VALIDATION_FAILED_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .feedUrls(feedUrlsJoined)
                            .renderTo(body);
                }
                break;
            case FEEDS_DEFECT_RATE_FAILED: {
                MessageContent.FeedsDefectRateFailed feedsDrFailed = (MessageContent.FeedsDefectRateFailed) messageContent;
                if (feedsDrFailed.getSeverity() == FeedsErrorSeverity.WARNING) {
                    I18nEmailTexts.FEEDS_DEFECT_RATE_WARNING_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_DEFECT_RATE_WARNING_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(body);
                } else {
                    I18nEmailTexts.FEEDS_DEFECT_RATE_ERROR_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(subject);
                    I18nEmailTexts.FEEDS_DEFECT_RATE_ERROR_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostId.toString())
                            .renderTo(body);
                }
                break;
            }
            case FEEDS_SCC_PASSED:
            {
                MessageContent.FeedsSccPassed feedsSccPassed = (MessageContent.FeedsSccPassed) messageContent;
                I18nEmailTexts.FEEDS_SCC_PASSED_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(subject);
                I18nEmailTexts.FEEDS_SCC_PASSED_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .feedUrls(feedsSccPassed.getFeedUrls().stream().map(f -> "<li>" + f + "</li>").limit(5).collect(Collectors.joining()))
                        .renderTo(body);
                break;
            }
            case NCA_CTLOG_CERTIFICATE:
                MessageContent.NcaCtlogCertificate ncaCtlogCertificate = (MessageContent.NcaCtlogCertificate) messageContent;

                I18nEmailTexts.NCA_CTLOG_CERTIFICATE_subject
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .renderTo(subject);

                I18nEmailTexts.NCA_CTLOG_CERTIFICATE_body
                        .newBuilder(language)
                        .utm(utmLabels)
                        .host(hostName)
                        .hostId(hostId.toString())
                        .domain(ncaCtlogCertificate.getDomain())
                        .ctlog(ncaCtlogCertificate.getCtlog())
                        .ctlogRow(String.valueOf(ncaCtlogCertificate.getCtlogRow()))
                        .renderTo(body);
                break;
            default:
                throw new IllegalArgumentException("Unsupported message type " + messageContent.getType());
        }

        // WMC-6616 check for content between HELLO and FOOTER
        if (body.length() == headerLength) {
            throw new MissingDataException("Empty email body, messageType = " + messageContent.getType() +
                    ", userId = " + notificationInfo.getPersonalInfo().getUserId() + ", hostId = " + hostId);
        }
        if (subject.length() == 0) {
            throw new MissingDataException("Empty email subject, messageType = " + messageContent.getType() +
                    ", userId = " + notificationInfo.getPersonalInfo().getUserId() + ", hostId = " + hostId);
        }

        if (messageType == MessageTypeEnum.TURBO_LISTINGS_AVAILABLE) { // TODO добавить автопарсер
            I18nEmailCommons.FOOTER_WITHOUT_UNSUBSCRIBE
                    .newBuilder(language)
                    .utm(utmLabels)
                    .year(String.valueOf(LocalDate.now().getYear()))
                    .renderTo(body);
        } else {
            I18nEmailCommons.NEW_FOOTER
                    .newBuilder(language)
                    .utm(utmLabels)
                    .unsubscribeLink(unsubscribeLink)//TODO: unsubscribe link
                    .mainPageLink(mainPageLink)
                    .year(String.valueOf(LocalDate.now().getYear()))
                    .blogLink(BLOG_LINK + utmLabels.getLabelsInQuery())
                    .zenLink(ZEN_LINK + utmLabels.getLabelsInQuery())
                    .footerLogo(footerLogo.get(language))
                    .renderTo(body);
        }

        return new EmailContent(subject.toString(), body.toString());
    }

    static void buildChecklistChangesMessage(LanguageEnum language, StringBuilder body, String hostName,
                                             String hostIdForUrl, UTMLabels utmLabels,
                                             MessageContent.ChecklistChanges msg, StringBuilder subject) {
        switch (msg.getProblemType()) {
            case SANCTIONS:
                I18nSiteProblemEmailBody.SANCTIONS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case SECURITY:
                I18nSiteProblemEmailBody.SECURITY.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case SITEMAP_NOT_SET:
                I18nSiteProblemEmailBody.SITEMAP_NOT_SET.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case ERRORS_IN_SITEMAPS:
                SiteProblemContent.ErrorsInSitemaps errorsInSitemaps = (SiteProblemContent.ErrorsInSitemaps) msg.getProblemContent();
                I18nSiteProblemEmailBody.ERRORS_IN_SITEMAPS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .sitemapSamples(errorsInSitemaps.getSitemaps().stream()
                                .map(si -> String.format(ERROR_IN_SITEMAP_SAMPLE_FORMAT, hostIdForUrl, si.getId(), si.getParentId(), utmLabels.getLabelsInQuery(), si.getUrl()))
                                .collect(Collectors.joining()))
                        .renderTo(body);
                break;
            case NO_REGIONS:
                I18nSiteProblemEmailBody.NO_REGIONS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_DICTIONARY_REGIONS:
                I18nSiteProblemEmailBody.NO_DICTIONARY_REGIONS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_SERPLINKS:
                I18nSiteProblemEmailBody.NO_SERPLINKS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DISALLOWED_IN_ROBOTS:
                I18nSiteProblemEmailBody.DISALLOWED_IN_ROBOTS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DNS_ERROR:
                I18nSiteProblemEmailBody.DNS_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case CONNECT_FAILED:
                I18nSiteProblemEmailBody.CONNECT_FAILED.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_404_ERRORS:
                I18nSiteProblemEmailBody.NO_404_ERRORS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case ROBOTS_TXT_ERROR:
                I18nSiteProblemEmailBody.ROBOTS_TXT_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DOMAIN_EXPIRES:
                I18nSiteProblemEmailBody.DOMAIN_EXPIRES.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case MORDA_ERROR:
                // логика для расширенного кода HTTP сперта с фронта
                SiteProblemContent.MordaError mordaError = (SiteProblemContent.MordaError) msg.getProblemContent();
                HttpCodeGroup httpCodeGroup = mordaError.getHttpCodeInfo() == null ? HttpCodeGroup.UNKNOWN :
                        mordaError.getHttpCodeInfo().getHttpCodeGroup();
                String httpCodeStatus;
                if (mordaError.getHttpStatusCode() != null && mordaError.getHttpStatusCode() >= 100 &&
                        mordaError.getHttpStatusCode() < 600) {
                    // если стандартный код - оставляем код и текст после
                    // отрезаем HTTP_число, заменяем подчеркивания на пробелы, приводим к Camel case
                    String desc = StringUtils.capitalize(httpCodeGroup.name().replaceFirst("HTTP_\\d+_", "")
                            .replaceAll("_", " ").toLowerCase());
                    httpCodeStatus = mordaError.getHttpStatusCode() + " " + desc;
                } else {
                    // для нестандартного берем текст из танкера
                    try {
                        httpCodeStatus = I18nHttpCodeGroup.fromEnum(httpCodeGroup).getText(language);
                    } catch (IllegalArgumentException e) {
                        httpCodeStatus = I18nHttpCodeGroup.UNKNOWN.getText(language);
                    }
                }
                I18nSiteProblemEmailBody.MORDA_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .httpCodeStatus(httpCodeStatus)
                        .renderTo(body);
                break;
            case MORDA_REDIRECTS:
                I18nSiteProblemEmailBody.MORDA_REDIRECTS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case MISSING_FAVICON:
                I18nSiteProblemEmailBody.MISSING_FAVICON.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TOO_MANY_URL_DUPLICATES:
                I18nSiteProblemEmailBody.TOO_MANY_URL_DUPLICATES.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_ROBOTS_TXT:
                I18nSiteProblemEmailBody.NO_ROBOTS_TXT.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_HOST_DIRECTIVE:
                I18nSiteProblemEmailBody.NO_HOST_DIRECTIVE.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case AMBIGUOUS_HOST_DIRECTIVES:
                I18nSiteProblemEmailBody.AMBIGUOUS_HOST_DIRECTIVES.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DOCUMENTS_MISSING_TITLE:
                I18nSiteProblemEmailBody.DOCUMENTS_MISSING_TITLE.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DOCUMENTS_MISSING_DESCRIPTION:
                I18nSiteProblemEmailBody.DOCUMENTS_MISSING_DESCRIPTION.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_SITEMAP_MODIFICATIONS:
                I18nSiteProblemEmailBody.NO_SITEMAP_MODIFICATIONS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TOO_MANY_BROKEN_LINKS:
                I18nSiteProblemEmailBody.TOO_MANY_BROKEN_LINKS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NOT_MOBILE_FRIENDLY:
                SiteProblemContent.NotMobileFriendly notMobileFriendly = (SiteProblemContent.NotMobileFriendly) msg.getProblemContent();
                int notMobileFriendlyStatus = notMobileFriendly.getNotMobileFriendlyStatus();
                StringBuilder alertText = new StringBuilder();
                if (NotMobileFriendlyStatus.NOT_MOBILE_FRIENDLY.isPartOf(notMobileFriendlyStatus)) {
                    alertText.append(I18nNotMobileFriendlyGroup.fromEnum(NotMobileFriendlyStatus.NOT_MOBILE_FRIENDLY).getText(language));
                }

                if (NotMobileFriendlyStatus.BAD_LINK_ALTERNATE.isPartOf(notMobileFriendlyStatus)) {
                    alertText.append(I18nNotMobileFriendlyGroup.fromEnum(NotMobileFriendlyStatus.BAD_LINK_ALTERNATE).getText(language));
                }

                if (NotMobileFriendlyStatus.BAD_HOST_REPLACED.isPartOf(notMobileFriendlyStatus)) {
                    alertText.append(I18nNotMobileFriendlyGroup.fromEnum(NotMobileFriendlyStatus.BAD_HOST_REPLACED).getText(language));
                }

                I18nSiteProblemEmailBody.NOT_MOBILE_FRIENDLY_EXT.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .alertText(alertText.toString())
                        .renderTo(body);
                break;
            //todo Вырезать когда перейдем на SLOW_AVG_RESPONSE_WITH_EXAMPLES
            case SLOW_AVG_RESPONSE:
                I18nSiteProblemEmailBody.SLOW_AVG_RESPONSE.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case SLOW_AVG_RESPONSE_WITH_EXAMPLES:
                final SiteProblemContent.SlowResponseWithExamples slowResponse = (SiteProblemContent.SlowResponseWithExamples) msg.getProblemContent();
                I18nSiteProblemEmailBody.SLOW_AVG_RESPONSE_WITH_EXAMPLES.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .slowUrlSamples(slowResponse.searializeSamplesAsHtml())
                        .renderTo(body);
                break;
            case THREATS:
                I18nSiteProblemEmailBody.THREATS.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_METRIKA_COUNTER:
                I18nSiteProblemEmailBody.NO_METRIKA_COUNTER.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_ERROR:
                SiteProblemContent.TurboError turboError = (SiteProblemContent.TurboError) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboError.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboError.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_RSS_ERROR:
                SiteProblemContent.TurboRssError turboRssError = (SiteProblemContent.TurboRssError) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_RSS_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboRssError.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboRssError.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_YML_ERROR:
                SiteProblemContent.TurboYmlError turboYmlError = (SiteProblemContent.TurboYmlError) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_YML_ERROR.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboYmlError.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboYmlError.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_WARNING:
                SiteProblemContent.TurboWarning turboWarning = (SiteProblemContent.TurboWarning) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_WARNING.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboWarning.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboWarning.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_RSS_WARNING:
                SiteProblemContent.TurboRssWarning turboRssWarning = (SiteProblemContent.TurboRssWarning) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_RSS_WARNING.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboRssWarning.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboRssWarning.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_YML_WARNING:
                SiteProblemContent.TurboYmlWarning turboYmlWarning = (SiteProblemContent.TurboYmlWarning) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_YML_WARNING.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboYmlWarning.getFeeds().stream()
                                .map(feed -> renderTurboHistoryLink(language, hostIdForUrl, feed))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboYmlWarning.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case YABROWSER_BADAD:
                I18nSiteProblemEmailBody.YABROWSER_BADAD.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_FEED_BAN:
                SiteProblemContent.TurboFeedBan turboFeedBan = (SiteProblemContent.TurboFeedBan) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_FEED_BAN.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboFeedBan.getFeeds().stream().map(link -> renderLink(link, link))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboFeedBan.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_DOCUMENT_BAN:
                SiteProblemContent.TurboDocumentBan turboDocumentBan = (SiteProblemContent.TurboDocumentBan) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_DOCUMENT_BAN.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .feeds(turboDocumentBan.getFeeds().stream().map(link -> renderLink(link, link))
                                .collect(Collectors.joining(TAG_BR)))
                        .showFeedList(Boolean.valueOf(!turboDocumentBan.getFeeds().isEmpty()).toString())
                        .renderTo(body);
                break;
            case TURBO_HOST_BAN_INFO:
                I18nSiteProblemEmailBody.TURBO_HOST_BAN_INFO.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_HOST_BAN:
                SiteProblemContent.TurboHostBan turboHostBan = (SiteProblemContent.TurboHostBan) msg.getProblemContent();
                subject.delete(0, subject.length());
                if (turboHostBan.getSeverity() == SiteProblemContent.TurboHostBan.TurboHostBanSeverity.BAN) {
                    I18nSiteProblemEmailBody.TURBO_HOST_BAN_severity_BAN.newBuilder(language)
                            .utm(utmLabels)
                            .date(msg.getDetectionDate())
                            .host(hostName)
                            .hostId(hostIdForUrl)
                            .renderTo(body);
                    I18nSiteProblemEmailSubject.TURBO_HOST_BAN_severity_BAN.newBuilder(language)
                            .host(hostName)
                            .renderTo(subject);
                } else if (turboHostBan.getSeverity() == SiteProblemContent.TurboHostBan.TurboHostBanSeverity.WARNING) {
                    I18nSiteProblemEmailBody.TURBO_HOST_BAN_severity_WARNING.newBuilder(language)
                            .utm(utmLabels)
                            .date(msg.getDetectionDate())
                            .host(hostName)
                            .hostId(hostIdForUrl)
                            .renderTo(body);
                    I18nSiteProblemEmailSubject.TURBO_HOST_BAN_severity_WARNING.newBuilder(language)
                            .host(hostName)
                            .renderTo(subject);
                } else {
                    log.error("UNKNOWN TURBO_HOST_BAN severity {}", turboHostBan.getSeverity());
                }
                break;
            case TURBO_INSUFFICIENT_CLICKS_SHARE: {
                SiteProblemContent.TurboInsufficientClicksShare content =
                        (SiteProblemContent.TurboInsufficientClicksShare) msg.getProblemContent();
                I18nSiteProblemEmailBody.TURBO_INSUFFICIENT_CLICKS_SHARE.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .percentage(String.valueOf(content.getSharePercentage()))
                        .renderTo(body);
                break;
            }
            case HOST_COMPANY_PROFILE_NOT_FILLED: {
                SiteProblemContent.HostCompanyProfileNotFilled hostCompanyProfileNotFilled =
                        (SiteProblemContent.HostCompanyProfileNotFilled) msg.getProblemContent();

                List<StringBuilder> companyLinks = new ArrayList<>();
                for (CompanyInfo info : hostCompanyProfileNotFilled.getCompanyInfos()) {
                    StringBuilder companyLink = new StringBuilder();
                    I18nSiteProblemEmail.HOST_COMPANY_link
                            .newBuilder(language)
                            .utm(utmLabels)
                            .companyName(info.getName())
                            .companyAddress(info.getAddress())
                            .tycoonId(String.valueOf(info.getTycoonId()))
                            .renderTo(companyLink);
                    companyLinks.add(companyLink);
                }
                String companyLinksStr = companyLinks.stream()
                        .collect(Collectors.joining(", "));
                I18nSiteProblemEmailBody.HOST_COMPANY_PROFILE_NOT_FILLED
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .companyLinks(companyLinksStr)
                        .renderTo(body);
                break;
            }
            case HOST_COMPANY_PROFILE_CREATED: {
                var hostCompanyProfileCreated = (SiteProblemContent.HostCompanyProfileCreated) msg.getProblemContent();
                I18nSiteProblemEmailBody.HOST_COMPANY_PROFILE_CREATED
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .tycoonLink(hostCompanyProfileCreated.getTycoonLink())
                        .renderTo(body);
                break;
            }
            case TOO_MANY_DOMAINS_ON_SEARCH: {
                SiteProblemContent.TooManyDomainsOnSearch tooManyDomainsOnSearch =
                        (SiteProblemContent.TooManyDomainsOnSearch) msg.getProblemContent();

                String domainsHtml = tooManyDomainsOnSearch.getSubdomains().stream()
                        .map(d -> String.format("<li>%s</li>", d))
                        .collect(Collectors.joining("\n", "<ul>", "</ul>"));
                I18nSiteProblemEmailBody.TOO_MANY_DOMAINS_ON_SEARCH
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .domainsHtml(domainsHtml)
                        .renderTo(body);
                break;
            }
            case MAIN_MIRROR_IS_NOT_HTTPS: {
                I18nSiteProblemEmailBody.MAIN_MIRROR_IS_NOT_HTTPS
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            }
            case SSL_CERTIFICATE_ERROR: {
                I18nSiteProblemEmailBody.SSL_CERTIFICATE_ERROR
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            }
            case FAVICON_ERROR: {
                I18nSiteProblemEmailBody.FAVICON_ERROR
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            }
            case NO_METRIKA_COUNTER_CRAWL_ENABLED:
                I18nSiteProblemEmailBody.NO_METRIKA_COUNTER_CRAWL_ENABLED
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case NO_METRIKA_COUNTER_BINDING:
                I18nSiteProblemEmailBody.NO_METRIKA_COUNTER_BINDING
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_URL_ERRORS:
                I18nSiteProblemEmailBody.TURBO_URL_ERRORS
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_INVALID_CART_URL:
                I18nSiteProblemEmailBody.TURBO_INVALID_CART_URL
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case TURBO_LISTING_ERROR: {
                throw new IllegalStateException("Should never happen. Checklist notifications for turbo listings errors are unavailable");
            }
            case BIG_FAVICON_ABSENT:
                I18nSiteProblemEmailBody.BIG_FAVICON_ABSENT
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case URL_ALERT_4XX:
                I18nSiteProblemEmailBody.URL_ALERT_4XX
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case URL_ALERT_5XX:
                I18nSiteProblemEmailBody.URL_ALERT_5XX
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case DISALLOWED_URLS_ALERT:
                I18nSiteProblemEmailBody.DISALLOWED_URLS_ALERT
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            case INSIGNIFICANT_CGI_PARAMETER: {
                SiteProblemContent.InsignificantCGIParameter problemContent = (SiteProblemContent.InsignificantCGIParameter) msg.getProblemContent();
                StringBuilder samplesBuilder = new StringBuilder(1024);
                List<Triple<Integer, String, String>> samples = new ArrayList<>();
                problemContent.getCgiParameters().forEach((cgi, urls) -> {
                    for (int i = 0; i < urls.size(); i++) {
                        samples.add(Triple.of(i, cgi, urls.get(i)));
                    }
                });
                samples.sort(Comparator.<Triple<Integer, String, String>, Integer>comparing(Triple::getLeft).thenComparing(Triple::getMiddle));
                List<Triple<Integer, String, String>> limitedSamples = samples.subList(0, Math.min(samples.size(), 5));
                limitedSamples.sort(Comparator.comparing(Triple::getMiddle));

                for (Triple<Integer, String, String> triple : limitedSamples) {
                    String cgi = triple.getMiddle();
                    String url = hostName + triple.getRight();
                    String mainUrl = UrlUtils.removeParameter(url, cgi);
                    samplesBuilder.append("<li>").append(cgi).append(":<br/>");
                    if (mainUrl != null) {
                        samplesBuilder.append(String.format("<a href=\"%1$s\">%1$s</a><br/>", mainUrl));
                    }
                    samplesBuilder.append(String.format("<a href=\"%1$s\">%1$s</a><br/>", url));
                    samplesBuilder.append("</li>");
                }
                I18nSiteProblemEmailBody.INSIGNIFICANT_CGI_PARAMETER
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .samples(samplesBuilder.toString())
                        .renderTo(body);
                break;
            }
            case DUPLICATE_CONTENT_ATTRS: {
                var content = (SiteProblemContent.DuplicateContentAttrs) msg.getProblemContent();
                String host = IdUtils.hostIdToUrl(msg.getHostId());
                StringBuilder titleSamples = new StringBuilder();
                for (var sample : content.getPrevailingTitles()) {
                    I18nSiteProblemEmail.DUPLICATE_TITLE_SAMPLE_item
                            .newBuilder(language)
                            .utm(utmLabels)
                            .value(StringUtils.abbreviate(sample.getValue(), 100))
                            .count(String.valueOf(sample.getCount()))
                            .pageSamples(sample.getPageSamples().stream().map(ps -> "<li>" + renderLink(host + ps.getPath(), ps.getPath()) + "</li>").collect(Collectors.joining()))
                            .renderTo(titleSamples);
                }
                StringBuilder descriptionSamples = new StringBuilder();
                for (var sample : content.getPrevailingDescriptions()) {
                    I18nSiteProblemEmail.DUPLICATE_DESCRIPTION_SAMPLE_item
                            .newBuilder(language)
                            .utm(utmLabels)
                            .value(StringUtils.abbreviate(sample.getValue(), 100))
                            .count(String.valueOf(sample.getCount()))
                            .pageSamples(sample.getPageSamples().stream().map(ps -> "<li>" + renderLink(host + ps.getPath(), ps.getPath()) + "</li>").collect(Collectors.joining()))
                            .renderTo(titleSamples);
                }

                I18nSiteProblemEmailBody.DUPLICATE_CONTENT_ATTRS
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .duplicateTitleSamples(titleSamples.toString())
                        .duplicateDescriptionSamples(descriptionSamples.toString())
                        .renderTo(body);
                break;
            }
            case DUPLICATE_PAGES: {
                var content = (SiteProblemContent.DuplicatePages) msg.getProblemContent();
                String host = IdUtils.hostIdToUrl(msg.getHostId());
                StringBuilder pageSamples = new StringBuilder();
                for (var sample : content.getSamples()) {
                    I18nSiteProblemEmail.DUPLICATE_PAGE_SAMPLE_item
                            .newBuilder(language)
                            .utm(utmLabels)
                            .url(host + sample.getPath())
                            .mainUrl(sample.getMainUrl())
                            .renderTo(pageSamples);
                }

                I18nSiteProblemEmailBody.DUPLICATE_PAGES
                        .newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .duplicatePageSamples(pageSamples.toString())
                        .renderTo(body);
                break;
            }
            case VIDEOHOST_OFFER_FAILED: {
                SiteProblemContent.VideohostOfferFailed problemContent = (SiteProblemContent.VideohostOfferFailed) msg.getProblemContent();
                I18nSiteProblemEmailBody.VIDEOHOST_OFFER_FAILED.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .offerUrl(problemContent.getOfferUrl())
                        .renderTo(body);
                break;
            }
            case VIDEOHOST_OFFER_IS_NEEDED: {
                I18nSiteProblemEmailBody.VIDEOHOST_OFFER_IS_NEEDED.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            }
            case VIDEOHOST_OFFER_NEED_PAPER: {
                I18nSiteProblemEmailBody.VIDEOHOST_OFFER_NEED_PAPER.newBuilder(language)
                        .utm(utmLabels)
                        .date(msg.getDetectionDate())
                        .host(hostName)
                        .hostId(hostIdForUrl)
                        .renderTo(body);
                break;
            }
            default:
                throw new IllegalStateException("Unknown site problem type: " + msg.getProblemType());
        }
        if (msg.getAttempt() > 0) {
            // блок для повторных сообщений
            I18nEmailCommons.REPEATED_PROBLEM_NOTIFICATION_FOOTER.newBuilder(language)
                    .utm(utmLabels)
                    .actualSince(Objects.requireNonNullElse(msg.getActualSince(), msg.getDetectionDate()))
                    .lastUpdate(msg.getDetectionDate())
                    .renderTo(body);
        }
    }

    public static void buildIksUpdateNotification(LanguageEnum language,
                                                  StringBuilder body,
                                                  StringBuilder subject,
                                                  String hostName,
                                                  String hostIdForUrl,
                                                  UTMLabels utmLabels,
                                                  MessageContent.IksNewInfo msg) {


        switch (msg.getMessageType()) {
            case UPDATE: {
                final IksMessageContent.IksUpdate msgUpdate = (IksMessageContent.IksUpdate) msg.getIksMessageContent();
                if ("TYPE_3".equals(msg.getGroupType())) {
                    I18nEmailTexts.IKS_UPDATE_V3_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .renderTo(subject);
                    final String iksName = I18nEmailTexts.IKS.newBuilder(language).utm(utmLabels).render();
                    StringBuilder hostLists = new StringBuilder();
                    for (int i = 0; i < msgUpdate.getHostIds().size(); i++) {
                        hostLists.append("<li>").append(IdUtils.hostIdToUrl(msgUpdate.getHostIds().get(i))).append(": ").append(iksName).append(" ").append(msgUpdate.getIksValues().get(i)).append("</li>");
                    }
                    I18nEmailTexts.IKS_UPDATE_V3_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .hostList(hostLists.toString())
                            .renderTo(body);

                } else {
                    I18nEmailTexts.IKS_UPDATE_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .renderTo(subject);
                    I18nEmailTexts.IKS_UPDATE_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostIdForUrl)
                            .renderTo(body);
                }
                break;
            }
            case RIVAL_UPDATE: {
                final IksMessageContent.IksRival msgUpdate = (IksMessageContent.IksRival) msg.getIksMessageContent();
                if ("TYPE_ONE".equals(msg.getGroupType())) {
                    I18nEmailTexts.IKS_RIVALS_ONE_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .renderTo(subject);
                    I18nEmailTexts.IKS_RIVALS_ONE_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .host(hostName)
                            .hostId(hostIdForUrl)
                            .renderTo(body);
                } else {
                    final List<String> rivalHosts = msgUpdate.getHosts();
                    StringBuilder linkText = new StringBuilder();
                    for (String rivalHost : rivalHosts) {
                        linkText.append("<li>");
                        I18nEmailTexts.IKS_RIVALS_MULTI_QUALITY_LINK
                                .newBuilder(language)
                                .utm(utmLabels)
                                .host(IdUtils.hostIdToReadableUrl(IdUtils.stringToHostId(rivalHost)))
                                .hostId(rivalHost)
                                .renderTo(linkText);
                        linkText.append("</li>");
                    }
                    I18nEmailTexts.IKS_RIVALS_MULTI_subject
                            .newBuilder(language)
                            .utm(utmLabels)
                            .renderTo(subject);
                    I18nEmailTexts.IKS_RIVALS_MULTI_body
                            .newBuilder(language)
                            .utm(utmLabels)
                            .hostList(linkText.toString())
                            .renderTo(body);
                }
                break;
            }
        }


    }

    private static String renderLink(String url, String text) {
        return "<a href=\"" + url + "\">" + text + "</a>";
    }

    private static String renderTurboHistoryLink(LanguageEnum language, String hostId, TurboFeedInfo feedInfo) {
        // ссылка на сам фид
        String feedLink = renderLink(feedInfo.getUrl(), feedInfo.getUrl());
        // ссылка на фид в истории
        String historyUrl = String.format(TURBO_FEED_HISTORY_LINK_FORMAT, hostId, feedInfo.getType() == TurboFeedType.RSS ? "turbo" : "shop",
                URLEncodeUtil.urlEncode(feedInfo.getUrl()), TURBO_DATE_FORMAT.print(feedInfo.getDownloadDate()));
        String historyLink = renderLink(historyUrl, I18nTurboSourceTexts.history_email.getText(language));
        return feedLink + " " + historyLink;
    }

    private static String hostIdForUrl(WebmasterHostId hostId) {
        return hostId == null ? null : hostId.toString();
    }

    private static String hostId2Name(WebmasterHostId hostId) {
        return hostId == null ? null : IdUtils.toHostString(hostId, false, true, false);
    }

    public static class EmailContent {
        private final String subject;
        private final String body;

        public EmailContent(String subject, String body) {
            this.subject = subject;
            this.body = body;
        }

        public String getSubject() {
            return subject;
        }

        public String getBody() {
            return body;
        }
    }

    public static String addHeaderFooter(ReceiverUserInfo personalInfo, String content) {
        LanguageEnum language = personalInfo.getLanguage();

        StringBuilder body = new StringBuilder();
        String utmCampaign = "service";
        SiteProblemSeverityEnum severity = SiteProblemSeverityEnum.RECOMMENDATION;
        // UTM-метки
        UTMLabels utmLabels = UTMLabels.createEmail(utmCampaign, LocalDate.now(TimeUtils.EUROPE_MOSCOW_ZONE), null);
        String unsubscribeLink = DEFAULT_UNSUBSCRIBE_LINK + utmLabels.getLabelsInQuery();
        String mainPageLink = DEFAULT_MAIN_PAGE_LINK + utmLabels.getLabelsInQuery();

        I18nEmailCommons.HEADER
                .newBuilder(language)
                .utm(utmLabels)
                .headerLogo(NotificationsTemplateUtil.headerLogo.get(language))
                .headerIcon(HEADER_ICON)
                .darkHeaderLogo(NotificationsTemplateUtil.headerDarkLogo.get(language))
                .darkHeaderIcon(HEADER_DARK_ICON)
                .mainPageLink(mainPageLink)
                .renderTo(body);

        I18nEmailSeverityStatus.fromEnum(severity)
                .createRenderableText(language)
                .renderTo(body);

        I18nEmailCommons.HELLO
                .newBuilder(language)
                .utm(utmLabels)
                .login(personalInfo.getLogin())
                .renderTo(body);

        body.append(content);

        I18nEmailCommons.FOOTER
                .newBuilder(language)
                .utm(utmLabels)
                .unsubscribeLink(unsubscribeLink)
                .mainPageLink(mainPageLink)
                .year(String.valueOf(LocalDate.now().getYear()))
                .blogLink(BLOG_LINK + utmLabels.getLabelsInQuery())
                .zenLink(ZEN_LINK + utmLabels.getLabelsInQuery())
                .footerLogo(footerLogo.get(language))
                .renderTo(body);

        return body.toString();
    }

    // WMC-7004
    public static EmailContent renderServiceSubscriptionNotification(UserPersonalInfo info) {
        Preconditions.checkNotNull(info);
        LanguageEnum language = info.getLanguage();
        String login = info.getLogin();
        StringBuilder body = new StringBuilder();
        StringBuilder subject = new StringBuilder();
        SiteProblemSeverityEnum severity = SiteProblemSeverityEnum.RECOMMENDATION;

        UTMLabels utm = UTMLabels.createEmail("subscribed_to_service_notifications", LocalDate.now(), null);
        I18nEmailCommons.HEADER
                .newBuilder(language)
                .utm(utm)
                .headerLogo(headerLogo.get(language))
                .headerIcon(HEADER_ICON)
                .darkHeaderLogo(NotificationsTemplateUtil.headerDarkLogo.get(language))
                .darkHeaderIcon(HEADER_DARK_ICON)
                .mainPageLink(DEFAULT_MAIN_PAGE_LINK + utm.getLabelsInQuery())
                .renderTo(body);

        I18nEmailSeverityStatus.fromEnum(severity)
                .createRenderableText(language)
                .renderTo(body);

        I18nEmailCommons.HELLO
                .newBuilder(language)
                .utm(utm)
                .login(login)
                .renderTo(body);
        I18nEmailTexts.SERVICE_SUBSCRIPTION_NOTIFICATION_subject
                .newBuilder(language)
                .utm(utm)
                .renderTo(subject);
        I18nEmailTexts.SERVICE_SUBSCRIPTION_NOTIFICATION_body
                .newBuilder(language)
                .utm(utm)
                .login(login)
                .renderTo(body);

        I18nEmailCommons.FOOTER_WITHOUT_UNSUBSCRIBE
                .newBuilder(language)
                .utm(utm)
                .year(String.valueOf(LocalDate.now().getYear()))
                .renderTo(body);

        return new EmailContent(subject.toString(), body.toString());
    }
}
