package ru.yandex.webmaster3.worker.notifications.sending;

import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.mutable.MutableInt;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterUser;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.delegation.UserDelegationsForSendYDao;
import ru.yandex.webmaster3.storage.events.data.events.DelegationToNotUserEvent;
import ru.yandex.webmaster3.storage.events.data.events.DelegationToUserEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.user.dao.UserLastVisitYDao;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 09.01.18.
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class SendAccessDelegationNotificationsPeriodicTask extends PeriodicTask<SendAccessDelegationNotificationsPeriodicTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(SendAccessDelegationNotificationsPeriodicTask.class);

    public static final Duration MAXIMUM_DURATION_NOT_VISIT = Duration.standardDays(30 * 6);
    public static final int UUIDS_COUNT_IN_TOKEN = 6;

    private final UserDelegationsForSendYDao userDelegationsForSendYDao;
    private final WMCEventsService wmcEventsService;
    private final UserLastVisitYDao userLastVisitYDao;
    private final UserNotificationSettingsService userNotificationSettingsService;
    private final UserHostsService userHostsService;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new TaskState());

        List<Long> userIds = userDelegationsForSendYDao.getAll();

        MutableInt countUsersSent = new MutableInt(0);
        MutableInt countNotUsersSent = new MutableInt(0);
        userIds.forEach(userId -> {
            if (useWebmaster(userId)) {
                wmcEventsService.addEvent(
                        new DelegationToUserEvent<>(
                                userId,
                                new MessageContent.HostAccessDelegatedToUser(),
                                false)
                );
                countUsersSent.increment();
            } else {
                String token = generateRandomToken();
                wmcEventsService.addEvent(
                        new DelegationToNotUserEvent<>(
                                userId,
                                new MessageContent.HostAccessDelegatedToNotUser(token),
                                false)
                );
                countNotUsersSent.increment();
            }
        });

        userDelegationsForSendYDao.delete(userIds);

        getState().countUsersSent = countUsersSent.getValue();
        getState().countNotUsersSent = countNotUsersSent.getValue();
        return new Result(TaskResult.SUCCESS);
    }

    private boolean useWebmaster(long userId) {
        try {
            if (!userHostsService.getVerifiedHosts(new WebmasterUser(userId)).isEmpty()) {
                return true;
            }
            Instant lastVisit = userLastVisitYDao.getVisit(userId);
            if (lastVisit == null || lastVisit.plus(MAXIMUM_DURATION_NOT_VISIT).isBeforeNow()) {
                return false;
            }
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Unable to get user info",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        }
        return userNotificationSettingsService.getUserEmailOrDefaultIfEmpty(userId) != null;
    }

    private String generateRandomToken() {
        StringBuilder token = new StringBuilder();
        for (int i = 0; i < UUIDS_COUNT_IN_TOKEN; i++) {
            token.append(UUID.randomUUID().toString());
        }
        return token.toString().replace("-", "");
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.SEND_NOTIFICATIONS_ACCESS_DELEGATION;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    public static class TaskState implements PeriodicTaskState {
        private int countUsersSent;
        private int countNotUsersSent;

        public int getCountUsersSent() {
            return countUsersSent;
        }

        public int getCountNotUsersSent() {
            return countNotUsersSent;
        }
    }
}
