package ru.yandex.webmaster3.worker.notifications.services;

import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.concurrent.graph.BlockingBatchConsumer;
import ru.yandex.webmaster3.core.util.concurrent.graph.GraphOutQueue;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.user.dao.UserNotificationEmailYDao;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.worker.notifications.info.UserHostsChannelsInfo;

/**
 * Created by ifilippov5 on 29.08.17.
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class EmailsResolver {
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));

    private final UserNotificationEmailYDao userNotificationEmailYDao;

    public BlockingBatchConsumer<UserHostsChannelsInfo> resolve(GraphOutQueue<UserHostsChannelsInfo> out) {
        return batch -> {
            Map<Long, String> emails = RetryUtils.query(RETRY_POLICY, () ->
                    userNotificationEmailYDao.getUserEmails(batch.stream().map(UserHostsChannelsInfo::getUserId).collect(Collectors.toSet())));
            for (UserHostsChannelsInfo rawInfo : batch) {
                String email = emails.get(rawInfo.getUserId());
                email = StringUtils.isEmpty(email) ? null : email;
                if (email == null) {
                    Map<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> newChannels = new HashMap<>();
                    for (Map.Entry<WebmasterHostId, Map<NotificationType, Set<NotificationChannel>>> hostEntry : rawInfo.getHost2Settings().entrySet()) {
                        for (Map.Entry<NotificationType, Set<NotificationChannel>> typeEntry : hostEntry.getValue().entrySet()) {
                            for (NotificationChannel notificationChannel : typeEntry.getValue()) {
                                if (notificationChannel != NotificationChannel.EMAIL) {
                                    newChannels.computeIfAbsent(hostEntry.getKey(), ign -> new EnumMap<>(NotificationType.class))
                                            .computeIfAbsent(typeEntry.getKey(), ign -> EnumSet.noneOf(NotificationChannel.class))
                                            .add(notificationChannel);
                                }
                            }
                        }
                    }
                    if (!newChannels.isEmpty()) {
                        out.put(new UserHostsChannelsInfo(rawInfo.getUserId(), newChannels, null, null));
                    }
                } else {
                    out.put(new UserHostsChannelsInfo(rawInfo.getUserId(), rawInfo.getHost2Settings(), email, null));
                }
            }
        };
    }
}
