package ru.yandex.webmaster3.worker.notifications.services;

import java.util.EnumMap;
import java.util.Map;

import org.joda.time.LocalDate;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.notification.LanguageEnum;
import ru.yandex.webmaster3.core.notification.UTMLabels;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.storage.postpone.PostWelcomeEmailData;
import ru.yandex.webmaster3.tanker.I18nPostWelcomeEmail;

/**
 * ishalaru
 * 11.03.2020
 **/
@Service("postWelcomeMessageBodyBuilderService")
public class PostWelcomeMessageBodyBuilderService {
    public static Map<LanguageEnum, String> headerLogo = new EnumMap<>(LanguageEnum.class);
    private static final String DEFAULT_UNSUBSCRIBE_LINK = "https://webmaster.yandex.ru/settings/?";


    static {
        headerLogo.put(LanguageEnum.EN, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-white.png");
        headerLogo.put(LanguageEnum.TR, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-en-white.png");
        headerLogo.put(LanguageEnum.RU, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-white.png");
        headerLogo.put(LanguageEnum.UK, "https://yastatic.net/s3/webmaster/webmaster/logo/webmaster-lang-ru-white.png");
    }

    public String buildSubject(PostWelcomeEmailData postWelcomeEmailData, boolean shortMessage) {
        if (shortMessage) {
            return I18nPostWelcomeEmail
                    .SUBJECT_SHORT
                    .newBuilder(postWelcomeEmailData.getLanguage())
                    .utm(makeUTMLabels(shortMessage))
                    .host(hostId2Name(postWelcomeEmailData.getHostId()))
                    .hostId(hostIdForUrl(postWelcomeEmailData.getHostId()))
                    .render();
        }
        return I18nPostWelcomeEmail
                .SUBJECT
                .newBuilder(postWelcomeEmailData.getLanguage())
                .utm(makeUTMLabels(shortMessage))
                .host(hostId2Name(postWelcomeEmailData.getHostId()))
                .hostId(hostIdForUrl(postWelcomeEmailData.getHostId()))
                .render();
    }


    public String buildBody(PostWelcomeEmailData postWelcomeEmailData, boolean shortMessage, boolean ecommerseSite) {

        StringBuilder body = new StringBuilder();
        String host = hostId2Name(postWelcomeEmailData.getHostId());
        String hostId = hostIdForUrl(postWelcomeEmailData.getHostId());
        UTMLabels label = makeUTMLabels(shortMessage);
        final LanguageEnum language = postWelcomeEmailData.getLanguage();
        I18nPostWelcomeEmail
                .HEADER
                .newBuilder(language)
                .utm(label)
                .host(host)
                .hostId(hostId)
                .headerLogo(headerLogo.get(language))
                .renderTo(body);
        if (shortMessage) {
            I18nPostWelcomeEmail
                    .BODY_SHORT
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);
            writeShortTurboInfo(ecommerseSite, language, label, host, hostId, body);
        } else {
            I18nPostWelcomeEmail
                    .BODY
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);
            writeTurboInfo(ecommerseSite, language, label, host, hostId, body);
        }
        I18nPostWelcomeEmail
                .FOOTER
                .newBuilder(language)
                .utm(label)
                .host(host)
                .hostId(hostId)
                .unsubscribeLink(DEFAULT_UNSUBSCRIBE_LINK + label)
                .year(String.valueOf(LocalDate.now().getYear()))
                .renderTo(body);
        return body.toString();
    }

    private void writeShortTurboInfo(boolean ecommerseSite, LanguageEnum language, UTMLabels label, String host, String hostId, StringBuilder body) {
        if (ecommerseSite) {
            I18nPostWelcomeEmail
                    .TURBO_ECOMMERSE_SHORT
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);
        } else {
            I18nPostWelcomeEmail
                    .TURBO_CONTENT_SHORT
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);

        }
    }

    private void writeTurboInfo(boolean ecommerseSite, LanguageEnum language, UTMLabels label, String host, String hostId, StringBuilder body) {
        if (ecommerseSite) {
            I18nPostWelcomeEmail
                    .TURBO_ECOMMERSE
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);
        } else {
            I18nPostWelcomeEmail
                    .TURBO_CONTENT
                    .newBuilder(language)
                    .utm(label)
                    .host(host)
                    .hostId(hostId)
                    .renderTo(body);

        }
    }

    private static String hostIdForUrl(WebmasterHostId hostId) {
        return hostId == null ? null : hostId.toString();
    }

    private static String hostId2Name(WebmasterHostId hostId) {
        return hostId == null ? null : IdUtils.toHostString(hostId, false, true, false);
    }

    public UTMLabels makeUTMLabels(boolean isShort) {
        return UTMLabels.createEmail(isShort ? "postwelcome_short" : "postwelcome_long", LocalDate.now(), null);
    }

}
