package ru.yandex.webmaster3.worker.queue;

import java.security.NoSuchAlgorithmException;
import java.util.BitSet;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.statistic.ApproximateHostCounter;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskType;

/**
 * @author aherman
 */
public class TaskHostStatistics {
    private static final Logger log = LoggerFactory.getLogger(TaskHostStatistics.class);

    private volatile Map<WorkerTaskType, ApproximateHostCounter> currentStatistics = Collections.emptyMap();
    private volatile DateTime statisticsDate;

    public void init() throws NoSuchAlgorithmException {
        updateStatsMap();
    }

    private void updateStatsMap() throws NoSuchAlgorithmException {
        Map<WorkerTaskType, ApproximateHostCounter> map = new HashMap<>();
        for (WorkerTaskType taskType : WorkerTaskType.values()) {
            map.put(taskType, new ApproximateHostCounter(18));
        }

        statisticsDate = DateTime.now();
        currentStatistics = map;
    }

    public Stats getCurrentStatistic(WorkerTaskType taskType) {
        DateTime currentDate = statisticsDate;
        Map<WorkerTaskType, ApproximateHostCounter> currentStatistics = this.currentStatistics;
        ApproximateHostCounter approximateHostCounter = currentStatistics.get(taskType);
        if (approximateHostCounter != null) {
            return new Stats(currentDate, approximateHostCounter.getBits(), approximateHostCounter.getBitSet());
        } else {
            return new Stats(DateTime.now(), 0, null);
        }
    }

    public void recordHostTask(WebmasterHostId hostId, WorkerTaskType taskType) {
        ApproximateHostCounter approximateHostCounter = getStatisticsMap().get(taskType);
        if (approximateHostCounter != null) {
            approximateHostCounter.addHost(hostId);
        }
    }

    private Map<WorkerTaskType, ApproximateHostCounter> getStatisticsMap() {
        if (DateTime.now().minusHours(1).isAfter(statisticsDate)) {
            try {
                updateStatsMap();
            } catch (NoSuchAlgorithmException e) {
                log.warn("Unable to create ");
            }
        }
        return currentStatistics;
    }

    public static class Stats {
        private final DateTime date;
        private final int bits;
        private final BitSet bitSet;

        public Stats(DateTime date, int bits, BitSet bitSet) {
            this.date = date;
            this.bits = bits;
            this.bitSet = bitSet;
        }

        public DateTime getDate() {
            return date;
        }

        public int getBits() {
            return bits;
        }

        public BitSet getBitSet() {
            return bitSet;
        }
    }
}
