package ru.yandex.webmaster3.worker.recommendedquery;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.events.RetranslateToUsersEvent;
import ru.yandex.webmaster3.storage.events.data.events.UserHostMessageEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.recommendedquery.dao.HostsWithOpenedRecommendedYDao;
import ru.yandex.webmaster3.storage.recommendedquery.dao.RecommendedQueriesCHDao;
import ru.yandex.webmaster3.storage.recommendedquery.dao.RecommendedQueriesLastVisitDateYDao;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.user.message.content.MessageContent;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 21.02.18.
 *
 * Закрывает рекомендованные запросы для хоста, если у них долго не было просмотров,
 * в целях экономии ресурсов Прогнозатора. Также инициирует рассылку предупредительных
 * уведомлений о скором закрытии.
 */
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class CloseRecommendedQueriesPeriodicTask extends PeriodicTask<CloseRecommendedQueriesPeriodicTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(CloseRecommendedQueriesPeriodicTask.class);

    public static final int ALLOWED_PERIOD_NON_OPENING_IN_MONTHS = 3;
    public static final int PERIOD_FOR_SEND_NOTIFICATION_ABOUT_CLOSE_IN_WEEKS = 1;

    private final RecommendedQueriesLastVisitDateYDao recommendedQueriesLastVisitDateYDao;
    private final HostsWithOpenedRecommendedYDao hostsWithOpenedRecommendedYDao;
    private final WMCEventsService wmcEventsService;
    private final SettingsService settingsService;
    private final RecommendedQueriesCHDao mdbRecommendedQueriesCHDao;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new TaskState());

        List<WebmasterHostId> needClose = new ArrayList<>();
        List<WebmasterHostId> needSendNotification = new ArrayList<>();
        Map<WebmasterHostId, Boolean> queriesExistMap = new HashMap<>();

        hostsWithOpenedRecommendedYDao.forEach(hostId -> {
            try {
                queriesExistMap.put(hostId, mdbRecommendedQueriesCHDao.exist(hostId, RecommendedQueriesCHDao.Include.REGULAR));
            } catch (ClickhouseException e) {
                log.error("Failed to get any recommended query", e);
                getState().failedCount++;

                return; // ignore
            }

            if (queriesExistMap.size() >= 1000) {
                batchUpdate(queriesExistMap, needClose, needSendNotification);
                queriesExistMap.clear();
            }
        });

        if (!queriesExistMap.isEmpty()) {
            batchUpdate(queriesExistMap, needClose, needSendNotification);
        }

        hostsWithOpenedRecommendedYDao.delete(needClose);
        sendNotifications(needSendNotification);

        // Для аналитики
        CommonDataState state = settingsService.getSettingOrNull(CommonDataType.COUNT_CLOSED_RECOMMENDED);
        int count = (state == null ? 0 : Integer.parseInt(state.getValue()));
        count += needClose.size();
        settingsService.update(CommonDataType.COUNT_CLOSED_RECOMMENDED, String.valueOf(count));

        getState().closedCount = needClose.size();
        getState().sentCount = needSendNotification.size();

        return new Result(TaskResult.SUCCESS);
    }

    private void batchUpdate(Map<WebmasterHostId, Boolean> queriesExistMap, List<WebmasterHostId> needClose,
                             List<WebmasterHostId> needSendNotification) {
        List<Pair<WebmasterHostId, LocalDate>> lastVisitAdd = new ArrayList<>();
        HashMap<WebmasterHostId, LocalDate> lastVisitGet;

        try {
            Preconditions.checkState(queriesExistMap.size() <= 1000);
            lastVisitGet = recommendedQueriesLastVisitDateYDao.getLastVisitDate(queriesExistMap.keySet()).stream().collect(W3Collectors.toHashMap());
        } catch (WebmasterYdbException e) {
            log.error("Unable to get last visit date", e);
            getState().failedCount += queriesExistMap.size();
            return; //ignore
        }

        for (Map.Entry<WebmasterHostId, Boolean> entry : queriesExistMap.entrySet()) {
            WebmasterHostId hostId = entry.getKey();
            LocalDate lastVisitDate = lastVisitGet.get(hostId);
            boolean queriesExist = entry.getValue();

            if (lastVisitDate == null) {
                // Рекомендованные только что посчитались и пользователь их еще не смотрел
                lastVisitDate = LocalDate.now();
                lastVisitAdd.add(Pair.of(hostId, lastVisitDate));
            }

            if (shouldSendWarnNotification(lastVisitDate)) {
                if (queriesExist) {
                    needSendNotification.add(hostId);
                } else {
                    //не будет отправлять уведомление, так как рекомендованных нет
                    //но чтобы потом отправить уведомление, если рекомендованные появятся, сместим lastVisitDate на день
                    lastVisitAdd.add(Pair.of(hostId, lastVisitDate.plusDays(1)));
                }
            }

            if (shouldCloseBecauseNoVisits(lastVisitDate) && queriesExist) {
                needClose.add(hostId);
            }
        }

        try {
            recommendedQueriesLastVisitDateYDao.batchInsert(lastVisitAdd);
        } catch (WebmasterYdbException e) {
            log.error("Unable to update lastVisitDate", e);
            //ignore
        }
    }

    private boolean shouldCloseBecauseNoVisits(LocalDate lastVisitDate) {
        LocalDate bound = getBoundDateForClose(lastVisitDate);
        LocalDate now = LocalDate.now();

        return now.isAfter(bound);
    }

    private boolean shouldSendWarnNotification(LocalDate lastVisitDate) {
        LocalDate bound = getBoundDateForSendNotification(lastVisitDate);
        LocalDate now = LocalDate.now();

        return now.equals(bound);
    }

    private LocalDate getBoundDateForClose(LocalDate date) {
        return date.plusMonths(ALLOWED_PERIOD_NON_OPENING_IN_MONTHS);
    }

    private LocalDate getBoundDateForSendNotification(LocalDate date) {
        return getBoundDateForClose(date).minusWeeks(PERIOD_FOR_SEND_NOTIFICATION_ABOUT_CLOSE_IN_WEEKS);
    }

    private DateTime getBoundDateForCloseAfterSendNotification(DateTime date) {
        return date.plusWeeks(PERIOD_FOR_SEND_NOTIFICATION_ABOUT_CLOSE_IN_WEEKS).withZone(TimeUtils.EUROPE_MOSCOW_ZONE);
    }

    private void sendNotifications(List<WebmasterHostId> needSendNotification) {
        for (WebmasterHostId hostId : needSendNotification) {
            wmcEventsService.addEvent(createEvent(hostId));
        }
    }

    private WMCEvent createEvent(WebmasterHostId hostId) {
        return WMCEvent.create(new RetranslateToUsersEvent<>(
                new UserHostMessageEvent<>(
                        hostId,
                        null,
                        new MessageContent.HostRecommendedClosed(
                                hostId,
                                getBoundDateForCloseAfterSendNotification(DateTime.now()),
                                getBoundDateForCloseAfterSendNotification(DateTime.now())
                        ),
                        NotificationType.RECOMMENDED_QUERIES,
                        false)
        ));
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.CLOSE_RECOMMENDED_QUERIES;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 13 * * *");
    }

    public static class TaskState implements PeriodicTaskState {
        int closedCount;
        int sentCount;
        int failedCount;

        public int getClosedCount() {
            return closedCount;
        }

        public int getSentCount() {
            return sentCount;
        }

        public int getFailedCount() {
            return failedCount;
        }
    }
}
