package ru.yandex.webmaster3.worker.searchbase;

import java.util.NavigableMap;
import java.util.NavigableSet;
import java.util.TreeSet;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.searchbase.SearchBaseUpdateInfo;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.host.service.MirrorService2;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.searchbase.dao.SearchBaseUpdatesYDao;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("checkJupiterBaseUpdateTask")
public class CheckJupiterBaseUpdateTask extends PeriodicTask<PeriodicTaskState> {

    private static final String META_ATTRIBUTE_NAME = "jupiter_meta";
    private static final String META_ITEM_NAME = "production_current_state";

    private final SearchBaseUpdatesYDao searchBaseUpdatesYDao;
    private final YtService ytService;
    private final MirrorService2 mirrorService2;

    @Value("${webmaster3.worker.jupiter.baseMeta.ytPath}")
    private YtPath jupiterMetaNode;

    @Override
    public Result run(UUID runId) throws Exception {
        NavigableMap<Instant, SearchBaseUpdateInfo> history = searchBaseUpdatesYDao.getUpdatesHistory();
        NavigableSet<Instant> knownBaseCollectionDates = history.values()
                .stream()
                .map(SearchBaseUpdateInfo::getBaseCollectionDate)
                .collect(Collectors.toCollection(TreeSet::new));
        ytService.withoutTransaction(cypressService -> {
            Instant baseSwitchDate = Instant.now();
            YtNode node = cypressService.getNode(jupiterMetaNode);
            if (node == null) {
                throw new WebmasterException("Failed to check jupiter base update - node not found " + jupiterMetaNode,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null));
            }

            JsonNode jupiterMetaObject = node.getNodeMeta().get(META_ATTRIBUTE_NAME);
            Instant baseCollectionDate = Instant.parse(jupiterMetaObject.get(META_ITEM_NAME).asText(), JupiterUtils.STATE_DATE_FORMAT);
            log.info("Base collection date: {}", new DateTime(baseCollectionDate));

            if (knownBaseCollectionDates.contains(baseCollectionDate)) {
                log.info("No search base switch detected");
            } else {
                Instant higherKnownCollectionDate = knownBaseCollectionDates.higher(baseCollectionDate);
                if (higherKnownCollectionDate != null) {
                    throw new WebmasterException("Detected switch of search base with collection date " +
                            baseCollectionDate + " while there is at least one known base with higher collection date " +
                            higherKnownCollectionDate,
                            new WebmasterErrorResponse.DataConsistencyErrorResponse(getClass(), null));
                }
                log.info("Search base switch detected: base collection date is {}", baseCollectionDate);
                try {
                    searchBaseUpdatesYDao.insertBaseUpdateInfo(new SearchBaseUpdateInfo(baseSwitchDate, baseCollectionDate));
                } catch (WebmasterYdbException e) {
                    throw new WebmasterException("Failed to store new search base info",
                            new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
                }
            }
            return true;
        });
        return new Result(TaskResult.SUCCESS);
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.CHECK_JUPITER_BASE_UPDATE;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 */5 * * * *");
    }
}
