package ru.yandex.webmaster3.worker.searchquery;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Iterables;
import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.proto.converter.QueryGroupConverter;
import ru.yandex.webmaster3.core.searchquery.*;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.AllHostsCacheService;
import ru.yandex.webmaster3.storage.searchquery.dao.QueriesGroupsRelationYDao;
import ru.yandex.webmaster3.storage.searchquery.dao.QueriesYDao;
import ru.yandex.webmaster3.storage.searchquery.dao.QueryGroupYDao;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.io.IOException;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.stream.StreamSupport;

import static com.fasterxml.jackson.core.JsonParser.Feature.ALLOW_SINGLE_QUOTES;
import static ru.yandex.webmaster3.storage.util.ydb.AbstractYDao.YDB_SELECT_ROWS_LIMIT;

/**
 * @author avhaliullin
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UploadQueryGroupsTask extends PeriodicTask<UploadQueryGroupsTask.State> {
    private static final Logger log = LoggerFactory.getLogger(UploadQueryGroupsTask.class);
    public static final String F_KEY = "key";
    public static final String F_SUBKEY = "subkey";
    public static final String F_VALUE = "value";
    public static final String F_GROUP_NAME = "group_name";

    private static final ObjectMapper OM = new ObjectMapper().configure(ALLOW_SINGLE_QUOTES, true);
    private static final String TABLE_SCHEMA = "[" +
            "{'name': 'key', 'type': 'string'}, " +
            "{'name': 'subkey', 'type': 'string'}, " +
            "{'name': 'value', 'type': 'string'}, " +
            "{'name': 'group_name', 'type': 'string'}]";

    private static final int TOTAL_THREADS = 16;

    private final AllHostsCacheService allHostsCacheService;
    private final QueriesYDao queriesYDao;
    private final QueriesGroupsRelationYDao queriesGroupsRelationYDao;
    private final QueryGroupYDao queryGroupYDao;
    private final YtService ytService;

    @Value("${webmaster3.worker.uploadQueryGroupsTask.groupExport.arnold}")
    private final YtPath arnoldGroupExportPath;
    @Value("${webmaster3.worker.uploadQueryGroupsTask.groupExport.hahn}")
    private final YtPath hahnGroupExportPath;


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_QUERY_GROUPS;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new State());
        getState().phase = Phase.LOAD_GROUPS;


        YtTableData ytTableData = null;
        try {
            ExecutorService executorService = Executors.newFixedThreadPool(TOTAL_THREADS);

            ytTableData = ytService.prepareTableData(arnoldGroupExportPath.getName(), tw -> {
                allHostsCacheService.processKeyBatch(hostIdBatch -> {
                        getState().hostsCount += hostIdBatch.size();

                        Map<QueryGroupId, GroupAndQueries> allGroups = new HashMap<>();
                        Map<QueryId, Query> allQueries = new HashMap<>();

                        List<QueryGroup> groupBatch = getQueriesGroups(executorService, hostIdBatch);
                        for (QueryGroup g : groupBatch) {
                            allGroups.computeIfAbsent(g.getQueryGroupId(), k -> new GroupAndQueries(g));
                        }

                        for (WebmasterHostId hostId : hostIdBatch) {
                            QueryGroupId allQueriesGroupId = new QueryGroupId(hostId, SpecialGroup.ALL_QUERIES);
                            if (!allGroups.containsKey(allQueriesGroupId)) {
                                QueryGroup allQueriesGroup =
                                        new QueryGroup(allQueriesGroupId, null, false, SpecialGroup.ALL_QUERIES);
                                allGroups.put(allQueriesGroupId, new GroupAndQueries(allQueriesGroup));
                                getState().groupsCount++;
                            }
                            QueryGroupId favoriteQueriesGroupId = new QueryGroupId(hostId, SpecialGroup.SELECTED_QUERIES);
                            if (!allGroups.containsKey(favoriteQueriesGroupId)) {
                                QueryGroup favoriteGroup =
                                        new QueryGroup(favoriteQueriesGroupId, null, true, SpecialGroup.SELECTED_QUERIES);
                                allGroups.put(favoriteQueriesGroupId, new GroupAndQueries(favoriteGroup));
                                getState().groupsCount++;
                            }
                        }

                        getState().groupsCount += groupBatch.size();

                        List<Pair<QueryGroupId, QueryId>> relationsBatch = getQueriesRelations(executorService, hostIdBatch);
                        for (Pair<QueryGroupId, QueryId> p : relationsBatch) {
                            QueryGroupId groupId = p.getKey();
                            GroupAndQueries groupAndQueries = allGroups.get(groupId);
                            if (groupAndQueries != null) {
                                groupAndQueries.relatedQueries.add(p.getRight());
                            }
                        }

                        getState().relatedQueriesCount += relationsBatch.size();
                        List<Query> queries = getQueries(executorService, hostIdBatch);
                        for (Query q : queries) {
                            allQueries.put(q.getQueryId(), q);
                        }
                        getState().queriesCount += queries.size();
                        log.info("Groups loaded: hosts={} groups={} relations={} queries={}",
                                getState().hostsCount,
                                getState().groupsCount,
                                getState().relatedQueriesCount,
                                getState().queriesCount);


                        for (GroupAndQueries groupAndQueries : allGroups.values()) {
                            try {
                                QueryGroup group = groupAndQueries.queryGroup;
                                List<String> q = new ArrayList<>();
                                for (QueryId queryId : groupAndQueries.relatedQueries) {
                                    Query query = allQueries.get(queryId);
                                    if (query != null) {
                                        q.add(query.getQuery());
                                    }
                                }

                                WebmasterHostId hostId = group.getQueryGroupId().getHostId();
                                tw.column(F_KEY, IdUtils.hostIdToUrl(hostId));
                                tw.column(F_SUBKEY, group.getGroupId().toString());
                                tw.column(F_VALUE, QueryGroupConverter.queryGroup2Proto(group, q).toByteArray());
                                tw.column(F_GROUP_NAME, group.getSpecialGroup() != null ? group.getSpecialGroup().name() :
                                        group.getName());

                                tw.rowEnd();
                            } catch (YtException e) {
                                throw new RuntimeException("Unable to save groups", e);
                            }
                        }
                }, TOTAL_THREADS * YDB_SELECT_ROWS_LIMIT);
            });

            executorService.shutdown();

            final YtTableData td = ytTableData;
            getState().phase = Phase.UPLOAD_TO_BANACH;
            log.info("Upload to arnold");
            ytService.inTransaction(arnoldGroupExportPath)
                    .withLock(arnoldGroupExportPath, YtLockMode.EXCLUSIVE)
                    .withTimeout(4, TimeUnit.HOURS)
                    .execute(cypressService -> uploadToYt(cypressService, arnoldGroupExportPath, td));

            getState().phase = Phase.UPLOAD_TO_HAHN;
            log.info("Upload to hahn");
            ytService.inTransaction(hahnGroupExportPath)
                    .withLock(hahnGroupExportPath, YtLockMode.EXCLUSIVE)
                    .withTimeout(4, TimeUnit.HOURS)
                    .execute(cypressService -> uploadToYt(cypressService, hahnGroupExportPath, td));

            getState().phase = Phase.FINISH;
        } finally {
            if (ytTableData != null) {
                ytTableData.delete();
            }
        }

        return new Result(TaskResult.SUCCESS);
    }

    @NotNull
    private List<Pair<QueryGroupId, QueryId>> getQueriesRelations(ExecutorService executorService, List<WebmasterHostId> hostIdBatch) {
        var callables = StreamSupport.stream(Iterables.partition(hostIdBatch, YDB_SELECT_ROWS_LIMIT).spliterator(), false)
                .<Callable<List<Pair<QueryGroupId, QueryId>>>>map(hosts -> {
                    var cpy = new ArrayList<>(hosts);
                    return () -> queriesGroupsRelationYDao.getRelations(cpy);
                })
                .toList();

        List<Pair<QueryGroupId, QueryId>> res = new ArrayList<>();
        try {
            for (var f : executorService.invokeAll(callables)) {
                res.addAll(f.get());
            }
        } catch (Exception e) {
            throw new WebmasterYdbException(e);
        }

        return res;
    }

    @NotNull
    private List<QueryGroup> getQueriesGroups(ExecutorService executorService, List<WebmasterHostId> hostIdBatch) {
        var callablesQueryGroup = StreamSupport.stream(Iterables.partition(hostIdBatch, YDB_SELECT_ROWS_LIMIT).spliterator(), false)
                .<Callable<List<QueryGroup>>>map(hosts -> {
                    var cpy = new ArrayList<>(hosts);
                    return () -> queryGroupYDao.getGroups(cpy);
                })
                .toList();

        List<QueryGroup> res = new ArrayList<>();
        try {
            for (var f : executorService.invokeAll(callablesQueryGroup)) {
                res.addAll(f.get());
            }
        } catch (Exception e) {
            throw new WebmasterYdbException(e);
        }

        return res;
    }

    @NotNull
    private List<Query> getQueries(ExecutorService executorService, List<WebmasterHostId> hostIdBatch) {
        var callables = StreamSupport.stream(Iterables.partition(hostIdBatch, YDB_SELECT_ROWS_LIMIT).spliterator(), false)
                .<Callable<List<Query>>>map(hosts -> {
                    var cpy = new ArrayList<>(hosts);
                    return () -> queriesYDao.getQueries(cpy);
                })
                .toList();

        List<Query> res = new ArrayList<>();
        try {
            for (var f : executorService.invokeAll(callables)) {
                res.addAll(f.get());
            }
        } catch (Exception e) {
            throw new WebmasterYdbException(e);
        }

        return res;
    }


    private boolean uploadToYt(YtCypressService cypressService, YtPath path,
                               YtTableData tableData)
            throws YtException {
        YtPath tmpPath = YtPath.path(path.getParent(), path.getName() + ".tmp");
        if (cypressService.exists(tmpPath)) {
            cypressService.remove(tmpPath);
        }
        YtNodeAttributes attributes = new YtNodeAttributes().setCompressionCodec("none");
        try {
            attributes.getAttributes().put("schema", OM.readTree(TABLE_SCHEMA));
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        cypressService.create(tmpPath, YtNode.NodeType.TABLE, false, attributes);
        cypressService.writeTable(tmpPath, tableData);
        cypressService.remove(path);
        cypressService.move(tmpPath, path, false);
        return true;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 30 */6 * * *");
//        return TaskSchedule.never();
    }

    private static class GroupAndQueries {
        private final QueryGroup queryGroup;
        private final List<QueryId> relatedQueries = new ArrayList<>();

        public GroupAndQueries(QueryGroup queryGroup) {
            this.queryGroup = queryGroup;
        }
    }

    public static enum Phase {
        LOAD_GROUPS,
        UPLOAD_TO_BANACH,
        UPLOAD_TO_HAHN,
        FINISH,
    }

    public static class State implements PeriodicTaskState {
        private Phase phase;
        private long hostsCount;
        private long groupsCount;
        private long relatedQueriesCount;
        private long queriesCount;

        public Phase getPhase() {
            return phase;
        }

        public long getHostsCount() {
            return hostsCount;
        }

        public long getGroupsCount() {
            return groupsCount;
        }

        public long getRelatedQueriesCount() {
            return relatedQueriesCount;
        }

        public long getQueriesCount() {
            return queriesCount;
        }
    }

}
