package ru.yandex.webmaster3.worker.searchquery.rivals;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.collect.Range;
import lombok.Setter;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.dao.ClickhouseTablesRepository;
import ru.yandex.webmaster3.storage.searchquery.DeviceType;
import ru.yandex.webmaster3.storage.searchquery.rivals.RivalsStats2;
import ru.yandex.webmaster3.storage.searchquery.rivals.RivalsStats2CHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;
import ru.yandex.webmaster3.worker.ytimport.AbstractYtClickhouseDataLoadTask;

import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState.DONE;

/**
 * Created by Oleg Bazdyrev on 08/11/2017.
 */
public class ImportRivalsStats2Task extends AbstractYtClickhouseDataLoadTask {

    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new ParameterNamesModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private static final int MAX_BATCH_SIZE = 10000;
    private static final DateTime NEVER_UPDATED = new DateTime(0L);

    @Setter
    private ClickhouseTablesRepository clickhouseTablesCDao;
    @Setter
    private RivalsStats2CHDao rivalsStats2CHDao;
    @Setter
    private PeriodicTaskType type;

    protected YtClickhouseDataLoad init(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // проверяем по modification_time таблицы
        return ytService.withoutTransactionQuery(cypressService -> {
            YtNode node = cypressService.getNode(tablePath);
            DateTime latestUpdateDate = Optional.ofNullable(latestImport.getData()).map(Long::parseLong)
                    .map(DateTime::new).orElse(NEVER_UPDATED);
            if (node.getUpdateTime().isAfter(latestUpdateDate)) {
                log.info("Found fresh table with modification_time = {}", node.getUpdateTime());
                LocalDate date = node.getUpdateTime().toLocalDate();
                return latestImport.withSourceTable(tablePath, date, date)
                        .withData(String.valueOf(node.getUpdateTime().getMillis()));
            } else {
                // ничего не нашли, сразу завершимся
                log.info("No fresh data to import");
                return latestImport.withState(DONE);
            }
        });
    }

    @Override
    protected YtClickhouseDataLoad prepare(YtClickhouseDataLoad imprt) throws Exception {
        // никакой подготовки реально нет
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad doImport(YtClickhouseDataLoad imprt) throws Exception {
        final String updateTimestamp = imprt.getData();
        // почистим возможный мусор
        rivalsStats2CHDao.dropTables(updateTimestamp);
        // просто импорт из таблички в yt
        ytService.inTransaction(imprt.getSourceTable()).execute(cypressService -> {
            AsyncTableReader<RawRivalsStatsRow> tableReader = new AsyncTableReader<>(cypressService,
                    imprt.getSourceTable(), Range.all(),
                    YtTableReadDriver.createYSONDriver(RawRivalsStatsRow.class, OM))
                    .splitInParts(10000L).withRetry(3);

            try (AsyncTableReader.TableIterator<RawRivalsStatsRow> iterator = tableReader.read()) {
                ClickhouseHost host = clickhouseServer.pickRandomAliveHost();
                rivalsStats2CHDao.createTable(updateTimestamp, host);
                List<RivalsStats2> stats = new ArrayList<>();
                while (iterator.hasNext()) {
                    RawRivalsStatsRow row = iterator.next();
                    // конвертим
                    DeviceType deviceType = row.isMobile ? (row.isPad ? DeviceType.PAD : DeviceType.MOBILE) : DeviceType.DESKTOP;
                    LocalDate rowDate = new DateTime(TimeUnit.SECONDS.toMillis(row.timestamp)).toLocalDate();
                    RivalsStats2 rivalsStats = new RivalsStats2(rowDate, row.thematics, deviceType, row.type,
                            row.ctr, row.shows, row.clicks, row.domains);
                    stats.add(rivalsStats);
                    if (stats.size() >= MAX_BATCH_SIZE) {
                        rivalsStats2CHDao.insert(stats, updateTimestamp, host);
                        stats.clear();
                    }
                }
                // flush all remains
                if (!stats.isEmpty()) {
                    rivalsStats2CHDao.insert(stats, updateTimestamp, host);
                }
            } catch (IOException e) {
                throw new WebmasterException("YT error",
                        new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            } catch (ClickhouseException e) {
                throw new WebmasterException("Clickhouse error",
                        new WebmasterErrorResponse.ClickhouseErrorResponse(getClass(), e.getQuery(), e), e);
            }
            return true;
        });
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        String dateString = imprt.getData();
        String tableName = RivalsStats2CHDao.TABLE.replicatedMergeTreeTableName(0, dateString);
        var command = clickhouseReplicationManager.nonShardedReplication(
                AbstractClickhouseDao.DB_WEBMASTER3_QUERIES, tableName,
                RivalsStats2CHDao.TABLE.createReplicatedMergeTreeSpec(0, dateString)
        );
        clickhouseReplicationManager.enqueueReplication(command);
        return imprt.withReplicationTaskIds(command.getReplicationTaskId()).withNextState();
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        clickhouseTablesCDao.update(RivalsStats2CHDao.TABLE.toClickhouseTableInfoWithoutShards(
                TableType.RIVALS_STATS2, imprt.getData()).withState(TableState.ON_LINE));
        return super.rename(imprt);
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.RIVALS_STATS2;
    }

    @Override
    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        RivalsStats2CHDao.TABLE.updateDistributedSymlink(clickhouseServer, imprt.getData());
        return imprt.withNextState();
    }

    @Override
    public PeriodicTaskType getType() {
        return type;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 5 * * * *");
    }

    public static class RawRivalsStatsRow {

        private final String thematics;
        private final String type;
        private final long timestamp;
        private final boolean isPad;
        private final boolean isMobile;
        private final long clicks;
        private final long shows;
        private final double ctr;
        private final long domains;

        @JsonCreator
        public RawRivalsStatsRow(
                @JsonProperty("Thematics") String thematics,
                @JsonProperty("Type") String type,
                @JsonProperty("Timestamp") long timestamp,
                @JsonProperty("IsPad") boolean isPad,
                @JsonProperty("IsMobile") boolean isMobile,
                @JsonProperty("Clicks") long clicks,
                @JsonProperty("Shows") long shows,
                @JsonProperty("CTR") double ctr,
                @JsonProperty("Domains") long domains) {
            this.thematics = thematics;
            this.type = type;
            this.timestamp = timestamp;
            this.isPad = isPad;
            this.isMobile = isMobile;
            this.clicks = clicks;
            this.shows = shows;
            this.ctr = ctr;
            this.domains = domains;
        }
    }
}
