package ru.yandex.webmaster3.worker.searchurl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.Setter;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.searchurl.offline.SearchBaseImportTablesService;
import ru.yandex.webmaster3.storage.searchurl.offline.dao.SearchBaseImportTablesRepository;
import ru.yandex.webmaster3.storage.searchurl.offline.dao.SearchBaseImportedTable;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportInfo;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportStageEnum;
import ru.yandex.webmaster3.storage.searchurl.offline.data.SearchBaseImportTaskType;
import ru.yandex.webmaster3.storage.searchurl.samples.dao.CommonSearchUrlSamplesTableUtil;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseException;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
public class CleanupSearchBaseImportsTask extends PeriodicTask<PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(CleanupSearchBaseImportsTask.class);

    @Setter
    private SearchBaseImportTablesRepository searchBaseImportTablesCDao;
    @Setter
    private SearchBaseImportTablesService searchBaseImportTablesService;
    @Setter
    private ClickhouseServer clickhouseServer;
    @Setter
    private ImportSearchUrlSamplesTask importSearchUrlSamplesTask;
    @Setter
    private ImportExcludedUrlSamplesTask importExcludedUrlSamplesTask;
    @Setter
    private ImportSearchUrlEventSamplesTask importSearchUrlEventSamplesTask;
    @Setter
    private PeriodicTaskType type;

    @Override
    public Result run(UUID runId) throws Exception {
        for (SearchBaseImportTaskType taskType : SearchBaseImportTaskType.values()) {
            SearchBaseImportedTable currentTable = searchBaseImportTablesService.getTable(taskType);
            if (currentTable != null) {
                List<SearchBaseImportInfo> importsOfType = searchBaseImportTablesCDao.listBases(taskType);
                for (SearchBaseImportInfo importInfo : importsOfType) {
                    if (importInfo.getStage() != SearchBaseImportStageEnum.DELETED) {
                        if (importInfo.getStage() == SearchBaseImportStageEnum.DELETING) {
                            log.info("Continue deleting data of type {} for base {}", importInfo.getTaskType(), importInfo.getSearchBaseDate());
                            doDelete(importInfo);
                        } else if (importInfo.getSearchBaseDate().isBefore(currentTable.getBaseUpdateInfo().getBaseCollectionDate())) {
                            doDelete(importInfo);
                        }
                    }
                }
            }
        }
        return new Result(TaskResult.SUCCESS);
    }

    private void doDelete(SearchBaseImportInfo importInfo) throws InterruptedException, ClickhouseException {
        assertDataInUse(importInfo);
        log.info("Deleting data of type {} for base {}", importInfo.getTaskType(), importInfo.getSearchBaseDate());
        searchBaseImportTablesCDao.insertStage(importInfo.withStage(SearchBaseImportStageEnum.DELETING));
        switch (importInfo.getTaskType()) {
            case SEARCH_URL_EVENT_SAMPLES:
                importSearchUrlEventSamplesTask.dropPreparedTables(importInfo);
                doDeleteEventSamples(importInfo);
                break;
            case SEARCH_URL_SAMPLES:
                importSearchUrlSamplesTask.dropPreparedTables(importInfo);
                doDeleteUrlSamples(importInfo);
                break;
            case EXCLUDED_URL_SAMPLES:
                importExcludedUrlSamplesTask.dropPreparedTables(importInfo);
                doDeleteExcludedUrlSamples(importInfo);
                break;
            case SEARCH_URL_EMAIL_SAMPLES:
                doDeleteEmailSamples(importInfo);
                break;

            default:
                throw new RuntimeException("Unknown import type " + importInfo.getTaskType());
        }
        log.info("Done deleting data of type {} for base {}", importInfo.getTaskType(), importInfo.getSearchBaseDate());
        searchBaseImportTablesCDao.insertStage(importInfo.withStage(SearchBaseImportStageEnum.DELETED));
    }

    private void doDeleteEmailSamples(SearchBaseImportInfo importInfo) throws InterruptedException, ClickhouseException {
        dropTables(importInfo.getDbName(), Collections.singletonList(importInfo.getReadTableName()));
    }

    private void doDeleteExcludedUrlSamples(SearchBaseImportInfo importInfo) throws InterruptedException, ClickhouseException {
        List<String> tablesToDelete = new ArrayList<>();
        for (int part = 0; part < importInfo.getPartsCount(); part++) {
            tablesToDelete.add(CommonSearchUrlSamplesTableUtil.partTableName(importInfo.getReadTableName(), part));
        }
        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.shardTableName(importInfo.getReadTableName()));
        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.distributedTableName(importInfo.getReadTableName()));
        dropTables(importInfo.getDbName(), tablesToDelete);
    }

    private void doDeleteUrlSamples(SearchBaseImportInfo importInfo) throws InterruptedException, ClickhouseException {
        List<String> tablesToDelete = new ArrayList<>();
        for (int part = 0; part < importInfo.getPartsCount(); part++) {
            tablesToDelete.add(CommonSearchUrlSamplesTableUtil.partTableName(importInfo.getReadTableName(), part));
        }

        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.shardTableName(importInfo.getReadTableName()));
        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.distributedTableName(importInfo.getReadTableName()));
        dropTables(importInfo.getDbName(), tablesToDelete);
    }

    private void doDeleteEventSamples(SearchBaseImportInfo importInfo) throws InterruptedException, ClickhouseException {
        List<String> tablesToDelete = new ArrayList<>();
        for (int part = 0; part < importInfo.getPartsCount(); part++) {
            tablesToDelete.add(CommonSearchUrlSamplesTableUtil.partTableName(importInfo.getReadTableName(), part));
        }

        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.shardTableName(importInfo.getReadTableName()));
        tablesToDelete.add(CommonSearchUrlSamplesTableUtil.distributedTableName(importInfo.getReadTableName()));
        dropTables(importInfo.getDbName(), tablesToDelete);
    }

    private void dropTables(String dbName, List<String> tables) throws InterruptedException, ClickhouseException {
        List<ClickhouseHost> upHostsFirst = clickhouseServer.getHosts()
                .stream()
                .sorted(Comparator.comparing(ClickhouseHost::isDown))
                .collect(Collectors.toList());
        for (ClickhouseHost host : upHostsFirst) {
            for (String table : tables) {
                RetryUtils.execute(RetryUtils.linearBackoff(5, Duration.standardMinutes(5)), () -> {
                    ClickhouseQueryContext.Builder chContext = ClickhouseQueryContext.useDefaults().setHost(host);
                    clickhouseServer.execute(chContext, ClickhouseServer.QueryType.INSERT,
                            String.format("DROP TABLE IF EXISTS %1$s.%2$s", dbName, table), Optional.empty(),
                            Optional.empty());
                });
            }
        }
    }

    private void assertDataInUse(SearchBaseImportInfo importInfo) {
        // Просто на всякий случай - убеждаемся, что мы не удалим то, что сейчас используется на проде
        SearchBaseImportedTable currentTable = searchBaseImportTablesService.getTable(importInfo.getTaskType());
        if (currentTable.getBaseUpdateInfo().getBaseCollectionDate().equals(importInfo.getSearchBaseDate())) {
            throw new RuntimeException("Alarm! Tried to delete currently used data of type " + importInfo.getTaskType() + " for base " + importInfo.getSearchBaseDate());
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return type;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }
}
