package ru.yandex.webmaster3.worker.sitetree;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 14.06.17.
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UploadWebmasterVerifiedHostsTask extends PeriodicTask<UploadWebmasterVerifiedHostsTask.TaskState> {
    private static final String[] VERIFIED_HOSTS_COLUMN_NAME = {"user_id", "host_id", "host_url", "verification_type",
            "verification_uin", "verification_date", "verification_until_date", "visible", "export_date"};

    private final UserHostsService userHostsService;
    private final UploadWebmasterHostsCommonService uploadWebmasterHostsCommonService;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.hahn.export.archive.verifiedHosts.path}")
    private final YtPath hahnVerifiedHostsExportPath;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.arnold.export.archive.verifiedHosts.path}")
    private final YtPath arnoldVerifiedHostsExportPath;
    //ссылки на последние таблицы
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.hahn.export.archive.verifiedHosts.latest.link.path}")
    private final YtPath hahnVerifiedHostsLatestLinkPath;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.arnold.export.archive.verifiedHosts.latest.link.path}")
    private final YtPath arnoldVerifiedHostsLatestLinkPath;
    @Value("${webmaster3.worker.uploadWebmasterHostsTask.verifiedHosts.tmp.file}")
    private final File tmpVerifiedHostsFile;

    private boolean isDataDownloaded = false;
    private TaskState state;


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_WEBMASTER_VERIFIED_HOSTS;
    }

    @Override
    public PeriodicTask.Result run(UUID runId) throws Exception {
        state = new TaskState();
        setState(state);

        runUploadVerifiedHosts();

        return new PeriodicTask.Result(TaskResult.SUCCESS);
    }

    private void runUploadVerifiedHosts() throws Exception {
        isDataDownloaded = false;

        if (tmpVerifiedHostsFile.exists()) {
            tmpVerifiedHostsFile.delete();
        }

        List<YtException> exceptions = new ArrayList<>();

        boolean needUploadToHahn = true;
        log.info("Check if the data already uploaded into hahn today...");
        try {
            if (uploadWebmasterHostsCommonService.existTodayTable(hahnVerifiedHostsExportPath)) {
                log.info("Data has already been uploaded into hahn today");
                needUploadToHahn = false;
            }
        } catch (YtException e) {
            log.info("Unable to check if the data already uploaded into hahn today", e);
            exceptions.add(e);
            needUploadToHahn = false;
        }

        if (needUploadToHahn) {
            log.info("Upload to hahn required");
            // если не загрузили раннее из кассандры
            if (!isDataDownloaded) {
                downloadDataFromCassandra();
            }
            final String tempName = "webmaster-verified-hosts." + UploadWebmasterHostsCommonService.DATE_FORMAT_IN_NODE_NAME.print(DateTime.now());
            try (BufferedReader br = new BufferedReader(new FileReader(tmpVerifiedHostsFile))) {
                uploadWebmasterHostsCommonService.executeYtService(hahnVerifiedHostsExportPath, hahnVerifiedHostsLatestLinkPath, tempName, br, VERIFIED_HOSTS_COLUMN_NAME,
                        () -> state.verifiedHostsSentToHahn++);
            } catch (YtException e) {
                log.warn("Unable upload verified hosts to hahn", e);
                exceptions.add(new YtException("Unable upload verified hosts to hahn", e));
            } catch (IOException e) {
                throw new IOException("Unable to read from temp file", e);
            }
            log.info("Verified hosts uploaded into hahn {}", state.verifiedHostsSentToHahn);
        }

        boolean needUploadToArnold = true;
        log.info("Check if the data already uploaded into arnold today...");
        try {
            if (uploadWebmasterHostsCommonService.existTodayTable(arnoldVerifiedHostsExportPath)) {
                log.info("Data has already been uploaded into arnold today");
                needUploadToArnold = false;
            }
        } catch (YtException e) {
            log.info("Unable to check if the data already uploaded into arnold today", e);
            exceptions.add(e);
            needUploadToArnold = false;
        }

        if (needUploadToArnold) {
            log.info("Upload to arnold required");
            // если не загрузили раннее из кассандры
            if (!isDataDownloaded) {
                downloadDataFromCassandra();
            }
            final String tempName = "webmaster-verified-hosts." + UploadWebmasterHostsCommonService.DATE_FORMAT_IN_NODE_NAME.print(DateTime.now());
            try (BufferedReader br = new BufferedReader(new FileReader(tmpVerifiedHostsFile))) {
                uploadWebmasterHostsCommonService.executeYtService(arnoldVerifiedHostsExportPath, arnoldVerifiedHostsLatestLinkPath, tempName, br, VERIFIED_HOSTS_COLUMN_NAME,
                        () -> state.verifiedHostsSentToArnold++);
            } catch (YtException e) {
                log.warn("Unable upload verified hosts to arnold", e);
                exceptions.add(new YtException("Unable upload verified hosts to arnold", e));
            } catch (IOException e) {
                throw new IOException("Unable to read from temp file", e);
            }
            log.info("Verified hosts uploaded into arnold {}", state.verifiedHostsSentToArnold);
        }

        if (!exceptions.isEmpty()) {
            YtMultiException multiException = new YtMultiException();
            exceptions.forEach(multiException::add);
            throw multiException;
        }
    }

    private void downloadDataFromCassandra() throws IOException {
        log.info("Download verified hosts from cassandra...");
        String exportDate = LocalDate.now().toString();
        try (PrintWriter pw = new PrintWriter(tmpVerifiedHostsFile)) {
            userHostsService.forEach(
                    pair -> {
                        long userId = pair.getLeft();
                        UserVerifiedHost userVerifiedHost = pair.getRight();
                        pw.println(userId);
                        pw.println(userVerifiedHost.getWebmasterHostId());
                        pw.println(IdUtils.hostIdToUrl(userVerifiedHost.getWebmasterHostId()));
                        pw.println(userVerifiedHost.getVerificationType());
                        pw.println(userVerifiedHost.getVerificationUin());
                        pw.println(userVerifiedHost.getVerificationDate());
                        pw.println(userVerifiedHost.getVerifiedUntilDate());
                        pw.println(userVerifiedHost.getVerificationType().isDisplayable());
                        pw.println(exportDate);
                        state.verifiedHostsLoaded++;
                    }
            );
        } catch (WebmasterYdbException e) {
            throw new WebmasterException("Failed to get verified hosts",
                    new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
        } catch (IOException e) {
            throw new IOException("Unable to write to temp file", e);
        }

        isDataDownloaded = true;
        log.info("Verified hosts downloaded from cassandra {}", state.verifiedHostsLoaded);
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    public static class TaskState implements PeriodicTaskState {
        private long verifiedHostsLoaded = 0;
        private long verifiedHostsSentToHahn = 0;
        private long verifiedHostsSentToArnold = 0;

        public long getVerifiedHostsLoaded() {
            return verifiedHostsLoaded;
        }

        public long getVerifiedHostsSentToHahn() {
            return verifiedHostsSentToHahn;
        }

        public long getVerifiedHostsSentToArnold() {
            return verifiedHostsSentToArnold;
        }
    }

    public static class YtMultiException extends Exception {
        private List<Throwable> nested;

        public YtMultiException() {
            super("YtException on one or more YT clusters");
        }

        public void add(Throwable e) {
            if(e == null) {
                throw new IllegalArgumentException();
            } else {
                if(this.nested == null) {
                    this.initCause(e);
                    this.nested = new ArrayList<>();
                } else {
                    this.addSuppressed(e);
                }
            }
        }

        public int size() {
            return this.nested == null?0:this.nested.size();
        }

        public String toString() {
            StringBuilder str = new StringBuilder();
            str.append(getClass().getSimpleName());
            if(this.nested != null && this.nested.size() > 0) {
                str.append(this.nested);
            } else {
                str.append("[]");
            }

            return str.toString();
        }
    }
}
