package ru.yandex.webmaster3.worker.spamban;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.apache.commons.lang3.mutable.MutableLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.user.UserVerifiedHost;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.spam.ISpamHostFilter;
import ru.yandex.webmaster3.storage.spam.SpamHostsYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * @author avhaliullin
 */
@Component("updateSpamHostsTask")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class UpdateSpamHostsTask extends PeriodicTask<PeriodicTaskState> {
    private static final Logger log = LoggerFactory.getLogger(UpdateSpamHostsTask.class);

    private final ISpamHostFilter fastSpamHostFilter;
    private final SpamHostsYDao spamHostsYDao;
    private final UserHostsService userHostsService;

    @Override
    public Result run(UUID runId) throws Exception {
        MutableLong userId = new MutableLong(-1L);
        List<WebmasterHostId> userHosts = new ArrayList<>();
        MutableBoolean isSpam = new MutableBoolean(false);
        userHostsService.forEach(userHostInfo -> {
            long curUser = userHostInfo.getLeft();
            UserVerifiedHost userVerifiedHost = userHostInfo.getRight();
            if (!userVerifiedHost.getVerificationType().isDisplayable()) {
                //Не нужно провязываться через читерские подтверждения - так мы вообще всех нафиг побаним
                return;
            }
            WebmasterHostId curHost = userVerifiedHost.getWebmasterHostId();
            if (!userId.getValue().equals(curUser)) {
                if (isSpam.getValue()) {
                    insertSpamHosts(userHosts, userId.getValue());
                    isSpam.setFalse();
                }
                userHosts.clear();
                userId.setValue(curUser);
            }
            userHosts.add(curHost);
            if (!isSpam.getValue() && fastSpamHostFilter.checkHost(curHost)) {
                isSpam.setTrue();
            }
        });
        if (isSpam.getValue()) {
            insertSpamHosts(userHosts, userId.getValue());
        }
        return new Result(TaskResult.SUCCESS);
    }

    private void insertSpamHosts(List<WebmasterHostId> hosts, long userId) {
        for (WebmasterHostId hostId : hosts) {
            log.info("Marking host {} owner by user {} as spam", hostId, userId);
            try {
                spamHostsYDao.insertHost(hostId);
            } catch (WebmasterYdbException e) {
                log.error("Failed to save spam host", e);
            }
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPDATE_SPAM_HOSTS;
    }

    @Override
    public TaskSchedule getSchedule() {
        //return TaskSchedule.startByCron("0 13 3 * * *");
        return TaskSchedule.never();
    }
}
