package ru.yandex.webmaster3.worker.state;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.PreDestroy;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.MoreObjects;
import com.google.common.base.StandardSystemProperty;
import lombok.RequiredArgsConstructor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.util.yt.lock.CypressProvider;

/**
 * @author tsyplyaev
 */
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class WorkerStateService {
    private static final Logger log = LoggerFactory.getLogger(WorkerStateService.class);

    private static final String WORKERS_DIR = "/workers";

    private final CypressProvider cypressProvider;
    @Value("${application.hostname}")
    private String hostname;
    @Value("${application.http.port}")
    private int port;
    private String username;

    private String nodeId;

    private static final ObjectMapper OM = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    public void init() {
        username = MoreObjects.firstNonNull(StandardSystemProperty.USER_NAME.value(), "UNKNOWN");
        nodeId = WORKERS_DIR + "/" + hostname + "_" + port;
        createCypressNode();
    }

    private void createCypressNode() {
        try {
            String data = JsonMapping.writeValueAsString(new WorkerInfo(hostname, port, username));
            cypressProvider.create().setMode(CypressProvider.CreateMode.EPHEMERAL).forPath(nodeId, data);
            log.info("Registered worker {}:{}", hostname, port);
        } catch (Exception e) {
            log.error("Could not register worker {}:{}", hostname, port, e);
        }
    }

    // ноды иногда не создаются или при перезапуске возникает конфликт, поэтому проверяем каждую минуту
    @Scheduled(fixedDelay=60_000,  initialDelay=100_000)
    private void checkCypressNode() {
        if (!cypressProvider.checkExists().forPath(nodeId))  {
            createCypressNode();
        }
    }

    @PreDestroy
    public void stop() {
        try {
            cypressProvider.delete().forPath(nodeId);
        } catch (Exception ex) {
            log.error("Could not unregister worker", ex);
            return;
        }
        log.info("Unregistered worker '{}:{}'", hostname, port);
    }


    public List<WorkerInfo> getWorkers() throws Exception {
        List<WorkerInfo> workerInfos = new ArrayList<>();
        List<String> nodes = cypressProvider.getChildren().forPath(WORKERS_DIR);
        if (nodes == null || nodes.isEmpty()) {
            log.warn("Zookeeper returned empty nodes list");
        } else {
            for (String node : nodes) {
                WorkerInfo workerInfo = OM.readValue(cypressProvider.getData().forPath(WORKERS_DIR + "/" + node), WorkerInfo.class);
                workerInfos.add(workerInfo);
            }
        }

        return workerInfos;
    }

    public static class WorkerInfo {
        final String hostname;
        final int port;
        final String username;

        public WorkerInfo(@JsonProperty("hostname") String hostname, @JsonProperty("port") int port, @JsonProperty("username") String username) {
            this.hostname = hostname;
            this.port = port;
            this.username = username;
        }

        public String getHostname() {
            return hostname;
        }

        public int getPort() {
            return port;
        }

        public String getUsername() {
            return username;
        }
    }

}
