package ru.yandex.webmaster3.worker.turbo;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import lombok.Setter;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.storage.clickhouse.ClickhouseTableInfo;
import ru.yandex.webmaster3.storage.clickhouse.TableSource;
import ru.yandex.webmaster3.storage.clickhouse.TableState;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.dao.ClickhouseTablesRepository;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationCommand;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationPriority;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.storage.ytimport.ImportPriority;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportCommand;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseTableRelation;
import ru.yandex.webmaster3.worker.ytimport.AbstractYtClickhouseDataLoadTask;

public abstract class AbstractYqlPrepareImportTask extends AbstractYtClickhouseDataLoadTask {

    private static final int PARTS_COUNT = 1;

    protected static final String INTERMEDIATE_TABLE = "@intermediate";
    protected static final DateTimeFormatter IN_YQL_QUERY_DATE_FORMATTER = DateTimeFormat.forPattern("yyyy-MM-dd");

    @Autowired
    protected YqlService yqlService;
    @Setter
    protected ClickhouseTablesRepository clickhouseTablesCDao;
    @Setter
    protected YtPath workDir;
    @Setter
    private boolean enableReplication = false;

    protected abstract CHTable getTable();

    protected abstract TableType getTableType();

    protected abstract YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt);

    @Override
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad imprt) throws Exception {
        return initByUpdateDate(imprt);
    }

    protected int getShardsCount() {
        return clickhouseServer.getShardsCount();
    }

    @Override
    protected YtClickhouseDataLoad prepare(YtClickhouseDataLoad imprt) {

        String dateString = IN_TABLE_NAME_DATE_FORMATTER.print(imprt.getDateTo());

        YqlQueryBuilder yqlQueryBuilder = prepareIntermediateTable(imprt);

        List<YtPath> preparedTables = new ArrayList<>();
        for (int shard  = 0; shard < getShardsCount(); shard++) {
            YtPath table = YtPath.path(workDir, dateString + "_shard_" + shard);
            preparedTables.add(table);
            yqlQueryBuilder.appendText("INSERT INTO")
                    .appendTable(table).appendText(" WITH TRUNCATE ")
                    .appendText("SELECT data FROM " + INTERMEDIATE_TABLE + " WHERE ShardId=" + shard + ";\n");
        }

        yqlService.execute(yqlQueryBuilder.build());

        return imprt.withPreparedTables(preparedTables).withNextState();
    }

    @Override
    protected YtClickhouseDataLoad doImport(YtClickhouseDataLoad imprt) throws Exception {

        List<YtPath> tables = imprt.getPreparedTables();
        UUID taskId = UUIDs.timeBased();
        log.info("Import taskId=" + taskId);

        List<YtClickhouseTableRelation> tablesRels = new ArrayList<>();
        int idx = 0;
        for (int shard = 0; shard < getShardsCount(); shard++) {
            String createSpec = enableReplication ?
                    getTable().createReplicatedMergeTreeSpec(-1, getTableSuffix(imprt)) :
                    getTable().createMergeTreeSpec(-1, getTableSuffix(imprt));
            tablesRels.add(new YtClickhouseTableRelation(
                    tables.get(idx++),
                    shard,
                    getTable().replicatedMergeTreeTableName(-1, getTableSuffix(imprt)),
                    createSpec
            ));
        }

        YtClickhouseImportCommand command = new YtClickhouseImportCommand(
                taskId,
                tablesRels,
                getTable().getDatabase(),
                getTable().importSpec(),
                ImportPriority.ONLINE
        );
        if (enableReplication) {
            ytClickhouseImportManager.startImport(command);
        } else {
            ytClickhouseImportManager.startImportOnAllShards(command);
        }
        return imprt.withImportTaskIds(taskId).withNextState();
    }

    protected String getTableSuffix(YtClickhouseDataLoad imprt) {
        return imprt.getData();
    }

    @Override
    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        if (enableReplication) {
            // TODO костыльный случай, когда таблица не шардирована
            if (getShardsCount() == 1 && !getTable().isSharded()) {
                return replicateToAllHosts(imprt);
            }

            UUID taskId = UUIDs.timeBased();
            log.info("Replication id {}", taskId);
            List<ClickhouseReplicationCommand.TableInfo> tables = new ArrayList<>();
            int shardsCount = imprt.getPreparedTables().size();
            for (int shard = 0; shard < shardsCount; shard++) {
                tables.add(new ClickhouseReplicationCommand.TableInfo(
                        getTable().replicatedMergeTreeTableName(-1, getTableSuffix(imprt)),
                        getTable().createReplicatedMergeTreeSpec(-1, getTableSuffix(imprt)),
                        shard
                ));
            }
            ClickhouseReplicationCommand command = new ClickhouseReplicationCommand(
                    taskId,
                    getTable().getDatabase(),
                    getReplicationPriority(),
                    tables
            );
            clickhouseReplicationManager.enqueueReplication(command);
            return imprt.withReplicationTaskIds(taskId).withNextState();
        }
        return imprt.withNextState();
    }

    protected YtClickhouseDataLoad replicateToAllHosts(YtClickhouseDataLoad imprt) throws Exception {
        var command = clickhouseReplicationManager.nonShardedReplication(
                getTable().getDatabase(),
                getTable().replicatedMergeTreeTableName(-1, getTableSuffix(imprt)),
                getTable().createReplicatedMergeTreeSpec(-1, getTableSuffix(imprt))
        );
        clickhouseReplicationManager.enqueueReplication(command);
        return imprt.withReplicationTaskIds(command.getReplicationTaskId()).withNextState();
    }

    @Override
    protected YtClickhouseDataLoad waitForReplication(YtClickhouseDataLoad imprt) throws Exception {
        if (enableReplication) {
            return super.waitForReplication(imprt);
        }
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        String chTableName = getTable().getDatabase() + "." + getTable().replicatedMergeTreeTableName(-1, getTableSuffix(imprt));
        clickhouseTablesCDao.update(
                new ClickhouseTableInfo(getTableType(), UUIDs.timeBased(),
                        TableState.ON_LINE, DateTime.now(), TableSource.YT_HAHN, null,
                        chTableName, chTableName, chTableName, imprt.getPreparedTables().size(), getTable().getParts())
        );
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        getTable().updateDistributedSymlink(clickhouseServer, getTableSuffix(imprt));
        return imprt.withNextState();
    }

    protected ClickhouseReplicationPriority getReplicationPriority() {
        return ClickhouseReplicationPriority.ONLINE;
    }
}
