package ru.yandex.webmaster3.worker.turbo;

import java.net.MalformedURLException;
import java.net.URL;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableMap;
import lombok.Setter;
import lombok.Value;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.checklist.data.TurboUrlErrorSample;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.sitestructure.RawSearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.solomon.metric.SolomonCounter;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricConfiguration;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistPageSamplesService;
import ru.yandex.webmaster3.storage.checklist.dao.ChecklistSamplesType;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.util.clickhouse2.AbstractClickhouseDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHPrimitiveType;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType.TURBO_SEARCHURLS_SAMPLES;

public class ImportTurboSearchUrlsSamplesTask extends AbstractYqlPrepareImportTask {
    private static final int BATCH_SIZE = 2000;
    private static final long DSAT_ALERT_THRESHOLD = 5L;
    private static final CHTable SEARCHURLS_SAMPLES_TABLE = CHTable.builder()
            .database(AbstractClickhouseDao.DB_WEBMASTER3_TURBO)
            .name("searchurls_samples_%s")
            .partitionBy("toYYYYMM(date)")
            .keyField("date", CHPrimitiveType.Date)
            .keyField("domain", CHPrimitiveType.String)
            .field("url", CHPrimitiveType.String)
            .field("feed", CHPrimitiveType.String)
            .field("source", CHPrimitiveType.String)
            .field("source_flags", CHPrimitiveType.UInt64)
            .field("is_searchable", CHPrimitiveType.UInt8)
            .build();
    private static final Map<TurboFeedType, TurboFeedType> SOURCE_TO_MONITORING_MAPPING = ImmutableMap.<TurboFeedType, TurboFeedType>builder()
            .put(TurboFeedType.AUTO, TurboFeedType.AUTO)
            .put(TurboFeedType.AUTO_DISABLED, TurboFeedType.AUTO)
            .put(TurboFeedType.API, TurboFeedType.API)
            .put(TurboFeedType.RSS, TurboFeedType.RSS)
            .put(TurboFeedType.YML, TurboFeedType.YML)
            .build();
    private int rowCount = 1024;
    @Setter
    private HandleCommonMetricsService handleCommonMetricsService;
    @Autowired
    private SiteProblemsService siteProblemsService;
    @Autowired
    private ChecklistPageSamplesService checklistPageSamplesService;
    @Setter
    private boolean enableAlerts = true;

    @Override
    protected CHTable getTable() {
        return SEARCHURLS_SAMPLES_TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.TURBO_SEARCHURLS_SAMPLES;
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {

        String dateString = IN_YQL_QUERY_DATE_FORMATTER.print(imprt.getDateTo());

        YqlQueryBuilder yqlQueryBuilder = YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM (")
                .appendText("SELECT")
                .appendText("(Digest::Fnv64(Domain) % " + getShardsCount() + ") as ShardId,")
                .appendText("((Digest::Fnv64(Url)) % " + rowCount + ") as RowId,")
                .appendText("('" + dateString + "'")
                .appendText("|| '\\t' || Domain")
                .appendText("|| '\\t' || String::EscapeC(Url)")
                .appendText("|| '\\t' || String::EscapeC(Feed)")
                .appendText("|| '\\t' || Source")
                .appendText("|| '\\t' || CAST(SourceFlags as String)")
                .appendText("|| '\\t' || CASE WHEN IsSearchable THEN '1' ELSE '0' END")
                .appendText("|| '\\n') as data")
                .appendText("FROM")
                .appendTable(tablePath)
                .appendText(") GROUP BY ShardId, RowId;")
                .appendText("COMMIT;");

        return yqlQueryBuilder;
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        if (enableAlerts) {
            monitorStatistics();
            collectTurboUrlsProblems();
        }
        return super.rename(imprt);
    }

    /**
     * WMC-8492 - собираем проблемы по урлам турбо
     *
     * @throws Exception
     */
    private void collectTurboUrlsProblems() throws Exception {
        DateTime updateStarted = DateTime.now();
        String query = "SELECT \n" +
                "    Domain, \n" +
                "    count_if(${CONDITION}) as Bad,\n" +
                "    count(*) as Total,\n" +
                "    Yson::SerializeJson(Yson::From(AGGREGATE_LIST_DISTINCT(\n" +
                "        if(${CONDITION},\n" +
                "            AsStruct(\n" +
                "                Url as url,\n" +
                "                Source as source,\n" +
                "                Feed as feed,\n" +
                "                UrlStatus as urlStatus\n" +
                "            ), \n" +
                "            null\n" +
                "        ),\n" +
                "        100\n" +
                "    ))) as Samples\n" +
                "FROM ${SOURCE_TABLE}\n" +
                "GROUP BY Domain\n" +
                "HAVING count_if(${CONDITION}) >= ${DSAT_ALERT_THRESHOLD};";

        query = StrSubstitutor.replace(query, Map.of("SOURCE_TABLE", tablePath.toYqlPath(), "DSAT_ALERT_THRESHOLD", DSAT_ALERT_THRESHOLD,
                "CONDITION", "(UrlStatus in (2, 11, 8, 14) and IsSearchable == false and Source in ('rss', 'api', 'yml'))"));
        Map<WebmasterHostId, ProblemSignal> problemSignalMap = new HashMap<>(BATCH_SIZE);
        Map<WebmasterHostId, List<String>> samplesMap = new HashMap<>(BATCH_SIZE);
        yqlService.query(query, resultSet -> resultSet, resultSet -> {
            String domain = resultSet.getString("Domain");
            long total = Long.parseLong(resultSet.getString("Total"));
            long bad = Long.parseLong(resultSet.getString("Bad"));
            List<TurboBadUrlSample> samples = JsonMapping.OM.readValue(resultSet.getBytes("Samples"), TurboBadUrlSample.TYPE_REFERENCE);
            ProblemSignal problemSignal = new ProblemSignal(new SiteProblemContent.TurboUrlErrors(bad, total), DateTime.now());
            WebmasterHostId hostId = IdUtils.urlToHostId(domain);
            problemSignalMap.put(hostId, problemSignal);
            samplesMap.put(hostId, samples.stream().map(TurboBadUrlSample::toSample).map(JsonMapping::writeValueAsStringAscii).collect(Collectors.toList()));
            if (problemSignalMap.size() >= BATCH_SIZE) {
                save(problemSignalMap, samplesMap);
            }
        });
        if (!problemSignalMap.isEmpty()) {
            save(problemSignalMap, samplesMap);
        }
        // остатки
        siteProblemsService.notifyCleanableProblemUpdateFinished(SiteProblemTypeEnum.TURBO_URL_ERRORS, updateStarted);
    }

    private void save(Map<WebmasterHostId, ProblemSignal> problemSignalMap, Map<WebmasterHostId, List<String>> samplesMap) {
        siteProblemsService.updateCleanableProblems(problemSignalMap, SiteProblemTypeEnum.TURBO_URL_ERRORS);
        checklistPageSamplesService.saveSamples(samplesMap, ChecklistSamplesType.TURBO_URL_ERRORS);
        problemSignalMap.clear();
        samplesMap.clear();
    }

    private void monitorStatistics() throws SQLException {
        // соберем статистику и зальем
        YqlQueryBuilder yqlQueryBuilder = YqlQueryBuilder.newBuilder()
                .cluster(tablePath)
                .appendText("SELECT Source, IsSearchable, UrlStatus, count(*) as Cnt FROM")
                .appendTable(tablePath)
                .appendText("GROUP BY Source, IsSearchable, UrlStatus;");

        SolomonMetricRegistry solomonMetricRegistry = new SolomonMetricRegistry();
        SolomonMetricConfiguration config = new SolomonMetricConfiguration();
        config.setEnable(true);
        config.setIndicator("turbo_urls_count");
        config.setGroupBy(Arrays.asList(
                "category,indicator", // total
                "category,indicator,source", // by type
                "category,indicator,is_searchable,url_status", // by jupiter status
                "category,indicator,source,is_searchable,url_status" // no grouping
        ));

        Map<TurboFeedType, Map<Boolean, Map<RawSearchUrlStatusEnum, SolomonCounter>>> counterMap = new EnumMap<>(TurboFeedType.class);
        SolomonKey baseKey = SolomonKey.create(SolomonSensor.LABEL_CATEGORY, "turbo");
        for (TurboFeedType source : Arrays.asList(TurboFeedType.RSS, TurboFeedType.YML, TurboFeedType.API, TurboFeedType.AUTO)) {
            Map<Boolean, Map<RawSearchUrlStatusEnum, SolomonCounter>> searchableMap = new HashMap<>();
            counterMap.put(source, searchableMap);
            SolomonKey sourceKey = baseKey.withLabel("source", source.getCode());
            for (Boolean isSearchable : Arrays.asList(false, true)) {
                Map<RawSearchUrlStatusEnum, SolomonCounter> statusMap = new EnumMap<>(RawSearchUrlStatusEnum.class);
                searchableMap.put(isSearchable, statusMap);
                SolomonKey searchableKey = sourceKey.withLabel("is_searchable", isSearchable.toString());
                for (RawSearchUrlStatusEnum status : RawSearchUrlStatusEnum.values()) {
                    statusMap.put(status, solomonMetricRegistry.createCounter(config, searchableKey.withLabel("url_status", status.toString())));
                }
            }
        }

        yqlService.query(yqlQueryBuilder.build(), resultSet -> resultSet, resultSet -> {
            String sources = resultSet.getString("Source");
            boolean isSearchable = resultSet.getBoolean("IsSearchable");
            RawSearchUrlStatusEnum urlStatus = RawSearchUrlStatusEnum.R.fromValueOrNull(resultSet.getInt("UrlStatus"));
            if (urlStatus == null) {
                return;
            }
            long count = resultSet.getLong("Cnt");
            for (String sourceString : new HashSet<>(Arrays.asList(sources.split(",")))) {
                TurboFeedType source = SOURCE_TO_MONITORING_MAPPING.get(TurboFeedType.byCode(sourceString));
                if (source == null) {
                    continue;
                }
                counterMap.get(source).get(isSearchable).get(urlStatus).add(count);
            }
        });
        // send data to solomon
        var snapshot = solomonMetricRegistry.getIndicatorsSnapshot();
        List<SolomonSensor> sensors = new ArrayList<>();
        long seconds = System.currentTimeMillis() / 1000L;
        snapshot.forEach((key, val) -> sensors.add(new SolomonSensor(key.getLabels(), seconds, val)));
        handleCommonMetricsService.handle(sensors, sensors.size() * 100);
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return TURBO_SEARCHURLS_SAMPLES;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    @Value
    private static final class TurboBadUrlSample {
        public static final TypeReference<List<TurboBadUrlSample>> TYPE_REFERENCE =
                new TypeReference<List<TurboBadUrlSample>>() {
                };
        String url;
        String feed;
        String source;
        int urlStatus;

        public TurboUrlErrorSample toSample() {
            String feed = null;
            try {
                new URL(this.feed);
                feed = this.feed;
            } catch (MalformedURLException e) {
            }
            return new TurboUrlErrorSample(url, feed, SearchUrlStatusUtil.raw2View(RawSearchUrlStatusEnum.R.fromValue(urlStatus), false));
        }
    }

}
