package ru.yandex.webmaster3.worker.turbo;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboDomainStatistics;
import ru.yandex.webmaster3.core.turbo.model.statistics.TurboHostStatisticsType;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.SettingsService;
import ru.yandex.webmaster3.storage.turbo.dao.statistics.TurboDomainStatisticsHistoryYDao;
import ru.yandex.webmaster3.storage.turbo.dao.statistics.TurboDomainStatisticsYDao;
import ru.yandex.webmaster3.storage.turbo.service.autoparser.TurboAutoparserInfoUtil;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 06/02/2018.
 */
@Component("importTurboStatsTask")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ImportTurboStatsTask extends PeriodicTask<PeriodicTaskState> {

    private static final Logger log = LoggerFactory.getLogger(ImportTurboStatsTask.class);
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModule(new ParameterNamesModule())
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private static final long BATCH_SIZE = 2500L;

    private static final Map<String, TurboHostStatisticsType> ACTION_TO_STATS_TYPE = ImmutableMap.of(
            "modify", TurboHostStatisticsType.PERSISTED,
            "delete", TurboHostStatisticsType.DELETED,
            "ban", TurboHostStatisticsType.BANNED
    );

    private final HostOwnerService hostOwnerService;
    private final SettingsService settingsService;
    private final TurboDomainStatisticsYDao turboDomainStatisticsYDao;
    private final TurboDomainStatisticsHistoryYDao turboDomainStatisticsHistoryYDao;
    private final YtService ytService;

    @Value("${webmaster3.worker.turbo.stats.sourceTable.path}")
    private YtPath sourceTablePath;

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Importing turbo stats from {}", sourceTablePath);
        DateTime now = DateTime.now();
        long startTime = now.getMillis();

        List<TurboDomainStatistics> statistics = new ArrayList<>();

        ytService.withoutTransaction(cypressService -> {
            AsyncTableReader<TurboStatsRow> reader = new AsyncTableReader<>(cypressService, sourceTablePath, Range.all(),
                    YtTableReadDriver.createYSONDriver(TurboStatsRow.class, OM)).withRetry(3).splitInParts(BATCH_SIZE);
            int rowNum = 0;

            try (AsyncTableReader.TableIterator<TurboStatsRow> iterator = reader.read()) {
                while (iterator.hasNext()) {
                    TurboStatsRow row = iterator.next();
                    TurboHostStatisticsType statsType = ACTION_TO_STATS_TYPE.get(row.getAction());
                    if (statsType == null) {
                        continue;
                    }
                    TurboFeedType source = TurboFeedType.byCode(row.getSource());
                    if (!Strings.isNullOrEmpty(row.getDomain())) {
                        TurboDomainStatistics stats = new TurboDomainStatistics(
                                row.getDomain(),
                                source,
                                row.getFeed(),
                                statsType,
                                row.getCount(),
                                now);
                        statistics.add(stats);
                    }
                    if (statistics.size() >= BATCH_SIZE) {
                        flushStats( statistics);
                        statistics.clear();
                    }
                    rowNum++;
                }
                flushStats(statistics);

            } catch (IOException e) {
                log.error("IO error at row " + rowNum, e);
                throw new WebmasterException("Yt IO error",
                        new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            }
            return true;
        });
        // save start time
        settingsService.update(CommonDataType.TURBO_HOST_STATS_LAST_UPDATE, Long.toString(startTime));

        return new Result(TaskResult.SUCCESS);
    }

    private void flushStats(List<TurboDomainStatistics> statistics) {
        List<TurboDomainStatistics> result = new ArrayList<>();
        Map<String, TurboDomainStatistics> autoparse = new HashMap<>();
        for (TurboDomainStatistics stat : statistics) {
            if (stat.getSource() != TurboFeedType.AUTO) {
                result.add(new TurboDomainStatistics(stat.getDomain(), stat.getSource(), stat.getFeed(), stat.getType(),
                        stat.getValue(), stat.getUpdateDate()));
            } else {
                autoparse.put(stat.getDomain(), stat);
            }
        }
        result.addAll(processAutoparseStats(autoparse));

        log.info("flushStats: {}", result.size());
        turboDomainStatisticsYDao.saveStatistics(result);
        turboDomainStatisticsHistoryYDao.saveStatistics(result);
    }

    private List<TurboDomainStatistics> processAutoparseStats(Map<String, TurboDomainStatistics> rows) {
        List<String> hosts = rows.keySet().stream()
                .sorted(Comparator.comparing(String::length).reversed())
                .collect(Collectors.toList());
        for (String host : hosts) {
            TurboDomainStatistics stats = rows.get(host);
            String owner = hostOwnerService.getHostOwner(host);
            List<String> parentsChain = TurboAutoparserInfoUtil.getParentsChain(host, owner);
            if (!parentsChain.isEmpty()) {
                //добавим прямому предку
                String parent = parentsChain.get(0);
                if (!rows.containsKey(parent)) {
                    rows.put(parent, new TurboDomainStatistics(parent, stats.getSource(), stats.getFeed(), stats.getType(),
                            stats.getValue(), stats.getUpdateDate()));
                } else {
                    TurboDomainStatistics parentStats = rows.get(parent);
                    long add = (stats.getType() == TurboHostStatisticsType.PERSISTED ? stats.getValue() : -stats.getValue());
                    rows.put(parent, parentStats.withValue(parentStats.getValue() + add));
                }
            }
        }
        return new ArrayList<>(rows.values());
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.IMPORT_TURBO_STATS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 30 1/6 * * *");
    }

    public static final class TurboStatsRow {
        private final String domain;
        private final String source;
        private final String feed;
        private final String action;
        private final long count;

        @JsonCreator
        public TurboStatsRow(@JsonProperty("Domain") String domain,
                             @JsonProperty("Source") String source,
                             @JsonProperty("Feed") String feed,
                             @JsonProperty("Action") String action,
                             @JsonProperty("Count") long count) {
            this.domain = domain;
            this.source = source;
            this.feed = feed;
            this.action = action;
            this.count = count;
        }

        public String getDomain() {
            return domain;
        }

        public String getSource() {
            return source;
        }

        public String getFeed() {
            try {
                return IdUtils.normalizeUrl(feed);
            } catch (IllegalArgumentException e) {
                return feed;
            }
        }

        public String getAction() {
            return action;
        }

        public long getCount() {
            return count;
        }
    }
}
