package ru.yandex.webmaster3.worker.turbo;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;
import lombok.Setter;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.clickhouse.TableType;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.turbo.dao.statistics.MdbTurboClicksDayStatisticsCHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.CHTable;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.yql.YqlQueryBuilder;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 17/03/2021.
 */
public class MdbImportTurboClickDayStatsTask extends AbstractYqlPrepareImportTask {

    private static final DateTimeFormatter TABLE_NAME_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");
    private static final int ROW_COUNT = 128;

    @Setter
    private ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;

    @Override
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad latestImport) throws Exception {
        return ytService.withoutTransactionQuery(cypressService -> {
            List<YtPath> tables = cypressService.list(tablePath);
            String lastProcessedTable = Objects.requireNonNullElse(latestImport.getData(), "");
            LocalDate expectedDate = TABLE_NAME_FORMAT.parseLocalDate(lastProcessedTable).plusDays(1);
            Optional<String> tableName = tables.stream().sorted().map(YtPath::getName).filter(n -> n.compareTo(lastProcessedTable) > 0).findFirst();
            if (tableName.isEmpty()) {
                return latestImport.withState(YtClickhouseDataLoadState.DONE);
            }
            LocalDate newDate = TABLE_NAME_FORMAT.parseLocalDate(tableName.get());
            Preconditions.checkState(newDate.equals(expectedDate), "Found gap in source data: expected data for date" + expectedDate + " but found " + newDate);
            return latestImport.withData(tableName.get()).withSourceTable(YtPath.path(tablePath, tableName.get()),
                    newDate,
                    newDate
            );
        });
    }

    @Override
    protected int getShardsCount() {
        return clickhouseServer.getShardsCount();
    }

    @Override
    protected YqlQueryBuilder prepareIntermediateTable(YtClickhouseDataLoad imprt) {
        int shardCount = clickhouseServer.getShardsCount();
        YqlQueryBuilder queryBuilder = new YqlQueryBuilder();
        queryBuilder
                .cluster(tablePath)
                .appendText("PRAGMA yt.MaxRowWeight = '128M';\n")
                .appendText("PRAGMA yt.DefaultMemoryLimit = '4G';\n\n")
                .appendText("INSERT INTO " + INTERMEDIATE_TABLE)
                .appendText("SELECT ShardId, RowId, Compress::Gzip(String::JoinFromList(AGGREGATE_LIST(data), ''), 6) as data FROM\n")
                .appendText("(\n")
                .appendText("  SELECT\n")
                .appendText("    (Digest::Fnv64(domain) % " + shardCount + ") as ShardId,\n")
                .appendText("    (Digest::CityHash(domain) % " + ROW_COUNT + ") as RowId,\n")
                .appendText("    (domain || '\\t' ||\n")
                .appendText("    '" + imprt.getDateFrom().toString() + "' || '\\t' ||\n")
                .appendText("    cast(total_clicks as String) || '\\t' ||\n")
                .appendText("    cast(turbo_clicks as String) || '\\t' ||\n")
                .appendText("    cast(autoparsed_clicks as String) || '\\t' ||\n")
                .appendText("    String::ReplaceAll(nvl(cast(Yson::SerializeJson(Yson::From(top_urls_without_turbo)) as String), '[]'), @@\\@@, @@\\\\@@) || '\\n'\n")
                .appendText("    ) as data\n")
                .appendText("  FROM").appendTable(imprt.getSourceTable()).appendText("\n")
                .appendText(")\n")
                .appendText("GROUP BY ShardId, RowId;\n")
                .appendText("COMMIT;\n\n");

        return queryBuilder;
    }

    @Override
    protected YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad waitForReplication(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        String database = getTable().getDatabase();
        String tempTableName = getTable().replicatedMergeTreeTableName(-1, imprt.getData());
        // удаляем из общей таблицы подливаемую дату и вливаем свежие данные
        for (ClickhouseHost host : clickhouseServer.getHosts()) {
            boolean hasTable = !clickhouseSystemTablesCHDao.getTables(host, database, Collections.singleton(tempTableName)).isEmpty();
            if (!hasTable) {
                continue;
            }
            // получим минимальный и максимальный ts в новой таблице
            ClickhouseQueryContext.Builder ctx = ClickhouseQueryContext.useDefaults().setHost(host).setTimeout(Duration.standardMinutes(10L));
            StrSubstitutor substitutor = new StrSubstitutor(Map.of(
                "DB", database,
                "FULL_TABLE", MdbTurboClicksDayStatisticsCHDao.FULL_TABLE_NAME,
                "TEMP_TABLE", tempTableName,
                "DATE", imprt.getDateFrom().toString(),
                "YEAR", String.valueOf(imprt.getDateFrom().getYear())
            ));
            for (String query : substitutor.replace(MERGE_TABLE_QUERY).split("\n\n")) {
                clickhouseServer.execute(ctx, query);
            }
        }
        return imprt.withNextState();
    }

    @Override
    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    private static final String MERGE_TABLE_QUERY = "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} DELETE WHERE date = '${DATE}';\n\n" +
            "" +
            "ALTER TABLE ${DB}.${FULL_TABLE} ATTACH PARTITION '${YEAR}' FROM ${DB}.${TEMP_TABLE}\n\n" +
            "" +
            "DROP TABLE ${DB}.${TEMP_TABLE};";

    @Override
    protected CHTable getTable() {
        return MdbTurboClicksDayStatisticsCHDao.TABLE;
    }

    @Override
    protected TableType getTableType() {
        return TableType.TURBO_CLICK_DAY_STATS;
    }

    @Override
    protected YtClickhouseDataLoadType getImportType() {
        return YtClickhouseDataLoadType.TURBO_CLICK_DAY_STATS;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.MDB_IMPORT_TURBO_CLICK_DAY_STATS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 18 * * * *");
    }

}
