package ru.yandex.webmaster3.worker.turbo.autoparser;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.turbo.model.autoparser.AutoparserToggleState;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.turbo.dao.autoparser.TurboAutoparsedYDao;
import ru.yandex.webmaster3.storage.turbo.service.autoparser.TurboAutoparserInfoService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.storage.util.yt.YtTransactionService;
import ru.yandex.webmaster3.storage.util.yt.YtUtils;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by ifilippov5 on 19.06.18.
 */
@Service
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ExportTurboAutoparseFlagsTask extends PeriodicTask<ExportTurboAutoparseFlagsTask.TaskState> {
    private static final Logger log = LoggerFactory.getLogger(ExportTurboAutoparseFlagsTask.class);

    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.linearBackoff(10, Duration.standardMinutes(2));

    private static final String TABLE_SCHEMA = "[" +
            "{'name': 'AutoparserByTitleEnabled', 'type': 'boolean'}, " +
            "{'name': 'Host', 'type': 'string'}]";

    @Value("${webmaster3.worker.turbo.autoparser.export.tablePath}")
    private final YtPath workTable;
    private final YtService ytService;
    private final TurboAutoparsedYDao turboAutoparsedYDao;
    private final TurboAutoparserInfoService turboAutoparserInfoService;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_TURBO_AUTOPARSE_FLAGS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 */30 * * * *");
    }

    @Override
    public PeriodicTask.Result run(UUID runId) throws Exception {
        state = new ExportTurboAutoparseFlagsTask.TaskState();
        setState(state);
        DateTime lastImportDate = turboAutoparserInfoService.getLastImportDate();
        if (lastImportDate == null) {
            log.info("Any import not found");
            return new PeriodicTask.Result(TaskResult.SUCCESS);
        }

        YtTableData table = null;
        try {
            table = ytService.prepareTableData(workTable.getName(), (tw) -> {
                turboAutoparsedYDao.forEach(info -> {
                    if (!turboAutoparserInfoService.isHostInfoActual(info, lastImportDate)) {
                        return;
                    }
                    if (!info.isImportedFromYt()) {
                        return;//отгружаем только тех, кто был в табличке от турбо
                    }
                    boolean autoparserEnabled = turboAutoparserInfoService.getAutoparseCheckBoxState(info.getHost()) == AutoparserToggleState.ON;
                    tw.column(YtRow.F_HOST, info.getHost());
                    tw.column(YtRow.F_AUTOPARSER_ENABLED, autoparserEnabled);
                    tw.rowEnd();
                });
            });

            YtTransactionService.TransactionProcess process = new YtUtils.TransactionWriterBuilder(workTable, table)
                    .withSchema(TABLE_SCHEMA)
                    .withRetry(RETRY_POLICY)
                    .build();

            YtUtils.TransactionExecutor writer = new YtUtils.TransactionExecutor(ytService, workTable);
            writer.execute(process);

        } finally {
            if (table != null) {
                table.delete();
            }
        }
        return new PeriodicTask.Result(TaskResult.SUCCESS);
    }

    public static class TaskState implements PeriodicTaskState {
    }

    private static class YtRow {
        static final String F_HOST = "Host";
        static final String F_AUTOPARSER_ENABLED = "AutoparserByTitleEnabled";
    }
}
