package ru.yandex.webmaster3.worker.turbo.commerce;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Supplier;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Range;
import lombok.AllArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.text.StrSubstitutor;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemContent.TurboInvalidCartUrl;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemState;
import ru.yandex.webmaster3.core.checklist.data.SiteProblemTypeEnum;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.checklist.data.ProblemSignal;
import ru.yandex.webmaster3.storage.checklist.service.SiteProblemsService;
import ru.yandex.webmaster3.storage.toloka.TolokaService;
import ru.yandex.webmaster3.storage.toloka.model.TolokaTask;
import ru.yandex.webmaster3.storage.toloka.model.TolokaTaskSuite;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.settings.TurboSettingsService;
import ru.yandex.webmaster3.storage.util.yt.AsyncTableReader;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtCypressService;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableReadDriver;
import ru.yandex.webmaster3.storage.util.yt.transfer.YtTransferManager;
import ru.yandex.webmaster3.storage.yql.YqlService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 22/04/2020.
 */
@Slf4j
@Component("turboCartUrlsModerationTask")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class TurboCartUrlsModerationTask extends PeriodicTask<PeriodicTaskState> {

    private static final int BATCH_SIZE = 2000;
    private static final String TELLURIUM_TASK_PREFIX = "turbo-cart-urls-moderation-";
    private static final long MAX_DOMAINS_PER_DAY = 3000L;
    private static final long SAMPLES_PER_DOMAIN = 2L;
    private static final long REVALIDATION_PERIOD_MILLIS = Duration.standardDays(15).getMillis();
    private static final int TOLOKA_TASKS_PER_SUITE = 10;
    private static final Supplier<Integer> TOLOKA_HONEYPOTS_PER_SUITE = () -> ThreadLocalRandom.current().nextInt(1, 3);
    private static final int TOLOKA_BATCH_SIZE = 2000;
    private static final String ATTR_LAST_PROCESSED_STATUS = "last-processed-status";
    private static final String ATTR_LAST_PROCESSED_TELLURIUM = "last-processed-tellurium";
    private static final String ATTR_LAST_PROCESSED_TOLOKA = "last-processed-toloka";
    private static final String ATTR_PROCESSED = "processed";
    private static final String ATTR_TELLURIUM_PARAMETERS = "tellurium.parameters";
    private static final String DEFAULT_TELLURIUM_PARAMETERS_STRING = "{\n" +
            "  \"engine\": \"tellurium.chrome.default\",\n" +
            "  \"wait\": 20,\n" +
            "  \"window\": {\n" +
            "    \"width\": 375,\n" +
            "    \"height\": 1334\n" +
            "  },\n" +
            "  \"browser\": {\n" +
            "    \"mobile\": true,\n" +
            "    \"useragent\": \"Mozilla/5.0 (Linux; arm_64; Android 8.1.0; YNDX-000SB) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/75.0.3770.100 YaBrowser/19.7.1.93.00 Mobile Safari/537.36\"\n" +
            "  }\n" +
            "}";


    private final SiteProblemsService siteProblemsService;
    private final TolokaService tolokaService;
    private final TurboFeedsService turboFeedsService;
    private final TurboSettingsService turboSettingsService;
    private final YtService ytService;
    private final YtTransferManager ytTransferManager;
    private final YqlService yqlService;

    //@Value("${webmaster3.worker.turbo.exportHosts.hahn.path}")
    @Value("hahn://home/webmaster/prod/export/turbo/turbo-hosts")
    private YtPath domainSettingsTable;
    @Value("${webmaster3.worker.tellurium.rootDir}")
    private YtPath telluriumRootDir;
    @Value("${external.yt.service.hahn.root.default}/turbo/cart-urls-moderation")
    private YtPath workDir;
    @Value("hahn://home/turborss/production/yt_pull_job/history_static")
    private YtPath turboUrlsTable;
    @Value("13813876")
    private String tolokaPoolId;

    private YtPath domainStatusTable;
    private YtPath telluriumUrlsDir;
    private YtPath preparedUrlsDir;
    private YtPath tolokaUrlsDir;
    private YtPath telluriumResultsDir;
    private YtPath tolokaHoneypotsTable;
    private YtPath tolokaResultsDir;

    public void init() {
        domainStatusTable = YtPath.path(workDir, "domain-status");
        telluriumUrlsDir = YtPath.path(workDir, "tellurium-urls");
        preparedUrlsDir = YtPath.path(workDir, "prepared-urls");
        tolokaUrlsDir = YtPath.path(workDir, "toloka-urls");
        telluriumResultsDir = YtPath.path(workDir, "tellurium-results");
        tolokaHoneypotsTable = YtPath.path(workDir, "toloka-honeypots");
        tolokaResultsDir = YtPath.path(workDir, "toloka-results");
    }

    @Override
    public Result run(UUID runId) throws Exception {
        //ensureTablesExists();
        // идея такая, каждый день берем порцию необработанных хостов, прокачиваем скрины, отсылаем задания в Толоку, обрабатываем результаты
        // обновляем статусы от Толоки (заодно и проставляем алерты)
        //processTolokaResults();
        // обрабатываем результаты скринов от Теллуриума
        //processTelluriumResults();
        // обновляем статусы доменов
        //updateDomainsStatuses();
        // генерируем свежие задания для Теллуриума
        //prepareTelluriumTask();
        // добавляем задания для Толоки
        //prepareTolokaTaskData();
        // обновим проблемы
        updateProblems();

        return new Result(TaskResult.SUCCESS);
    }

    // логика такая - собираем результаты с CONFIDENCE_page_type >= 80% и одинаковыми результатами для оунера
    private static final String YQL_PROCESS_TOLOKA_RESULTS = "" +
            "use hahn;\n" +
            "pragma yt.ExternalTx = '${TRANSACTION_ID}';\n" +
            "\n" +
            "$tolokaUrls = (SELECT DISTINCT ResultingUrl, Owner, CartUrlTail FROM RANGE(`${TOLOKA_URLS_DIR}`));\n" +
            "\n" +
            "$results = (\n" +
            "    SELECT Owner, CartUrlTail, \n" +
            "        IF (MinPageType <> MaxPageType OR MinIsValidOffer <> MaxIsValidOffer OR MinConfidence < 80.0, \n" +
            "            'UNCERTAIN',\n" + // нет уверенности в результате
            "            IF (MinPageType <> 'cart', \n" +
            "                'ERROR_NOT_CART', \n" +  // точно не корзина
            "                IF (MinIsValidOffer == false, \n" +
            "                    'ERROR_OFFER_NOT_ADDED', \n" + // товара в корзине нет
            "                    'OK'\n" + // все ок
            "                )\n" +
            "            )\n" +
            "        ) as Status, \n" +
            "        ${CURRENT_TIMESTAMP} as LastUpdate, \n" +
            "        UrlSamples \n" +
            "    FROM (\n" +
            "        SELECT Owner, CartUrlTail, \n" +
            "            MIN(CONFIDENCE_page_type) as MinConfidence, \n" +
            "            MIN(OUTPUT_page_type) as MinPageType, \n" +
            "            MAX(OUTPUT_page_type) as MaxPageType, \n" +
            "            MIN(OUTPUT_is_valid_offer) as MinIsValidOffer, \n" +
            "            MAX(OUTPUT_is_valid_offer) as MaxIsValidOffer, \n" +
            "            AGGREGATE_LIST(u.ResultingUrl) as UrlSamples \n" +
            "        FROM $tolokaUrls as u INNER JOIN ${TOLOKA_RESULTS_TABLE} as r \n" +
            "            ON u.ResultingUrl == r.INPUT_resulting_url \n" +
            "        GROUP BY u.Owner as Owner, u.CartUrlTail as CartUrlTail \n" +
            "    )\n" +
            ");" +
            "\n" +
            "-- обновим статусы для корзин\n" +
            "INSERT INTO ${DOMAIN_STATUS_TABLE} WITH TRUNCATE\n" +
            "SELECT dst.Owner as Owner, dst.CartUrlTail as CartUrlTail, dst.Domains as Domains, dst.CartUrls as CartUrls,\n" +
            "  COALESCE(r.Status, dst.Status) as Status, " +
            "  COALESCE(r.LastUpdate, dst.LastUpdate) as LastUpdate \n" +
            "FROM ${DOMAIN_STATUS_TABLE} as dst\n" +
            "LEFT JOIN $results as r\n" +
            "ON dst.Owner == r.Owner AND dst.CartUrlTail == r.CartUrlTail;\n" +
            "";


    //ERROR_NOT_CART,
    //        ERROR_OFFER_NOT_ADDED,
    //        UNCERTAIN, // недостаточно достоверности

    private void processTolokaResults() throws Exception {
        ytService.inTransaction(tolokaResultsDir).execute(cypressService -> {
            List<YtPath> tables = cypressService.list(tolokaResultsDir);
            Collections.sort(tables);
            for (YtPath table : tables) {
                if (isProcessed(cypressService, table)) {
                    continue;
                }
                // обновляем статусы
                String date = table.getName();
                StrSubstitutor builder = new StrSubstitutor(commonParameters(cypressService, date).build());
                yqlService.execute(builder.replace(YQL_PROCESS_TOLOKA_RESULTS));
                markProcessed(cypressService, table);
            }
            return true;
        });
    }

    private static final String YQL_CREATE_URLS_FOR_TOLOKA =
            "use hahn;\n" +
                    "pragma yt.ExternalTx = '${TRANSACTION_ID}';\n" +
                    "\n" +
                    "$telluriumDomainResults = (\n" +
                    "  SELECT Owner, CartUrlTail, \n" +
                    "    IF(ListHas(AGGREGATE_LIST(tr.status), 'success'), null, 'ERROR_UNAVAILABLE') as Status," +
                    "    ${CURRENT_TIMESTAMP} as LastUpdate\n" +
                    "  FROM ${PREPARED_URLS_TABLE} as pr\n" +
                    "  INNER JOIN ${TELLURIUM_RESULTS_TABLE} as tr" +
                    "  ON pr.ResultingUrl == tr.url\n" +
                    "  GROUP By pr.Owner as Owner, pr.CartUrlTail as CartUrlTail\n" +
                    ");\n" +
                    "\n" +
                    "-- обновим статусы для корзин\n" +
                    "INSERT INTO ${DOMAIN_STATUS_TABLE} WITH TRUNCATE\n" +
                    "SELECT dst.Owner as Owner, dst.CartUrlTail as CartUrlTail, dst.Domains as Domains, dst.CartUrls as CartUrls,\n" +
                    "  COALESCE(tdr.Status, dst.Status) as Status, COALESCE(tdr.LastUpdate, dst.LastUpdate) as LastUpdate \n" +
                    "FROM ${DOMAIN_STATUS_TABLE} as dst\n" +
                    "LEFT JOIN $telluriumDomainResults as tdr\n" +
                    "ON dst.Owner == tdr.Owner AND dst.CartUrlTail == tdr.CartUrlTail;\n" +
                    "\n" +
                    "-- сделаем табличку с заданиями для Толоки\n" +
                    "INSERT INTO ${TOLOKA_URLS_TABLE} WITH TRUNCATE\n" +
                    "SELECT Owner, CartUrlTail, Domain, CartUrl, OfferId, OfferName, OfferPrice, ResultingUrl, tr.screenshot as Screenshot\n" +
                    "FROM ${PREPARED_URLS_TABLE} as pr \n" +
                    "INNER JOIN ${TELLURIUM_RESULTS_TABLE} as tr\n" +
                    "ON pr.ResultingUrl == tr.url\n" +
                    "WHERE tr.status == 'success';\n";

    private void processTelluriumResults() throws Exception {
        // перенесем данные обратно на Хан
        ytService.inTransaction(telluriumRootDir).execute(cypressService -> {
            // поищем наши результаты
            YtPath outDir = YtPath.path(telluriumRootDir, "out");
            List<YtPath> resultTables = cypressService.list(outDir);
            for (YtPath resultTable : resultTables) {
                if (resultTable.getName().startsWith(TELLURIUM_TASK_PREFIX)) {
                    YtPath dest = YtPath.path(telluriumResultsDir, resultTable.getName());
                    ytTransferManager.waitForTaskCompletion(ytTransferManager.addTask(resultTable, dest));
                    cypressService.remove(resultTable);
                }
            }
            return true;
        });
        // обрабатываем
        ytService.inTransaction(telluriumResultsDir).execute(cypressService -> {
            List<YtPath> tellResults = cypressService.list(telluriumResultsDir);
            for (YtPath telluriumResult : tellResults) {
                if (!telluriumResult.getName().startsWith(TELLURIUM_TASK_PREFIX)) {
                    continue;
                }
                if (isProcessed(cypressService, telluriumResult)) {
                    continue;
                }
                String date = telluriumResult.getName().substring(TELLURIUM_TASK_PREFIX.length());
                YtPath preparedTable = YtPath.path(preparedUrlsDir, date);
                StrSubstitutor builder = new StrSubstitutor(commonParameters(cypressService, date)
                        .put("TELLURIUM_RESULTS_TABLE", telluriumResult.toYqlPath()).build());
                if (cypressService.exists(preparedTable)) {
                    yqlService.execute(builder.replace(YQL_CREATE_URLS_FOR_TOLOKA));
                }
                markProcessed(cypressService, telluriumResult);
            }
            return true;

        });
    }

    private boolean isProcessed(YtCypressService cypressService, YtPath table) {

        YtNode node = cypressService.getNode(table);
        return Optional.ofNullable(node).map(YtNode::getNodeMeta).map(n -> n.get(ATTR_PROCESSED)).map(JsonNode::asBoolean).orElse(false);
    }

    private void markProcessed(YtCypressService cypressService, YtPath table) {
        cypressService.set(YtPath.attribute(table, ATTR_PROCESSED), BooleanNode.TRUE);
    }

    private ImmutableMap.Builder<String, String> commonParameters(YtCypressService cypressService, String date) {
        return ImmutableMap.<String, String>builder()
                .put("TRANSACTION_ID", cypressService.getTransactionId())
                .put("DOMAIN_SETTINGS_TABLE", domainSettingsTable.toYqlPath())
                .put("DOMAIN_STATUS_TABLE", domainStatusTable.toYqlPath())
                .put("PREPARED_URLS_TABLE", YtPath.path(preparedUrlsDir, date).toYqlPath())
                .put("TOLOKA_URLS_TABLE", YtPath.path(tolokaUrlsDir, date).toYqlPath())
                .put("TOLOKA_URLS_DIR", tolokaUrlsDir.toYtPath())
                .put("TOLOKA_RESULTS_TABLE", YtPath.path(tolokaResultsDir, date).toYqlPath())
                .put("URLS_FOR_TELLURIUM_TABLE", YtPath.path(telluriumUrlsDir, date).toYqlPath())
                .put("CURRENT_TIMESTAMP", String.valueOf(System.currentTimeMillis()))
                .put("REVALIDATION_DATE", String.valueOf(System.currentTimeMillis() - REVALIDATION_PERIOD_MILLIS))
                .put("TURBO_URLS_TABLE", turboUrlsTable.toYqlPath())
                .put("SAMPLES_PER_DOMAIN", String.valueOf(SAMPLES_PER_DOMAIN))
                .put("MAX_DOMAINS_PER_DAY", String.valueOf(MAX_DOMAINS_PER_DAY));
    }

    private static final String YQL_UPDATE_DOMAIN_STATUSES = "-- WMC-8744 - collecting data for Toloka\n" +
            "use hahn;\n" +
            "pragma yt.ExternalTx = '${TRANSACTION_ID}';\n" +
            "\n" +
            "$replaceUtm = Re2::Replace('[&\\?](utm_source|utm_medium|utm_term|utm_campaign)=[^&]*');\n" +
            "\n" +
            "$prepareUrl = ($u) -> {\n" +
            "    $u = if(String::Contains($u, '{offer_url}'), $u, Url::GetTail($u));\n" +
            "    $u = String::Strip($u);    \n" +
            "    $u = $replaceUtm($u, '');\n" +
            "    $u = IF(String::EndsWith($u, '/') and Length($u) > 1, String::RemoveLast($u, '/'), $u);\n" +
            "    return $u; \n" +
            "};\n" +
            "\n" +
            "-- домены со старой корзиной (только с включенными yml-фидами)\n" +
            "$domainWithCartUrls = (\n" +
            "    SELECT Owner, CartUrlTail, \n" +
            "           Yson::Serialize(Yson::FromStringList(AGGREGATE_LIST(Url::CutWWW(Url::GetHost(Host)), 20))) AS Domains, \n" +
            "           Yson::Serialize(Yson::FromStringList(AGGREGATE_LIST(Yson::ConvertToString(ProductInfo.cart_url), 20))) AS CartUrls \n" +
            "    FROM ${DOMAIN_SETTINGS_TABLE} \n" +
            "    WHERE LENGTH(Yson::ConvertToString(ProductInfo.cart_url, Yson::Options(true, false))) > 0\n" +
            "    AND ListHasItems(ListFilter(Yson::ConvertToList(Yson::ParseJson(Yson::ConvertToString(Feeds))), ($f) -> { \n" +
            "            return Yson::ConvertToString($f.type) == 'YML' and Yson::ConvertToString($f.state) == 'ACTIVE';\n" +
            "        }))\n" +
            "    GROUP BY Url::GetOwner(Url::GetHost(Host)) as Owner, $prepareUrl(Yson::ConvertToString(ProductInfo.cart_url)) AS CartUrlTail" +
            ");\n" +
            "\n" +
            "-- обновим статусы корзин по доменам\n" +
            "INSERT INTO ${DOMAIN_STATUS_TABLE} WITH TRUNCATE\n" +
            "SELECT \n" +
            "    COALESCE(dcu.Owner, dst.Owner) AS Owner, \n" +
            "    COALESCE(dcu.CartUrlTail, dst.CartUrlTail) AS CartUrlTail, \n" +
            "    COALESCE(dcu.Domains, dst.Domains) AS Domains, \n" +
            "    COALESCE(dcu.CartUrls, dst.CartUrls) AS CartUrls, \n" +
            "    IF(dcu.Owner IS NULL, 'OFF', COALESCE(dst.Status, 'NEW')) AS Status,\n" +
            "    COALESCE(dst.LastUpdate, ${CURRENT_TIMESTAMP}) AS LastUpdate\n" +
            "FROM\n" +
            "    $domainWithCartUrls AS dcu\n" +
            "FULL JOIN\n" +
            "    ${DOMAIN_STATUS_TABLE} AS dst\n" +
            "ON\n" +
            "    dcu.Owner == dst.Owner and dcu.CartUrlTail == dst.CartUrlTail;\n" +
            "";

    private void updateDomainsStatuses() throws Exception {
        ytService.inTransaction(workDir).execute(cypressService -> {
            // проверим дату последнего обновления статусов
            LocalDate lastProcessed = getProcessedDate(cypressService, ATTR_LAST_PROCESSED_STATUS);
            LocalDate today = LocalDate.now();
            if (today.equals(lastProcessed)) {
                log.info("Domain status already processed, skipping");
                return true;
            }
            StrSubstitutor builder = new StrSubstitutor(commonParameters(cypressService, today.toString()).build());
            String query = builder.replace(YQL_UPDATE_DOMAIN_STATUSES);
            yqlService.execute(query);
            cypressService.set(YtPath.attribute(domainStatusTable, ATTR_LAST_PROCESSED_STATUS), TextNode.valueOf(today.toString()));
            return true;
        });
    }

    private LocalDate getProcessedDate(YtCypressService cypressService, String attr) {
        return Optional.ofNullable(cypressService.exists(domainStatusTable) ? cypressService.getNode(domainStatusTable) : null)
                .map(YtNode::getNodeMeta)
                .map(n -> n.get(attr))
                .map(JsonNode::asText)
                .map(LocalDate::parse).orElse(null);
    }

    private interface F {
        YtSchema DOMAIN_STATUS_SCHEMA = new YtSchema();
        YtColumn<String> OWNER = DOMAIN_STATUS_SCHEMA.addColumn("Owner", YtColumn.Type.STRING);
        YtColumn<List<String>> DOMAINS = DOMAIN_STATUS_SCHEMA.addColumn("Domains", YtColumn.Type.any());
        YtColumn<List<String>> CART_URLS = DOMAIN_STATUS_SCHEMA.addColumn("CartUrls", YtColumn.Type.any());
        YtColumn<String> CART_URL_TAIL = DOMAIN_STATUS_SCHEMA.addColumn("CartUrlTail", YtColumn.Type.STRING);
        YtColumn<String> STATUS = DOMAIN_STATUS_SCHEMA.addColumn("Status", YtColumn.Type.STRING);
        YtColumn<Long> LAST_UPDATE = DOMAIN_STATUS_SCHEMA.addColumn("LastUpdate", YtColumn.Type.INT_64);
    }

    private void ensureTablesExists() throws Exception {
        ytService.withoutTransaction(cypressService -> {
            if (!cypressService.exists(domainStatusTable)) {
                YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.DOMAIN_STATUS_SCHEMA);
                cypressService.create(domainStatusTable, YtNode.NodeType.TABLE, true, attributes, true);
            }
            cypressService.create(telluriumResultsDir, YtNode.NodeType.MAP_NODE, true, null, true);
            cypressService.create(tolokaUrlsDir, YtNode.NodeType.MAP_NODE, true, null, true);
            cypressService.create(preparedUrlsDir, YtNode.NodeType.MAP_NODE, true, null, true);
            cypressService.create(telluriumUrlsDir, YtNode.NodeType.MAP_NODE, true, null, true);
            return true;
        });
    }

    private static final String YQL_PREPARE_TELLURIUM_URLS = "" +
            "use hahn;\n" +
            "pragma yt.ExternalTx = '${TRANSACTION_ID}';\n" +
            "\n" +
            "$offerUrlPrefixCapture = Re2::Capture('(\\{offer_url\\})([\\?\\&]).*');\n" +
            "$offerUrlPrefixReplace = Re2::Replace('(\\{offer_url\\})([\\?\\&])');" +
            "\n" +
            "$correctCartUrlCgiDelimiters = ($cartUrl, $offerUrl) -> {\n" +
            "    $questionPos = Find($offerUrl, '?');\n" +
            "    $prefixCapture = $offerUrlPrefixCapture($cartUrl);\n" +
            "    return if ($prefixCapture is not null,\n" +
            "        if ($questionPos is null,  \n" +
            "            if ($prefixCapture._2 == '&', \n" +
            "                $offerUrlPrefixReplace($cartUrl, '\\\\1?'),\n" +
            "                $cartUrl\n" +
            "            ),    \n" +
            "            if (String::EndsWith($offerUrl, '?') or String::EndsWith($offerUrl, '&'),\n" +
            "                Substring($cartUrl, 0, cast((Length($cartUrl) - 1) as Uint32)),\n" +
            "                if ($prefixCapture._2 == '?', $offerUrlPrefixReplace($cartUrl, '\\\\1&'), $cartUrl)\n" +
            "            )\n" +
            "        ),    \n" +
            "        -- true\n" +
            "        $cartUrl\n" +
            "    );    \n" +
            "\n" +
            "};\n" +
            "\n" +
            "$correctCartUrlHttpPrefix = ($cartUrl) -> {\n" +
            "    return if(String::StartsWith($cartUrl, 'http://') or String::StartsWith($cartUrl, 'https://'), $cartUrl, 'http://' || $cartUrl);\n" +
            "};\n" +
            "\n" +
            "$buildCartUrl = ($cartUrl, $offerUrl, $offerId) -> {\n" +
            "    $cartUrl = String::Strip($cartUrl);\n" +
            "    $cartUrl = $correctCartUrlCgiDelimiters($cartUrl, $offerUrl);\n" +
            "    $result = String::ReplaceAll(String::ReplaceAll($cartUrl, '{offer_url}', $offerUrl), '{offer_id}', $offerId);\n" +
            "    return $correctCartUrlHttpPrefix($result);\n" +
            "};" +
            "\n" +
            "$getRandomItemFromList = ($l, $r) -> { return $l[RandomNumber($r) % ListLength($l)]; };\n" +
            "-- соберем N доменов, которые будем сегодня проверять\n" +
            "$domainToCheck = (\n" +
            "    SELECT Owner, CartUrlTail, $getRandomItemFromList(Yson::ConvertToStringList(Domains), Owner) as Domain, \n" +
            "        $getRandomItemFromList(Yson::ConvertToStringList(CartUrls), Owner) as CartUrl, LastUpdate \n" +
            "    FROM ${DOMAIN_STATUS_TABLE} \n" +
            "    WHERE (Status == 'NEW' OR LastUpdate < ${REVALIDATION_DATE}) AND Status <> 'OFF'\n" +
            "    ORDER BY LastUpdate, RANDOM(Owner)\n" +
            "    LIMIT ${MAX_DOMAINS_PER_DAY}\n" +
            ");\n" +
            "\n" +
            "$productInfo = ($value) -> { return Yson::ParseJson($value).attrs.body[0].product_info; };\n" +
            "\n" +
            "INSERT INTO @PREPARED_URLS_TABLE_TMP SELECT * FROM $domainToCheck;\n" +
            "\n" +
            "-- соберем урлы заданий для Толоки и Теллуриума\n" +
            "INSERT INTO ${PREPARED_URLS_TABLE} WITH TRUNCATE \n" +
            "SELECT  \n" +
            "    s.Owner AS Owner,\n" +
            "    s.CartUrlTail AS CartUrlTail,\n" +
            "    s.Domain AS Domain,\n" +
            "    String::Strip(s.CartUrl) AS CartUrl,\n" +
            "    String::Strip(s.OfferUrl) AS OfferUrl,\n" +
            "    s.OfferId AS OfferId,\n" +
            "    String::Strip(s.OfferName) AS OfferName,\n" +
            "    s.OfferPrice AS OfferPrice,\n" +
            "    String::Strip(s.ResultingUrl) AS ResultingUrl\n" +
            "FROM\n" +
            "(\n" +
            "    SELECT \n" +
            "        Owner, \n" +
            "        CartUrlTail, \n" +
            "        AGGREGATE_LIST(AsStruct(\n" +
            "            Owner AS Owner,\n" +
            "            CartUrlTail AS CartUrlTail,\n" +
            "            Domain AS Domain,\n" +
            "            CartUrl AS CartUrl,\n" +
            "            UNWRAP(Yson::ConvertToString(meta.original_url)) AS OfferUrl,\n" +
            "            UNWRAP(Yson::ConvertToString($productInfo(value).offer_id)) AS OfferId,\n" +
            "            UNWRAP(Yson::ConvertToString($productInfo(value).offer_name)) AS OfferName,\n" +
            "            Yson::ConvertToString($productInfo(value).offer_price.value) AS OfferPrice,\n" +
            "            $buildCartUrl(CartUrl, \n" +
            "                UNWRAP(Yson::ConvertToString(meta.original_url)), \n" +
            "                UNWRAP(Yson::ConvertToString($productInfo(value).offer_id))\n" +
            "            ) AS ResultingUrl\n" +
            "        ), ${SAMPLES_PER_DOMAIN}) AS Samples \n" +
            "    FROM ${TURBO_URLS_TABLE} AS hs\n" +
            "    INNER JOIN $domainToCheck AS wh\n" +
            "        ON Url::CutWWW(Url::GetHost(hs.key)) == wh.Domain \n" +
            "    WHERE \n" +
            "        hs.`action` == 'modify' -- available offer (?) \n" +
            "        AND Yson::Contains(meta, 'original_url')\n" +
            "        AND NOT Yson::Contains(meta, 'test')\n" +
            "        AND Yson::ConvertToString(meta.source) == 'yml'\n" +
            "        AND Yson::Contains($productInfo(value), 'available')\n" +
            "        AND Yson::ConvertToBool($productInfo(value).available) == true\n" +
            "    GROUP BY wh.Owner AS Owner, wh.CartUrlTail AS CartUrlTail" +
            "    ORDER BY Random(Owner)\n" +
            ")\n" +
            "FLATTEN BY Samples AS s;\n" +
            "\n" +
            "COMMIT;\n" +
            "\n" +
            "INSERT INTO ${URLS_FOR_TELLURIUM_TABLE} WITH TRUNCATE \n" +
            "SELECT ResultingUrl as url FROM ${PREPARED_URLS_TABLE} WHERE ResultingUrl IS NOT NULL;\n" +
            "-- сразу обновим LastUpdate обрабатываемым доменам, чтобы не послать их повторно на проверку\n" +
            "INSERT INTO ${DOMAIN_STATUS_TABLE} WITH TRUNCATE\n" +
            "SELECT \n" +
            "    dst.Owner AS Owner, \n" +
            "    dst.CartUrlTail AS CartUrlTail," +
            "    dst.Domains AS Domains, \n" +
            "    dst.CartUrls AS CartUrls,\n" +
            "    IF(pr.Owner IS NOT NULL AND pr.ResultingUrls == 0, 'CHECK_UNAVAILABLE', dst.Status) AS Status,\n" +
            "    IF(pr.Owner IS NULL, dst.LastUpdate, ${CURRENT_TIMESTAMP}) AS LastUpdate\n" +
            "FROM ${DOMAIN_STATUS_TABLE} AS dst\n" +
            "LEFT JOIN\n" +
            "(" +
            "    SELECT Owner, CartUrlTail, COUNT(pr.ResultingUrl) AS ResultingUrls FROM @PREPARED_URLS_TABLE_TMP as pr_all\n" +
            "    LEFT JOIN ${PREPARED_URLS_TABLE} as pr ON pr_all.Owner == pr.Owner AND pr_all.CartUrlTail == pr.CartUrlTail\n" +
            "    GROUP BY pr_all.Owner AS Owner, pr_all.CartUrlTail AS CartUrlTail\n" +
            ") AS pr\n" +
            "ON dst.CartUrlTail == pr.CartUrlTail and dst.Owner == pr.Owner;\n" +
            "\n";

    private void prepareTelluriumTask() throws Exception {
        LocalDate today = LocalDate.now();
        YtPath telluriumUrlsTable = YtPath.path(telluriumUrlsDir, today.toString());
        // соберем N урлов для прокачки Теллуриумом
        ytService.inTransaction(workDir).execute(cypressService -> {
            if (cypressService.exists(telluriumUrlsTable)) {
                log.info("Tellurium already processed, skipping");
                return true;
            }

            StrSubstitutor builder = new StrSubstitutor(commonParameters(cypressService, today.toString()).build());
            String query = builder.replace(YQL_PREPARE_TELLURIUM_URLS);
            yqlService.execute(query);
            cypressService.set(YtPath.attribute(domainStatusTable, ATTR_LAST_PROCESSED_TELLURIUM), TextNode.valueOf(today.toString()));
            cypressService.set(YtPath.attribute(telluriumUrlsTable, ATTR_TELLURIUM_PARAMETERS),
                    JsonMapping.readValue(DEFAULT_TELLURIUM_PARAMETERS_STRING, ObjectNode.class));
            // заливаем на место
            return true;
        });

        ytService.withoutTransaction(cypressService -> {
            YtPath resultTelluriumTable = YtPath.path(telluriumRootDir, "in/" + TELLURIUM_TASK_PREFIX + today.toString());
            if (!isProcessed(cypressService, telluriumUrlsTable)) {
                String transferTaskId = ytTransferManager.addTask(telluriumUrlsTable, resultTelluriumTable);
                log.info("Added transfer manager task {}", transferTaskId);
                ytTransferManager.waitForTaskCompletion(transferTaskId);
                markProcessed(cypressService, telluriumUrlsTable);
            }
            return true;
        });
    }

    private void prepareTolokaTaskData() throws Exception {
        // читаем табличку для Толоки и грузим ее
        ytService.inTransaction(tolokaUrlsDir).execute(cypressService -> {
            List<YtPath> tables = cypressService.list(tolokaUrlsDir);
            for (YtPath table : tables) {
                if (isProcessed(cypressService, table)) {
                    log.info("Toloka table {} already processed, skipping", table);
                    continue;
                }
                List<TolokaTask> honeypots = loadHoneypots(cypressService);

                // создадим наборы заданий
                AsyncTableReader<TolokaUrlRow> tableReader = new AsyncTableReader<>(cypressService, table, Range.all(),
                        YtTableReadDriver.createYSONDriver(TolokaUrlRow.class))
                        .withThreadName("toloka-urls-reader").needLock(false);
                List<TolokaTask> tasks = new ArrayList<>();
                try (AsyncTableReader.TableIterator<TolokaUrlRow> iterator = tableReader.read()) {
                    while (iterator.hasNext()) {
                        TolokaUrlRow row = iterator.next();
                        TolokaTask task = TolokaTask.builder().poolId(tolokaPoolId).overlap(3).inputValues(JsonMapping.OM.createObjectNode()
                                .put("resulting_url", row.getResultingUrl())
                                .put("offer_name", row.getOfferName())
                                .put("cart_screenshot", row.getScreenshot())
                        ).build();
                        tasks.add(task);
                    }
                    Collections.shuffle(tasks);
                    Collections.shuffle(honeypots);
                    List<TolokaTaskSuite> suites = new ArrayList<>();
                    int honeypotsIndex = 0;
                    for (int taskIndex = 0; taskIndex < tasks.size(); ) {
                        TolokaTaskSuite suite = TolokaTaskSuite.builder().poolId(tolokaPoolId).overlap(3).tasks(new ArrayList<>()).build();
                        // подмешаем 1-2 ханипотов
                        int suiteIndex = 0;
                        if (!honeypots.isEmpty()) {
                            for (int i = 0; i < TOLOKA_HONEYPOTS_PER_SUITE.get(); i++, honeypotsIndex++, suiteIndex++) {
                                suite.getTasks().add(honeypots.get(honeypotsIndex % honeypots.size()));
                            }
                        }
                        for (; suiteIndex < TOLOKA_TASKS_PER_SUITE; suiteIndex++, taskIndex++) {
                            suite.getTasks().add(tasks.get(taskIndex % tasks.size()));
                        }
                        Collections.shuffle(suite.getTasks());
                        suites.add(suite);
                        if (suites.size() >= TOLOKA_BATCH_SIZE) {
                            tolokaService.postTaskSuites(suites);
                            suites.clear();
                        }
                    }
                    if (!suites.isEmpty()) {
                        tolokaService.postTaskSuites(suites);
                    }

                } catch (IOException e) {
                    throw new WebmasterException("Error when reading toloka ulrs", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                } catch (Exception e) {
                    throw new WebmasterException("Error when posting toloka tasks", new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Toloka error"), e);
                }

                markProcessed(cypressService, table);
            }
            return true;
        });
    }

    private List<TolokaTask> loadHoneypots(YtCypressService cypressService) throws InterruptedException {
        List<TolokaTask> honeypots = new ArrayList<>();
        if (!cypressService.exists(tolokaHoneypotsTable)) {
            return honeypots;
        }
        // загрузим все ханипоты
        AsyncTableReader<TolokaUrlRow> honeypotsReader = new AsyncTableReader<>(cypressService, tolokaHoneypotsTable, Range.all(),
                YtTableReadDriver.createYSONDriver(TolokaUrlRow.class))
                .withThreadName("toloka-urls-reader").needLock(false);

        try (AsyncTableReader.TableIterator<TolokaUrlRow> iterator = honeypotsReader.read()) {
            while (iterator.hasNext()) {
                TolokaUrlRow row = iterator.next();
                TolokaTask task = TolokaTask.builder()
                        .poolId(tolokaPoolId)
                        .overlap(3)
                        .inputValues(
                                JsonMapping.OM.createObjectNode()
                                        .put("resulting_url", row.getResultingUrl())
                                        .put("offer_name", row.getOfferName())
                                        .put("cart_screenshot", row.getScreenshot()
                                        )
                        ).knownSolutions(Collections.singletonList(
                                TolokaTask.TolokaTaskKnownSolution.builder().correctnessWeight(1.0)
                                        .outputValues(JsonMapping.OM.createObjectNode()
                                                .put("is_cart_page", row.getIsCartPage().toString())
                                                .put("is_valid_offer", row.getIsValidOffer().toString())
                                        )
                                        .build()))
                        .build();
                honeypots.add(task);
            }
        } catch (IOException e) {
            throw new WebmasterException("Error when reading honeypots ulrs", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
        }
        return honeypots;
    }

    private void updateProblems() throws Exception {
        ytService.withoutTransaction(cypressService -> {
            AsyncTableReader<DomainStatusRow> tableReader = new AsyncTableReader<>(cypressService, domainStatusTable, Range.all(),
                    YtTableReadDriver.createYSONDriver(DomainStatusRow.class)).withThreadName("domain-status-reader");
            DateTime updateStarted = DateTime.now();
            try (AsyncTableReader.TableIterator<DomainStatusRow> iterator = tableReader.read()) {
                Map<WebmasterHostId, ProblemSignal> map = new HashMap<>(BATCH_SIZE);
                while (iterator.hasNext()) {
                    DomainStatusRow row = iterator.next();
                    // создадим алерт
                    TurboInvalidCartUrl.CartUrlStatus status = TurboInvalidCartUrl.CartUrlStatus.valueOf(row.getStatus());
                    DateTime lastUpdate = new DateTime(row.getLastUpdate());
                    // делаем проблему для нужных хостов (у которых та же корзина)
                    for (int i = 0; i < row.getDomains().size(); i++) {
                        String domain = row.getDomains().get(i);
                        String cartUrl = row.getCartUrls().get(i);
                        String actualCartUrl = Optional.ofNullable(turboSettingsService.getSettings(domain))
                                .map(TurboHostSettings::getCommerceSettings)
                                .filter(s -> !s.isTurboCartEnabled()) // при включенной турбо-корзине алерт не нужен
                                .map(TurboCommerceSettings::getCartUrl).orElse(null);
                        // YML должен быть включен
                        boolean hasActiveYmlFeeds = turboFeedsService.getFeeds(domain).stream()
                                .anyMatch(f -> f.isActive() && f.getType() == TurboFeedType.YML);
                        ProblemSignal signal;
                        if (hasActiveYmlFeeds && cartUrl.equals(actualCartUrl) && status.isBad()) {
                            signal = new ProblemSignal(new TurboInvalidCartUrl(status), lastUpdate);
                        } else {
                            signal = new ProblemSignal(SiteProblemTypeEnum.TURBO_INVALID_CART_URL, SiteProblemState.ABSENT, lastUpdate);
                        }

                        map.put(IdUtils.urlToHostId(domain), signal);
                        if (map.size() >= BATCH_SIZE) {
                            siteProblemsService.updateCleanableProblems(map, SiteProblemTypeEnum.TURBO_INVALID_CART_URL);
                            map.clear();
                        }
                    }
                }
                siteProblemsService.updateCleanableProblems(map, SiteProblemTypeEnum.TURBO_INVALID_CART_URL);
                map.clear();
                siteProblemsService.notifyCleanableProblemUpdateFinished(SiteProblemTypeEnum.TURBO_INVALID_CART_URL, updateStarted);
            } catch (IOException e) {
                throw new WebmasterException("Yt error", new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
            }
            return true;
        });
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.VALIDATE_CART_URLS_FOR_TURBO;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    private static final class TolokaUrlRow {
        @JsonProperty("Domain")
        String domain;
        @JsonProperty("CartUrl")
        String cartUrl;
        @JsonProperty("ResultingUrl")
        String resultingUrl;
        @JsonProperty("OfferId")
        String offerId;
        @JsonProperty("OfferName")
        String offerName;
        @JsonProperty("OfferPrice")
        String offerPrice;
        @JsonProperty("Screenshot")
        String screenshot;
        @JsonProperty("IsCartPage")
        Boolean isCartPage;
        @JsonProperty("IsValidOffer")
        Boolean isValidOffer;
    }

    @lombok.Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    private static final class DomainStatusRow {
        @JsonProperty("Owner")
        String owner;
        @JsonProperty("CartUrlTail")
        String cartUrlTail;
        @JsonProperty("Status")
        String status;
        @JsonProperty("LastUpdate")
        Long lastUpdate;
        @JsonProperty("Domains")
        List<String> domains;
        @JsonProperty("CartUrls")
        List<String> cartUrls;
    }
}
