package ru.yandex.webmaster3.worker.turbo.export;

import java.net.IDN;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.LongNode;
import com.google.common.base.Strings;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.With;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.service.HostOwnerService;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.solomon.HandleCommonMetricsService;
import ru.yandex.webmaster3.core.solomon.Indicators;
import ru.yandex.webmaster3.core.solomon.SolomonSensor;
import ru.yandex.webmaster3.core.turbo.model.TurboAccessSettings;
import ru.yandex.webmaster3.core.turbo.model.TurboHostHeader;
import ru.yandex.webmaster3.core.turbo.model.TurboHostHeaderType;
import ru.yandex.webmaster3.core.turbo.model.TurboHostSettings;
import ru.yandex.webmaster3.core.turbo.model.advertising.AdvertisingSettings;
import ru.yandex.webmaster3.core.turbo.model.analytics.AnalyticsSettings;
import ru.yandex.webmaster3.core.turbo.model.app.TurboAppSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboCommerceSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboPaymentsSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.TurboPlusSettings;
import ru.yandex.webmaster3.core.turbo.model.commerce.delivery.DeliverySection;
import ru.yandex.webmaster3.core.turbo.model.desktop.TurboDesktopSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedSettings;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedState;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedType;
import ru.yandex.webmaster3.core.turbo.model.menu.TurboMenuItem;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.host.AllVerifiedHostsCacheService;
import ru.yandex.webmaster3.storage.host.CommonDataState;
import ru.yandex.webmaster3.storage.host.CommonDataType;
import ru.yandex.webmaster3.storage.settings.dao.CommonDataStateYDao;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDesktopSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.TurboDomainsStateCHDao;
import ru.yandex.webmaster3.storage.turbo.dao.app.TurboAppSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.commerce.TurboPlusSettingsYDao;
import ru.yandex.webmaster3.storage.turbo.dao.logo.DefaultTurboLogoYDao;
import ru.yandex.webmaster3.storage.turbo.dao.scc.TurboSccService;
import ru.yandex.webmaster3.storage.turbo.logo.TurboLogoData;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService.TurboDomainState;
import ru.yandex.webmaster3.storage.turbo.service.TurboDomainsStateService.TurboMarketFeed;
import ru.yandex.webmaster3.storage.turbo.service.TurboFeedsService;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboAuthorizationSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboAutoRelatedSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboCommentsSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboDesktopSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboDisplaySettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboHostSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboMenuItemData;
import ru.yandex.webmaster3.storage.turbo.service.preview.TurboSearchSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.app.TurboAppSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.DeliveryInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.HeaderParamsInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.MarketInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.NewCommerceSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.OrderInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.PaymentInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.ProductInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.ShopInfo;
import ru.yandex.webmaster3.storage.turbo.service.preview.commerce.TurboPlusSettingsData;
import ru.yandex.webmaster3.storage.turbo.service.validation.TurboParserService;
import ru.yandex.webmaster3.storage.util.yt.TableWriter;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtOperationId;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * Created by Oleg Bazdyrev on 27/07/2017.
 */
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class ExportTurboFeedsTask extends PeriodicTask<ExportTurboFeedsTask.TaskState> {

    private static final Logger log = LoggerFactory.getLogger(ExportTurboFeedsTask.class);
    private static final String CATEGORY_LABEL_VALUE = "turbo";
    private static final String HOSTS_TYPE = "hosts";
    private static final String FEEDS_TYPE = "feeds";
    private static final String SCC_TYPE = "turbo_scc";
    private static final String SCC_ACTIVE_TYPE = "turbo_scc_active";
    private static final String STATE_LABEL = "state";
    private static final long ALIGN_SECONDS = 1800;
    private static final String ATTR_TIMESTAMP = "@timestamp";

    private static final ObjectMapper OM = new ObjectMapper()
            .configure(JsonParser.Feature.ALLOW_SINGLE_QUOTES, true)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL);

    private interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<String> HOST = SCHEMA.addColumn("Host", YtColumn.Type.STRING);
        YtColumn<String> OWNER = SCHEMA.addColumn("Owner", YtColumn.Type.STRING);
        YtColumn<Object> FEEDS = SCHEMA.addColumn("Feeds", YtColumn.Type.ANY);
        YtColumn<Object> ANALYTICS = SCHEMA.addColumn("Analytics", YtColumn.Type.ANY);
        YtColumn<Object> ADVERTISING = SCHEMA.addColumn("Advertising", YtColumn.Type.ANY);
        YtColumn<String> TITLE = SCHEMA.addColumn("Title", YtColumn.Type.STRING);
        YtColumn<Object> LOGO = SCHEMA.addColumn("Logo", YtColumn.Type.ANY);
        YtColumn<Long> HEADER_TYPE = SCHEMA.addColumn("HeaderType", YtColumn.Type.INT_64);
        YtColumn<Object> HEADER_PARAMS = SCHEMA.addColumn("HeaderParams", YtColumn.Type.any());
        YtColumn<String> LOGO_ID = SCHEMA.addColumn("LogoId", YtColumn.Type.STRING);
        YtColumn<Long> LOGO_GROUP_ID = SCHEMA.addColumn("LogoGroupId", YtColumn.Type.INT_64);
        YtColumn<Object> MENU = SCHEMA.addColumn("Menu", YtColumn.Type.ANY);
        YtColumn<Object> FEEDBACK = SCHEMA.addColumn("Feedback", YtColumn.Type.ANY);
        YtColumn<String> CSS = SCHEMA.addColumn("Css", YtColumn.Type.STRING);
        YtColumn<Boolean> AUTO_RELATED = SCHEMA.addColumn("AutoRelated", YtColumn.Type.BOOLEAN);
        YtColumn<TurboAutoRelatedSettingsData> AUTO_RELATED_SETTINGS = SCHEMA.addColumn("AutoRelatedSettings", YtColumn.Type.any());
        YtColumn<Boolean> AUTO_MORDA = SCHEMA.addColumn("AutoMorda", YtColumn.Type.BOOLEAN);
        YtColumn<YtProductInfo> PRODUCT_INFO = SCHEMA.addColumn("ProductInfo", YtColumn.Type.any());
        YtColumn<TurboDesktopSettingsData> DESKTOP = SCHEMA.addColumn("Desktop", YtColumn.Type.any());
        YtColumn<String> EXPERIMENT = SCHEMA.addColumn("Experiment", YtColumn.Type.STRING);
        YtColumn<TurboAuthorizationSettingsData> AUTHORIZATION = SCHEMA.addColumn("Authorization", YtColumn.Type.any());
        YtColumn<TurboCommentsSettingsData> COMMENTS = SCHEMA.addColumn("Comments", YtColumn.Type.any());
        YtColumn<TurboSearchSettingsData> SEARCH = SCHEMA.addColumn("Search", YtColumn.Type.any());
        YtColumn<List<TurboMenuItemData>> CATEGORY_INFO = SCHEMA.addColumn("CategoryInfo", YtColumn.Type.any());
        YtColumn<TurboAppSettingsData> TURBO_APP = SCHEMA.addColumn("TurboApp", YtColumn.Type.any());
        YtColumn<TurboPlusSettingsData> TURBO_PLUS = SCHEMA.addColumn("TurboPlus", YtColumn.Type.any());
        YtColumn<String> DESCRIPTION = SCHEMA.addColumn("Description", YtColumn.Type.STRING); // описание ТурбоАппа
        YtColumn<TurboHostSettingsData.Logo> LOGO_SQUARE = SCHEMA.addColumn("LogoSquare", YtColumn.Type.any()); // логотип ТурбоАппа
        YtColumn<String> THEME = SCHEMA.addColumn("Theme", YtColumn.Type.STRING); // тема ТурбоАппа
        YtColumn<NewCommerceSettingsData> ECOMMERCE = SCHEMA.addColumn("Ecommerce", YtColumn.Type.any());
        YtColumn<Long> SCC_CHECK_START_DATE = SCHEMA.addColumn("SccCheckStartDate", YtColumn.Type.INT_64);
        YtColumn<Object> DISPLAY_PARAMS = SCHEMA.addColumn("DisplayParams", YtColumn.Type.any());
        YtColumn<Object> MARKET_INFO = SCHEMA.addColumn("MarketInfo", YtColumn.Type.any());
    }

    private final CommonDataStateYDao commonDataStateYDao;
    private final DefaultTurboLogoYDao defaultTurboLogoYDao;
    private final HandleCommonMetricsService handleCommonMetricsService;
    private final HostOwnerService hostOwnerService;
    private final TurboAppSettingsYDao turboAppSettingsYDao;
    private final TurboDesktopSettingsYDao turboDesktopSettingsYDao;
    private final TurboDomainSettingsYDao turboDomainSettingsYDao;
    private final TurboDomainsStateCHDao turboDomainsStateCHDao;
    private final TurboFeedsService turboFeedsService;
    private final TurboParserService turboParserService;
    private final TurboPlusSettingsYDao turboPlusSettingsYDao;
    private final YtService ytService;
    private final AllVerifiedHostsCacheService allVerifiedHostsCacheService;

    @Setter
    private List<YtPath> tablePaths;

    private Set<String> hostWhiteList = Collections.emptySet();

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Exporting turbo feeds to YT table");
        TaskState state = new TaskState();
        setState(state);
        state.taskStart = System.currentTimeMillis();
        // готовим данные для экспорта
        log.info("Start prepare data at millis time {}", System.currentTimeMillis());
        YtTableData tableData = ytService.prepareTableData("turbo-hosts", tableWriter -> {
            List<Pair<String, TurboHostSettings>> batch = new ArrayList<>();
            turboDomainSettingsYDao.forEach(pair -> {
                batch.add(pair);
                if (batch.size() >= 1000) {
                    writeBatch(batch, tableWriter);
                    batch.clear();
                }
            });
            if (!batch.isEmpty()) {
                writeBatch(batch, tableWriter);
            }
        });
        log.info("Finish prepare data at millis time {}", System.currentTimeMillis());


        // пишем подгтовленные данные на место

        long nanos1 = System.nanoTime();
        tablePaths.parallelStream()
                .forEach(tablePath -> writeDataToYt(tableData, tablePath));
        long nanos2 = System.nanoTime();
        state.addWatchNs(WatchType.EXPORT_DATA_TO_YT, nanos2 - nanos1);


        // сохраним кол-во активных хостов для лендинга
        commonDataStateYDao.update(new CommonDataState(CommonDataType.TURBO_ACTIVE_HOSTS_COUNT,
                String.valueOf(state.turboActiveHostsCount), DateTime.now()));
        // чистим за собой
        tableData.delete();
        // отправляем статистику в Solomon
        sendStatisticsToSolomon(state.taskStart);
        return new Result(state.tablesSuccess.isEmpty() ? TaskResult.FAIL : TaskResult.SUCCESS);
    }

    private void writeDataToYt(YtTableData tableData, YtPath tablePath) {
        log.info("Start write data to {} at millis time {}", tablePath, System.currentTimeMillis());
        try {
            ytService.inTransaction(tablePath).execute(cypressService -> {
                try {
                    // создаем табличку
                    YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
                    YtPath tmpTablePath = YtPath.create(tablePath.getCluster(), tablePath.getPathWithoutCluster() + "_tmp");
                    cypressService.remove(tmpTablePath, false, true);
                    cypressService.create(tmpTablePath, YtNode.NodeType.TABLE, true, attributes, true);
                    // пишем
                    cypressService.writeTable(tmpTablePath, tableData);
                    final YtOperationId operationId = cypressService.sort(tmpTablePath, tmpTablePath, F.HOST.getName());
                    if (!cypressService.waitFor(operationId)) {
                        throw new WebmasterException("Could not wait for sort operation completion",
                                new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), null));
                    }
                    cypressService.move(tmpTablePath, tablePath, false, true);
                    // сохраним таймстемп начала экспорта
                    cypressService.set(YtPath.path(tablePath, ATTR_TIMESTAMP), new LongNode(state.taskStart / 1000L));
                } catch (YtException e) {
                    log.warn("Cannot write turbo hosts to " + tableData, e);
                    throw new RuntimeException(e);
                }
                return true;
            });
            state.tablesSuccess.add(tablePath);
        } catch (Exception e) {
            log.error("Failed to write turbo hosts to " + tableData, e);
            state.tablesFailed.add(tablePath);
        }
        log.info("Finish write data to {} at millis time {}", tablePath, System.currentTimeMillis());
    }


    private void sendStatisticsToSolomon(long timestamp) {
        List<SolomonSensor> solomonSensors = new ArrayList<>();
        solomonSensors.add(SolomonSensor.createAligned(timestamp, ALIGN_SECONDS, state.turboHostsCount)
                .withLabel(SolomonSensor.LABEL_CATEGORY, CATEGORY_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, HOSTS_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE));

        solomonSensors.add(SolomonSensor.createAligned(timestamp, ALIGN_SECONDS, state.totalTurboFeedsCount)
                .withLabel(SolomonSensor.LABEL_CATEGORY, CATEGORY_LABEL_VALUE)
                .withLabel(SolomonSensor.LABEL_DATA_TYPE, FEEDS_TYPE)
                .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                .withLabel(STATE_LABEL, "(all)"));

        for (YtTurboFeedState feedState : YtTurboFeedState.values()) {
            Long value = state.turboFeedsCount.getOrDefault(feedState, 0L);
            solomonSensors.add(SolomonSensor.createAligned(timestamp, ALIGN_SECONDS, value)
                    .withLabel(SolomonSensor.LABEL_CATEGORY, CATEGORY_LABEL_VALUE)
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, FEEDS_TYPE)
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.QUEUE_SIZE)
                    .withLabel(STATE_LABEL, feedState.name().toLowerCase()));
        }

        for (var p : state.sccStatuses.entrySet()) {
            solomonSensors.add(SolomonSensor.createAligned(timestamp, p.getValue())
                    .withLabel(STATE_LABEL, p.getKey().name())
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, SCC_TYPE)
                    .withLabel(SolomonSensor.LABEL_CATEGORY, CATEGORY_LABEL_VALUE));
        }

        for (var p : state.sccStatusesWithActiveYml.entrySet()) {
            solomonSensors.add(SolomonSensor.createAligned(timestamp, p.getValue())
                    .withLabel(STATE_LABEL, p.getKey().name())
                    .withLabel(SolomonSensor.LABEL_INDICATOR, Indicators.DATA_AGE)
                    .withLabel(SolomonSensor.LABEL_DATA_TYPE, SCC_ACTIVE_TYPE)
                    .withLabel(SolomonSensor.LABEL_CATEGORY, CATEGORY_LABEL_VALUE));
        }

        handleCommonMetricsService.handle(solomonSensors, 400);
    }

    private static byte[] writeValueAsBytes(Object value) {
        try {
            return OM.writeValueAsBytes(value);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private void writeBatch(List<Pair<String, TurboHostSettings>> batch, TableWriter tw) {

        long nanos1 = System.nanoTime();
        //выфильтровываем неподтвержденные хосты
        batch = batch.stream()
                .filter(pair -> {
                    final List<WebmasterHostId> hostIds = IdUtils.allHostsForDomain(pair.getKey());
                    return hostIds.stream().anyMatch(allVerifiedHostsCacheService::contains);
                }).collect(Collectors.toList());
        state.addWatchNs(WatchType.FILTER_UNVERIFIED_HOSTS, System.nanoTime() - nanos1);

        // соберем owner-ов
        List<String> domains = batch.stream().map(Pair::getLeft).collect(Collectors.toList());
        if (domains.isEmpty()) {
            return;
        }
        nanos1 = System.nanoTime();
        // пофигу из какого хоста брать оунера
        Map<WebmasterHostId, String> owners = hostOwnerService.getLongestOwners(domains.stream()
                .map(IdUtils::urlToHostId).collect(Collectors.toList()));
        long nanos2 = System.nanoTime();
        state.addWatchNs(WatchType.OWNERS, nanos2 - nanos1);
        nanos1 = System.nanoTime();
        Map<String, TurboDomainState> domainStates = turboDomainsStateCHDao.getDomainsState(domains);
        nanos2 = System.nanoTime();
        state.addWatchNs(WatchType.DOMAINS_STATE, nanos2 - nanos1);
        Map<String, List<TurboFeedSettings>> feedsMap = turboFeedsService.getFeeds(domains).stream()
                .collect(Collectors.groupingBy(TurboFeedSettings::getDomain));
        nanos1 = System.nanoTime();
        state.addWatchNs(WatchType.FEEDS, nanos1 - nanos2);
        nanos2 = System.nanoTime();
        Map<String, TurboDesktopSettings> desktopSettingsMap = turboDesktopSettingsYDao.getSettings(domains);
        nanos1 = System.nanoTime();
        state.addWatchNs(WatchType.DESKTOP, nanos1 - nanos2);
        Map<String, TurboAppSettings> appSettingsMap = turboAppSettingsYDao.getSettings(domains);
        nanos2 = System.nanoTime();
        state.addWatchNs(WatchType.APP_SETTINGS, nanos2 - nanos1);
        Map<String, TurboPlusSettings> plusSettingsMap = turboPlusSettingsYDao.getSettings(domains);
        nanos1 = System.nanoTime();
        state.addWatchNs(WatchType.PLUS_SETTINGS, nanos1 - nanos2);


        for (Pair<String, TurboHostSettings> pair : batch) {
            var nanoStartRow = System.nanoTime();
            String domain = pair.getLeft();
            if (!hostWhiteList.isEmpty() && !hostWhiteList.contains(domain)) {
                continue;
            }
            WebmasterHostId hostId = IdUtils.urlToHostId(domain);
            TurboHostSettings settings = pair.getRight();
            final TurboAppSettings appSettings = appSettingsMap.get(domain);
            Optional<TurboDomainState> domainStateOpt = Optional.ofNullable(domainStates.get(domain));
            final TurboDomainState domainState = domainStateOpt.orElse(TurboDomainState.empty(domain));
            final TurboSccService.FrontModerationStatus wmcSccStatus =
                    TurboSccService.getFrontModerationStatus(settings, feedsMap.getOrDefault(domain, Collections.emptyList()));
            // грузим фиды
            List<TurboFeedSettings> feeds = feedsMap.getOrDefault(domain, Collections.emptyList())
                    .stream().map(feed ->
                            // фиды выключются при смене статуса, но на всякий случай если вдруг фиды не отключились
                            // отправляем в турбо выключенные
                            feed.getType().isYml() && wmcSccStatus == TurboSccService.FrontModerationStatus.BANNED ?
                                    feed.activate(false) : feed).toList();
            // оставим только провалидированные фиды
            List<YtTurboFeed> turboFeeds = feeds.stream().filter(feed -> feed.getState().isExportable())
                    .map(feed -> new YtTurboFeed(settings, feed)).collect(Collectors.toList());
            F.HOST.set(tw, IdUtils.hostIdToUrl(hostId));
            F.OWNER.set(tw, owners.get(hostId));
            F.FEEDS.set(tw, writeValueAsBytes(turboFeeds));
            F.ANALYTICS.set(tw, writeValueAsBytes(settings.getAnalyticsSettings().stream()
                    .map(AnalyticsSettings::toJson).filter(Objects::nonNull).collect(Collectors.toList())));
            F.ADVERTISING.set(tw, writeValueAsBytes(settings.getAdvertisingSettingsList().stream()
                    .map(AdvertisingSettings::toJson).filter(Objects::nonNull).collect(Collectors.toList())));
            TurboCommerceSettings commerceSettings = settings.getCommerceSettings();
            boolean hasYmlFeeds = feeds.stream().anyMatch(tfs -> tfs.getType() == TurboFeedType.YML) ||
                    (commerceSettings != null && commerceSettings.isOrderMethodSelected());

            boolean hasActiveYmlFeeds = feeds.stream().anyMatch(tfs -> tfs.getType() == TurboFeedType.YML && tfs.isActive() && tfs.isNotDeleted());
            List<TurboMenuItemData> menu = TurboMenuItemData.fromMenu(settings.getTopMenuSettings(), false);
            menu.addAll(TurboMenuItemData.fromMenu(settings.getMenuSettings(), !hasYmlFeeds));
            F.MENU.set(tw, writeValueAsBytes(menu));

            TurboHostSettingsData.FeedbackSettings feedbackSettings =
                    TurboHostSettingsData.FeedbackSettings.fromTurboFeedbackSettings(settings);
            F.FEEDBACK.set(tw, writeValueAsBytes(feedbackSettings));
            TurboHostHeader header = settings.getHeader();
            F.TITLE.set(tw, header.getTitle());
            TurboHostHeaderType enabledHeaderType = header.getType();
            F.HEADER_TYPE.set(tw, (long) enabledHeaderType.value());
            HeaderParamsInfo headerParamsInfo = HeaderParamsInfo.fromHeaderSettings(header);
            F.HEADER_PARAMS.set(tw, headerParamsInfo == null ? null : writeValueAsBytes(headerParamsInfo));
            if (enabledHeaderType != TurboHostHeaderType.NOLOGO) {
                Pair<String, Long> parsedLogo = TurboLogoData.parseFrontLogoId(header.getLogoInfo().getLogoId());
                F.LOGO_ID.set(tw, parsedLogo.getLeft());
                F.LOGO_GROUP_ID.set(tw, parsedLogo.getRight());
            }
            TurboHostSettingsData.Logo logo = TurboHostSettingsData.Logo.fromTurboLogo(header.getLogoInfo());
            F.LOGO.set(tw, logo == null ? null : writeValueAsBytes(logo));
            F.CSS.set(tw, ObjectUtils.firstNonNull(settings.getMinifiedCss(), settings.getCss()));
            F.AUTO_RELATED.set(tw, settings.getAutoRelated());
            F.AUTO_RELATED_SETTINGS.set(tw, TurboAutoRelatedSettingsData.fromSettings(settings.getAutoRelatedSettings()));
            F.AUTO_MORDA.set(tw, settings.getAutoMorda());
            Long marketPartnerId = domainStateOpt.flatMap(st -> st.getMarketFeeds().stream().map(TurboMarketFeed::getPartnerId).findAny()).orElse(null);
            F.PRODUCT_INFO.set(tw, productInfoFromSettings(domain, settings, marketPartnerId));
            F.DESKTOP.set(tw, TurboDesktopSettingsData.fromDesktopSettings(settings, desktopSettingsMap.get(domain)));
            F.EXPERIMENT.set(tw, Strings.emptyToNull(domainStateOpt.map(TurboDomainState::getExperiment).orElse(null)));
            F.AUTHORIZATION.set(tw, TurboAuthorizationSettingsData.fromSettings(settings.getAuthorizationSettings()));
            F.COMMENTS.set(tw, TurboCommentsSettingsData.fromSettings(settings.getCommentsSettings()));
            if (!hasYmlFeeds) {
                F.SEARCH.set(tw, TurboSearchSettingsData.fromSearchSettings(settings.getSearchSettings()));
            }
            List<TurboMenuItem> mergedAutoMenu = TurboMenuItem.mergeUserAndAutoMenu(settings.getAutoMenuSettings(),
                    domainStateOpt.map(TurboDomainState::getCommerceCategories).orElse(null), true);
            F.CATEGORY_INFO.set(tw, TurboMenuItemData.fromMenu(mergedAutoMenu, true));
            if (hasYmlFeeds) {
                state.addStatus(wmcSccStatus);
            }
            if (hasActiveYmlFeeds) {
                state.addStatusWithActiveYml(wmcSccStatus);
            }
            // TurboApp
            if (appSettings != null) {
                F.TURBO_APP.set(tw, TurboAppSettingsData.fromAppSettings(appSettings,
                        domainState.getAppReviewsInfoOrDefault(),
                        settings, feeds));
                F.DESCRIPTION.set(tw, appSettings.getDescription());
                F.LOGO_SQUARE.set(tw, TurboHostSettingsData.Logo.fromTurboLogo(appSettings.getLogoInfo()));
                F.THEME.set(tw, appSettings.getTurboColorsString());
                boolean turboCartOn = commerceSettings != null && !commerceSettings.isCartUrlEnabled() && (commerceSettings.isCheckoutEmailEnabled() || commerceSettings.isTurboCartEnabled());
                if (appSettings.getSccCheckStartDate() != null && turboCartOn && hasActiveYmlFeeds && domainState.getShopState().isReady()) {
                    F.SCC_CHECK_START_DATE.set(tw, appSettings.getSccCheckStartDate().getMillis());
                }
            } else if (hasYmlFeeds) {
                // default logo
                nanos1 = System.nanoTime();
                F.LOGO_SQUARE.set(tw, TurboHostSettingsData.Logo.fromTurboLogo(defaultTurboLogoYDao.getLogo(IDN.toUnicode(domain).substring(0, 1))));
                nanos2 = System.nanoTime();
                state.addWatchNs(WatchType.LOGO, nanos2 - nanos1);
            }
            // TurboPlus
            F.TURBO_PLUS.set(tw, TurboPlusSettingsData.fromPlusSettings(plusSettingsMap.get(domain)));
            F.ECOMMERCE.set(tw, NewCommerceSettingsData.fromInternalSettings(settings, appSettings));
            F.DISPLAY_PARAMS.set(tw, OM.valueToTree(TurboDisplaySettingsData.fromSettings(settings, desktopSettingsMap.get(domain))));
            F.MARKET_INFO.set(tw, MarketInfo.fromTurboSettings(settings));
            tw.rowEnd();
            state.turboHostsCount++;
            if (turboFeeds.stream().anyMatch(feed -> feed.getState() == YtTurboFeedState.ACTIVE)) {
                state.turboActiveHostsCount++;
            }
            state.totalTurboFeedsCount += turboFeeds.size();
            turboFeeds.forEach(turboFeed ->
                    state.turboFeedsCount.compute(turboFeed.getState(), (k, v) -> v == null ? 1 : v + 1));

            state.addWatchNs(WatchType.CREATE_ROW, System.nanoTime() - nanoStartRow);
        }
    }

    public void setHostWhiteList(String whiteList) {
        if (!Strings.isNullOrEmpty(whiteList)) {
            hostWhiteList = Arrays.stream(whiteList.split(",")).map(StringUtils::trimToEmpty)
                    .collect(Collectors.toSet());
        }
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_TURBO_FEEDS;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 3/30 * * * *");
    }

    private enum YtTurboFeedState {
        ACTIVE,
        INACTIVE,
        DELETED
    }

    @Getter
    @AllArgsConstructor
    @With
    private static class YtTurboFeed {

        private final String url;
        private final YtTurboFeedState state;
        private final TurboFeedType type;
        private final String login;
        private final String password;
        private final long timestamp;

        public YtTurboFeed(TurboHostSettings hostSettings, TurboFeedSettings feedSettings) {
            url = feedSettings.getUrl();
            if (feedSettings.getState() == TurboFeedState.DELETED) {

                state = YtTurboFeedState.DELETED;
            } else {
                state = feedSettings.isActive() ? YtTurboFeedState.ACTIVE : YtTurboFeedState.INACTIVE;
            }
            this.type = feedSettings.getType();
            timestamp = feedSettings.getAddDate() == null ? 0L : feedSettings.getAddDate().getMillis();
            TurboAccessSettings accessSettings = hostSettings.getAccessSettings();
            if (accessSettings != null && !Strings.isNullOrEmpty(accessSettings.getLogin()) &&
                    !Strings.isNullOrEmpty(accessSettings.getPassword())) {
                login = accessSettings.getLogin();
                password = accessSettings.getPassword();
            } else {
                login = password = null;
            }
        }
    }

    public static final class YtProductInfo extends ProductInfo {

        private final JsonNode accordion;

        public YtProductInfo(String cartUrl, JsonNode accordion, OrderInfo orderInfo,
                             boolean checkoutEmailEnabled, boolean turboCartEnabled,
                             boolean turboListingsEnabled,
                             boolean paymentEnabled, PaymentInfo paymentInfo, ShopInfo shopInfo,
                             DeliveryInfo deliveryInfo, boolean turboMainPageEnabled, Long marketPartnerId, boolean marketDeliveryEnabled) {
            super(cartUrl, null, orderInfo, checkoutEmailEnabled, turboCartEnabled, turboListingsEnabled,
                    paymentEnabled, paymentInfo, shopInfo, deliveryInfo, turboMainPageEnabled, marketPartnerId, marketDeliveryEnabled);
            this.accordion = accordion;
        }

        @JsonProperty("accordion")
        public JsonNode getAccordion() {
            return accordion;
        }
    }

    private YtProductInfo productInfoFromSettings(String domain, TurboHostSettings settings, Long marketPartnerId) {
        TurboCommerceSettings commerce = settings.getCommerceSettings();
        if (commerce == null) {
            return null;
        }

        final boolean marketDeliveryEnabled = Optional.ofNullable(commerce.getDeliverySection())
                .map(DeliverySection::getMarketDeliveryEnabled)
                .map(marketDelivery -> marketDelivery && marketPartnerId != null)
                .orElse(false);

        String cartUrl = commerce.isCartUrlEnabled() ? commerce.getCartUrl() : null;
        TurboPaymentsSettings paymentsSettings = null;
        try {
            var nanoStart = System.nanoTime();
            paymentsSettings = turboParserService.validatePaymentsSettings(domain, commerce.getPaymentsSettings(), null);
            var nanoEnd = System.nanoTime();
            state.addWatchNs(WatchType.PAYMENTS, nanoEnd - nanoStart);
        } catch (Exception e) {
            log.warn("Error when validating payments settings for domain " + domain, e);
        }
        Pair<Boolean, PaymentInfo> pair = PaymentInfo.fromPaymentsSettings(paymentsSettings);
        ShopInfo shopInfo = ShopInfo.fromPaymentsSettings(paymentsSettings);
        // WMC-10124: при отключенной турбо-корзине не должно быть включено ничего лишнего
        boolean turboCartOn = commerce.isCheckoutEmailEnabled() || commerce.isTurboCartEnabled();
        return new YtProductInfo(
                cartUrl,
                commerce.getParsedAccordion(),
                OrderInfo.fromSettings(settings),
                commerce.isCheckoutEmailEnabled(),
                commerce.isTurboCartEnabled(),
                commerce.isTurboListingsEnabled(),
                turboCartOn ? pair.getLeft() : false,
                turboCartOn ? pair.getRight() : null,
                turboCartOn ? shopInfo : null,
                turboCartOn ? DeliveryInfo.create(commerce.getDeliverySection()) : null,
                Boolean.TRUE.equals(commerce.getAutoMorda()),
                marketPartnerId,
                marketDeliveryEnabled);
    }

    private enum WatchType {
        OWNERS,
        FEEDS,
        CREATE_ROW,
        PAYMENTS,
        EXPORT_DATA_TO_YT,
        DOMAINS_STATE,
        DESKTOP,
        APP_SETTINGS,
        LOGO,
        PLUS_SETTINGS,
        FILTER_UNVERIFIED_HOSTS
    }

    @Getter
    public static class TaskState implements PeriodicTaskState {

        private long taskStart;
        private long turboHostsCount;
        private long turboActiveHostsCount;
        private long totalTurboFeedsCount;
        private EnumMap<YtTurboFeedState, Long> turboFeedsCount = new EnumMap<>(YtTurboFeedState.class);
        private List<YtPath> tablesSuccess = new CopyOnWriteArrayList<>();
        private List<YtPath> tablesFailed = new CopyOnWriteArrayList<>();
        private Map<WatchType, Long> watchNs = EnumSet.allOf(WatchType.class).stream()
                .collect(W3Collectors.toEnumMap(Function.identity(), ign -> 0L, WatchType.class));

        private Map<TurboSccService.FrontModerationStatus, Long> sccStatuses = EnumSet.allOf(TurboSccService.FrontModerationStatus.class).stream()
                .collect(W3Collectors.toEnumMap(Function.identity(), ign -> 0L, TurboSccService.FrontModerationStatus.class));

        private Map<TurboSccService.FrontModerationStatus, Long> sccStatusesWithActiveYml = EnumSet.allOf(TurboSccService.FrontModerationStatus.class).stream()
                .collect(W3Collectors.toEnumMap(Function.identity(), ign -> 0L, TurboSccService.FrontModerationStatus.class));

        // чтобы не смотреть на "голые" наносекунды
        public Map<WatchType, String> getWatchNs() {
            return watchNs.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, x -> Duration.ofNanos(x.getValue()).toString()));
        }

        @SuppressWarnings("ConstantConditions")
        public void addWatchNs(WatchType key, long time) {
            watchNs.compute(key, (k, v) -> v + time);
        }

        public void addStatus(TurboSccService.FrontModerationStatus key) {
            sccStatuses.merge(key, 1L, Long::sum);
        }
        public void addStatusWithActiveYml(TurboSccService.FrontModerationStatus key) {
            sccStatusesWithActiveYml.merge(key, 1L, Long::sum);
        }
    }
}
