package ru.yandex.webmaster3.worker.url.checker3;

import NUrlChecker.Response;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.mutable.MutableObject;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Seconds;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckRequestParams;
import ru.yandex.webmaster3.core.util.RetryUtils;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockState;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.service.UrlCheckDataBlocksService;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtRowMapper;
import ru.yandex.webmaster3.storage.util.yt.YtService;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;


/**
 * @author leonidrom
 */
@Slf4j
public abstract class AbstractUrlCheckDataBlockFetcher<T> {
    private static final RetryUtils.RetryPolicy RETRY_POLICY = RetryUtils.instantRetry(3);

    @Autowired
    protected UrlCheckDataBlocksService urlCheckDataBlocksService;
    @Autowired
    protected YtService ytService;
    protected final ExecutorService executorService = Executors.newCachedThreadPool();

    public boolean fetchBlock(UUID requestId, DateTime searchBaseDate, UrlCheckRequestParams requestParams) {
        var block = getBlockToFetch(requestId);
        if (block == null) {
            return false;
        }

        // получим данные блока
        var blockType = getBlockType();
        T blockData;
        try {
            log.info("Block fetch started: {}, {}", requestId, blockType);
            blockData = RetryUtils.query(RETRY_POLICY, () -> doFetchBlock(searchBaseDate, requestParams));
            var fetchTime = Seconds.secondsBetween(block.getAddDate(), DateTime.now());
            log.info("Block fetch finished: {}, {}, fetch time {}s", requestId, blockType, fetchTime);
        } catch (Exception e) {
            urlCheckDataBlocksService.closeRequestInternalError(requestId, blockType, e);
            return false;
        }

        return storeBlockData(block, blockData);
    }

    public boolean fetchBlock(UUID requestId, Response.TUrlCheckResponse response) {
        var block = getBlockToFetch(requestId);
        if (block == null) {
            return false;
        }

        // получим данные блока
        var blockType = getBlockType();
        T blockData;
        try {
            log.info("Block fetch started: {}, {}", requestId, blockType);
            blockData = RetryUtils.query(RETRY_POLICY, () -> doFetchBlock(response));
            var fetchTime = Seconds.secondsBetween(block.getAddDate(), DateTime.now());
            log.info("Block fetch finished: {}, {}, fetch time {}s", requestId, blockType, fetchTime);
        } catch (Exception e) {
            urlCheckDataBlocksService.closeRequestInternalError(requestId, blockType, e);
            return false;
        }

        return storeBlockData(block, blockData);
    }

    public boolean storeNullBlockData(UUID requestId) {
        var block = getBlockToFetch(requestId);
        if (block == null) {
            return false;
        }

        T blockData = null;
        try {
            blockData = getDataBlockClass().getDeclaredConstructor().newInstance();
        } catch (Exception e) {
            // ignore
            return false;
        }

        return storeBlockData(block, blockData);
    }


    private UrlCheckDataBlock getBlockToFetch(UUID requestId) {
        var blockType = getBlockType();
        log.info("Block to fetch: {}, {}", requestId, blockType);

        // получим запрос про блок из базы
        var block = urlCheckDataBlocksService.getWithoutData(requestId, blockType);
        if (block == null) {
            log.error("Block fetch request not found: {}, {}", requestId, blockType);
            return null;
        }

        // проверим не завершен ли он уже
        if (block.getFetchState().isTerminal()) {
            log.error("Block fetch request is already completed: {}, {}, {}", requestId, blockType, block.getFetchState());
            return null;
        }

        return block;
    }

    private boolean storeBlockData(UrlCheckDataBlock block, T blockData) {
        if (blockData == null) {
            return false;
        }

        var requestId = block.getRequestId();
        var blockType = block.getBlockType();

        // проверим не случился ли таймаут пока доставались данные
        if (block.checkTimedOut()) {
            log.error("Block fetch timeout: {}, {}", requestId, blockType);
            urlCheckDataBlocksService.closeRequest(requestId, blockType, UrlCheckDataBlockState.TIMED_OUT);
            return false;
        }

        // положим данные блока в базу
        String blockDataStr = JsonMapping.writeValueAsString(blockData);
        urlCheckDataBlocksService.closeRequest(requestId, blockType, blockDataStr);

        return true;
    }

    /**
     * Параллельно читает данные по ключу (Host, Path) из копий таблиц
     * на хане и арнольде и возвращает наиболее быстрый ответ.
     */
    protected <RowT> List<RowT> readTablesParallel(String tablePath, String url, Class<RowT> rowClass) throws Exception {
        YtPath arnoldTablePath = YtPath.create("arnold", tablePath);
        YtPath hahnTablePath = YtPath.create("hahn", tablePath);

        Callable<List<RowT>> arnoldCallable = () -> readTable(arnoldTablePath, url, rowClass);
        Callable<List<RowT>> hahnCallable = () -> readTable(hahnTablePath, url, rowClass);
        var res = executorService.invokeAny(List.of(arnoldCallable, hahnCallable));
        log.info("Got result for {}", tablePath);

        return res;
    }

    /**
     * Читает данные по ключу (Host, Path) из таблицы на Yt
     */
    protected  <RowT> List<RowT> readTable(YtPath tablePath, String url, Class<RowT> rowClass) throws Exception {
        log.info("Started read table: {} for url {}", tablePath.toYqlPath(), url);

        Pair<String, String> hostAndPath = splitUrlToHostAndPath(url);
        List<Object> rowKeys = List.of(hostAndPath.getLeft(), hostAndPath.getRight());
        MutableObject<List<RowT>> rowsObj = new MutableObject<>();
        ytService.withoutTransaction(cypressService -> {
            rowsObj.setValue(cypressService.getRows(tablePath, rowKeys, rowClass));
            return true;
        });

        log.info("Finished read table: {}", tablePath.toYqlPath());

        return rowsObj.getValue();
    }

    protected  <RowT> List<RowT> readTable(YtPath tablePath, String url, YtRowMapper<RowT> rowMapper) throws Exception {
        log.info("Started read table: {} for url {}", tablePath.toYqlPath(), url);

        Pair<String, String> hostAndPath = splitUrlToHostAndPath(url);
        List<Object> rowKeys = List.of(hostAndPath.getLeft(), hostAndPath.getRight());
        MutableObject<List<RowT>> rowsObj = new MutableObject<>();
        ytService.withoutTransaction(cypressService -> {
            rowsObj.setValue(cypressService.getRows(tablePath, rowKeys, rowMapper));
            return true;
        });

        log.info("Finished read table: {}", tablePath.toYqlPath());

        return rowsObj.getValue();
    }

    /**
     * Разбивает урл на хостовую (схема, хост, порт) и остальную часть
     */
    protected Pair<String, String> splitUrlToHostAndPath(String urlS) throws MalformedURLException {
        URL url = new URL(urlS);

        StringBuilder hostB = new StringBuilder(url.getProtocol());
        hostB.append("://").append(url.getHost());
        int port = url.getPort();
        if (port != -1) {
            hostB.append(port);
        }
        String host = hostB.toString();

        StringBuilder pathB = new StringBuilder(url.getPath());
        if (url.getQuery() != null) {
            pathB.append('?').append(url.getQuery());
        }

        String path = pathB.toString();
        if (StringUtils.isEmpty(path)) {
            path = "/";
        }

        return Pair.of(host, path);
    }

    public abstract T doFetchBlock(DateTime searchBaseDate, UrlCheckRequestParams requestParams) throws Exception;
    public abstract T doFetchBlock(Response.TUrlCheckResponse response) throws Exception;
    public abstract Class<T> getDataBlockClass();

    @NotNull
    public abstract UrlCheckDataBlockType getBlockType();
}
