package ru.yandex.webmaster3.worker.url.checker3;

import NUrlChecker.Response;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.logbroker.reader.IDataProcessing;
import ru.yandex.webmaster3.core.logbroker.reader.MessageContainer;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.service.UrlCheckDataBlocksService;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.*;

import static NUrlChecker.Response.TUrlCheckResponse.EStatusCode;

/**
 * @author leonidrom
 *
 * Вычитывает из LB ответы Роботного сервиса
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class UrCheckResultsProcessingService implements IDataProcessing {
    private final UrlCheckDataBlockFetchersRegistry urlCheckDataBlockFetchersRegistry;
    private final UrlCheckDataBlocksService urlCheckDataBlocksService;

    @Override
    public void process(MessageContainer messageContainer) {
        log.info("Got {} messages", messageContainer.getRawMessages().size());
        for (byte[] rawMessage: messageContainer.getRawMessages()) {
            Response.TUrlCheckResponse response;
            try {
                response = Response.TUrlCheckResponse.parseFrom(new ByteArrayInputStream(rawMessage));
            } catch (IOException e) {
                log.error("Failed to parse message", e);
                continue;
            }

            UUID requestId;
            try {
                requestId = UUID.fromString(response.getRequestId());
            } catch (Exception e) {
                log.error("Failed to parse requestId as UUID: {}", response.getRequestId(), e);
                continue;
            }

            var requestedBlockTypes = urlCheckDataBlocksService.getWithoutData(requestId).stream()
                    .map(UrlCheckDataBlock::getBlockType).toList();
            if (requestedBlockTypes.isEmpty()) {
                // не наш request id
                continue;
            }

            var failedBlockTypesMap = getFailedBlockTypesMap(response);
            var toFailBlockTypes = failedBlockTypesMap.values().stream()
                    .flatMap(List::stream)
                    .filter(requestedBlockTypes::contains)
                    .toList();

            for (var e: failedBlockTypesMap.entrySet()) {
                e.getValue().forEach(blockType -> {
                    EStatusCode statusCode = e.getKey();
                    urlCheckDataBlocksService.closeRequestRobotInternalError(requestId, blockType, statusCode);
                });
            }

            var nullBlockTypes = getNullBlockTypes(response);
            var toNullBlockTypes = nullBlockTypes.stream().filter(requestedBlockTypes::contains).toList();
            for (var blockType: toNullBlockTypes) {
                var fetcher = urlCheckDataBlockFetchersRegistry.getFetcher(blockType);
                if (fetcher != null) {
                    fetcher.storeNullBlockData(requestId);
                }
            }

            log.info("Got response for request {}, failed blocks: {}, null blocks: {}",
                    requestId, Arrays.toString(toFailBlockTypes.toArray()), Arrays.toString(toNullBlockTypes.toArray()));

            var presentBlockTypes = getPresentBlockTypes(response);
            var toFetchBlockTypes = presentBlockTypes.stream().filter(requestedBlockTypes::contains).toList();
            log.info("Got response for request {}, present blocks: {}", requestId, Arrays.toString(toFetchBlockTypes.toArray()));

            for (var blockType: toFetchBlockTypes) {
                var fetcher = urlCheckDataBlockFetchersRegistry.getFetcher(blockType);
                if (fetcher == null) {
                    log.error("No fetcher found for block {}", blockType);
                    continue;
                }

                fetcher.fetchBlock(requestId, response);
            }
        }

        log.info("All messages processed");
        messageContainer.commit();
    }

    private List<UrlCheckDataBlockType> getPresentBlockTypes(Response.TUrlCheckResponse response) {
        List<UrlCheckDataBlockType> res = new ArrayList<>();

        if (response.hasRotorfullScreenshot()) {
            res.add(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_ON);
        }

        if (response.hasRotorlessScreenshot()) {
            res.add(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_OFF);
        }

        if (response.hasRotorfullFetchResult()) {
            res.add(UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_ON);
        }

        if (response.hasRotorfullArchive()) {
            res.add(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON);
        }

        if (response.hasRotorlessFetchResult()) {
            res.add(UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_OFF);
        }

        if (response.hasRotorlessArchive()) {
            res.add(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_OFF);
        }

        if (response.hasMobileCrawlResult() || response.hasDesktopCrawlResult()) {
            res.add(UrlCheckDataBlockType.ROBOT_INDEXING_INFO);
        }

        return res;
    }

    private Map<EStatusCode, List<UrlCheckDataBlockType>> getFailedBlockTypesMap(Response.TUrlCheckResponse response) {
        Map<EStatusCode, List<UrlCheckDataBlockType>> res = new HashMap<>();
        for (var statusCode: response.getStatusList()) {
            var blockTypes = getFailedBlockTypeFromStatusCode(statusCode);
            if (blockTypes.isEmpty()) {
                continue;
            }

            res.put(statusCode, blockTypes);
        }

        return res;
    }

    @NotNull
    private static List<UrlCheckDataBlockType> getFailedBlockTypeFromStatusCode(EStatusCode statusCode) {
        return switch (statusCode) {
            case ESC_ROTORLESS_FETCH_REQUEST_FAILED -> List.of(
                    UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_OFF,
                    UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_OFF);
            case ESC_ROTORLESS_ARCHIVE_REQUEST_FAILED -> List.of(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_OFF);
            case ESC_ROTORFULL_FETCH_REQUEST_FAILED -> List.of(
                    UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_ON,
                    UrlCheckDataBlockType.ROBOT_SERVER_RESPONSE_RENDER_ON);
            case ESC_ROTORFULL_ARCHIVE_REQUEST_FAILED -> List.of(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON);
            default -> Collections.emptyList();
        };
    }

    private List<UrlCheckDataBlockType> getNullBlockTypes(Response.TUrlCheckResponse response) {
        return response.getStatusList().stream()
                .map(UrCheckResultsProcessingService::getNullBlockTypeFromStatusCode)
                .flatMap(List::stream)
                .toList();
    }

    @NotNull
    private static List<UrlCheckDataBlockType> getNullBlockTypeFromStatusCode(EStatusCode statusCode) {
        return switch (statusCode) {
            case ESC_NULL_CRAWL_INFO_RESULT -> List.of(UrlCheckDataBlockType.ROBOT_INDEXING_INFO);
            case ESC_NULL_ROTORLESS_SCREENSHOT_RESULT -> List.of(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_OFF);
            case ESC_NULL_ROTORLESS_ARCHIVE_RESULT -> List.of(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_OFF);
            case ESC_NULL_ROTORFULL_SCREENSHOT_RESULT -> List.of(UrlCheckDataBlockType.ROBOT_ROTOR_CHECK_RENDER_ON);
            case ESC_NULL_ROTORFULL_ARCHIVE_RESULT -> List.of(UrlCheckDataBlockType.ROBOT_ARCHIVE_RENDER_ON);
            default -> Collections.emptyList();
        };
    }
}
