package ru.yandex.webmaster3.worker.url.checker3.fetchers;

import NUrlChecker.Response;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.sitestructure.RawSearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusEnum;
import ru.yandex.webmaster3.core.sitestructure.SearchUrlStatusUtil;
import ru.yandex.webmaster3.core.url.checker3.UrlCheckRequestParams;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.jupiter.JupiterUtils;
import ru.yandex.webmaster3.storage.searchurl.SearchUrlSamplesService;
import ru.yandex.webmaster3.storage.searchurl.samples.data.UrlStatusInfo;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlock;
import ru.yandex.webmaster3.storage.url.checker3.data.UrlCheckDataBlockType;
import ru.yandex.webmaster3.storage.url.checker3.data.blocks.SearchInfoData;
import ru.yandex.webmaster3.worker.url.checker3.AbstractUrlCheckDataBlockFetcher;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author leonidrom
 *
 * Ходит в urls_for_webmaster_simple и content_attrs
 */
@Component
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class SearchInfoDataFetcher extends AbstractUrlCheckDataBlockFetcher<SearchInfoData> {
    private final String JUPITER_TABLE_PATH_PATTERN = "//home/jupiter/acceptance/%s/urls_for_webmaster_simple";
    private final String URLS_CONTENT_ATTRS_TABLE_PATH_PATTERN = "//home/jupiter/export/%s/webmaster/content_attrs";

    @Override
    public SearchInfoData doFetchBlock(DateTime searchBaseDate, UrlCheckRequestParams requestParams) throws Exception {
        var jupiterRowsFuture = executorService.submit(() -> readJupiterTable(searchBaseDate, requestParams));
        var contentAttrsRowsFuture = executorService.submit(() -> readContentAttrsTable(searchBaseDate, requestParams));

        List<YtJupiterRow> jupiterRows = jupiterRowsFuture.get(UrlCheckDataBlock.FETCH_TIMEOUT.getSeconds(), TimeUnit.SECONDS);
        YtJupiterRow jupiterRow = jupiterRows.isEmpty()? null : jupiterRows.get(0);

        List<YtContentAttrsRow> contentAttrRows = contentAttrsRowsFuture.get(UrlCheckDataBlock.FETCH_TIMEOUT.getSeconds(), TimeUnit.SECONDS);
        YtContentAttrsRow contentAttrsRow = contentAttrRows.isEmpty()? null : contentAttrRows.get(0);

        return convertYtRows(jupiterRow, contentAttrsRow);
    }

    @Override
    public SearchInfoData doFetchBlock(Response.TUrlCheckResponse response) throws Exception {
        return null;
    }

    @Override
    public Class<SearchInfoData> getDataBlockClass() {
        return SearchInfoData.class;
    }

    private List<YtJupiterRow> readJupiterTable(DateTime searchBaseDate, UrlCheckRequestParams params) throws Exception {
        String tablePath = JUPITER_TABLE_PATH_PATTERN.formatted(searchBaseDate.toString(JupiterUtils.STATE_DATE_FORMAT));
        return readTablesParallel(tablePath, params.getUrl(), YtJupiterRow.class);
    }

    private List<YtContentAttrsRow> readContentAttrsTable(DateTime searchBaseDate, UrlCheckRequestParams params) throws Exception {
        String tablePath = URLS_CONTENT_ATTRS_TABLE_PATH_PATTERN.formatted(searchBaseDate.toString(JupiterUtils.STATE_DATE_FORMAT));
        return readTablesParallel(tablePath, params.getUrl(), YtContentAttrsRow.class);
    }


    private SearchInfoData convertYtRows(YtJupiterRow jupiterRow, YtContentAttrsRow contentAttrsRow) {
        if (jupiterRow == null && contentAttrsRow == null) {
            return new SearchInfoData(null, null, null);
        }

        String title = null;
        String description = null;
        if (contentAttrsRow != null) {
            title = contentAttrsRow.title;
            description = contentAttrsRow.description;
        }

        SearchInfoData.SearchInfo searchInfo = null;
        if (jupiterRow != null) {
            if (jupiterRow.isFake) {
                searchInfo = SearchInfoData.FAKE;
            } else {
                // в некоторых случаях, например для страниц запрещенных к индексированию
                // посредством robots.txt, http код из Yt может быть null
                int httpCode = jupiterRow.httpCode == null? 0 : jupiterRow.httpCode;

                DateTime addTime = null;
                if (jupiterRow.addTime != null) {
                    addTime = TimeUtils.unixTimestampToDate(jupiterRow.addTime, TimeUtils.EUROPE_MOSCOW_ZONE);
                }

                DateTime lastAccess = null;
                if (jupiterRow.lastAccess != null) {
                    lastAccess = TimeUtils.unixTimestampToDate(jupiterRow.lastAccess, TimeUtils.EUROPE_MOSCOW_ZONE);
                }

                RawSearchUrlStatusEnum rawStatus = getRawSearchUrlStatusEnum(jupiterRow.urlStatus);
                title = SearchUrlSamplesService.hideDownloadEvidence(rawStatus, title);
                SearchUrlStatusEnum status = SearchUrlStatusUtil.raw2View(rawStatus, jupiterRow.isSearchable);

                UrlStatusInfo statusInfo = new UrlStatusInfo(status, addTime, jupiterRow.beautyUrl, httpCode,
                        jupiterRow.mainHost, jupiterRow.mainPath, jupiterRow.redirTarget, jupiterRow.relCanonicalTarget, description,
                        false, false, false, false, false);

                searchInfo = new SearchInfoData.SearchInfo(httpCode, lastAccess, statusInfo, false, jupiterRow.isProcessedByRotor);
            }
        }

        return new SearchInfoData(title, description, searchInfo);
    }

    private static RawSearchUrlStatusEnum getRawSearchUrlStatusEnum(int statusInt) {
        RawSearchUrlStatusEnum status = RawSearchUrlStatusEnum.R.fromValueOrNull(statusInt);
        if (status == null) {
            status = RawSearchUrlStatusEnum.OTHER;
        }

        return status;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static final class YtJupiterRow {
        private final String host;
        private final String path;
        private final int urlStatus;
        private final String beautyUrl;
        private final Integer lastAccess;
        private final Integer addTime;
        private final String mainHost;
        private final String mainPath;
        private final String redirTarget;
        private final Integer httpCode;
        private final String relCanonicalTarget;
        private final boolean isFake;
        private final boolean isSearchable;
        private final boolean isProcessedByRotor;


        @JsonCreator
        public YtJupiterRow(
                @JsonProperty("Host") String host,
                @JsonProperty("Path") String path,
                @JsonProperty("UrlStatus") int urlStatus,
                @JsonProperty("BeautyUrl") String beautyUrl,
                @JsonProperty("LastAccess") Integer lastAccess,
                @JsonProperty("AddTime") Integer addTime,
                @JsonProperty("MainHost") String mainHost,
                @JsonProperty("MainPath") String mainPath,
                @JsonProperty("RedirTarget") String redirTarget,
                @JsonProperty("HttpCode") Integer httpCode,
                @JsonProperty("RelCanonicalTarget") String relCanonicalTarget,
                @JsonProperty("IsFake") boolean isFake,
                @JsonProperty("IsSearchable") boolean isSearchable,
                @JsonProperty("IsProcessedByRotor") boolean isProcessedByRotor) {
            this.host = host;
            this.path = path;
            this.urlStatus = urlStatus;
            this.beautyUrl = beautyUrl;
            this.lastAccess = lastAccess;
            this.addTime = addTime;
            this.mainHost = mainHost;
            this.mainPath = mainPath;
            this.redirTarget = redirTarget;
            this.httpCode = httpCode;
            this.relCanonicalTarget = relCanonicalTarget;
            this.isFake = isFake;
            this.isSearchable = isSearchable;
            this.isProcessedByRotor = isProcessedByRotor;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static final class YtContentAttrsRow {
        @JsonProperty("Host") private final String host;
        @JsonProperty("Path") private final String path;
        @JsonProperty("TitleRawUTF8") private final String title;
        @JsonProperty("MetaDescription") private final String description;

        @JsonCreator
        private YtContentAttrsRow(
                String host,
                String path,
                String title,
                String description) {
            this.host = host;
            this.path = path;
            this.title = title;
            this.description = description;
        }
    }

    @Override
    public @NotNull UrlCheckDataBlockType getBlockType() {
        return UrlCheckDataBlockType.SEARCH_INFO;
    }
}
