package ru.yandex.webmaster3.worker.user;

import java.util.Collections;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.storage.events.data.WMCEvent;
import ru.yandex.webmaster3.storage.events.data.WMCEventContent;
import ru.yandex.webmaster3.storage.events.data.events.DelegationToUserEvent;
import ru.yandex.webmaster3.storage.events.service.WMCEventsObserver;
import ru.yandex.webmaster3.storage.notifications.NotificationChannel;
import ru.yandex.webmaster3.storage.notifications.UserNotificationConfiguration;
import ru.yandex.webmaster3.storage.notifications.service.UserNotificationSettingsService;
import ru.yandex.webmaster3.storage.user.UserPersonalInfo;
import ru.yandex.webmaster3.storage.user.notification.NotificationType;
import ru.yandex.webmaster3.storage.user.service.UserPersonalInfoService;
import ru.yandex.webmaster3.worker.notifications.auto.AutoNotificationsSenderService;
import ru.yandex.webmaster3.worker.notifications.auto.NotificationInfo;

/**
 * Created by ifilippov5 on 20.02.18.
 */
public class DelegationToUserEventObserver implements WMCEventsObserver {
    private static final Logger log = LoggerFactory.getLogger(DelegationToUserEventObserver.class);

    private UserNotificationSettingsService userNotificationSettingsService;
    private NotificationChannel sendByChannel;
    private AutoNotificationsSenderService autoNotificationsSenderService;
    private UserPersonalInfoService userPersonalInfoService;

    @Override
    public boolean observe(WMCEvent event) {
        WMCEventContent content = event.getContent();
        if (!(content instanceof DelegationToUserEvent)) {
            return false;
        }

        DelegationToUserEvent delegationToUserEvent = (DelegationToUserEvent) content;
        UserNotificationConfiguration notificationSettings = userNotificationSettingsService
                .getUserNotificationsSettings(delegationToUserEvent.getUserId(), NotificationType.GLOBAL_NOTIFICATION);

        if (notificationSettings == null) {
            log.info("No notification settings found for user {}", delegationToUserEvent.getUserId());
            return false;
        }

        boolean sendNotification = notificationSettings.getChannels().contains(sendByChannel);
        if (!sendNotification) {
            return false;
        }

        long userId = delegationToUserEvent.getUserId();
        UserPersonalInfo personalInfo = getPersonalInfo(userId);
        if (personalInfo == null) {
            log.warn("Personal info for user {} not found - user may be missing", userId);
            return false;
        }

        String email = notificationSettings.getEmail();
        if (email == null && sendByChannel == NotificationChannel.EMAIL) {
            log.info("No email found for user {}", delegationToUserEvent.getUserId());
            return false;
        }
        final NotificationInfo notificationInfo = NotificationInfo.builder()
                .id(event.getId())
                .email(email)
                .userId(userId)
                .personalInfo(getPersonalInfo(userId))
                .messageContent(delegationToUserEvent.getMessageContent())
                .channel(sendByChannel)
                .critical(delegationToUserEvent.isCritical())
                .build();
        autoNotificationsSenderService.sendMessage(notificationInfo);

        return true;
    }

    private UserPersonalInfo getPersonalInfo(long userId) {
        Map<Long, UserPersonalInfo> personalInfos = userPersonalInfoService.getUsersPersonalInfos(
                Collections.singleton(userId)
        );
        return personalInfos.get(userId);
    }

    @Required
    public void setUserNotificationSettingsService(UserNotificationSettingsService userNotificationSettingsService) {
        this.userNotificationSettingsService = userNotificationSettingsService;
    }

    @Required
    public void setSendByChannel(NotificationChannel sendByChannel) {
        this.sendByChannel = sendByChannel;
    }

    @Required
    public void setAutoNotificationsSenderService(AutoNotificationsSenderService autoNotificationsSenderService) {
        this.autoNotificationsSenderService = autoNotificationsSenderService;
    }

    @Required
    public void setUserPersonalInfoService(UserPersonalInfoService userPersonalInfoService) {
        this.userPersonalInfoService = userPersonalInfoService;
    }
}
