package ru.yandex.webmaster3.worker.user;

import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.user.dao.DashboardSurveyAnswersYDao;
import ru.yandex.webmaster3.storage.util.yt.YtColumn;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtNodeAttributes;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtSchema;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.util.yt.YtTableData;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * ishalaru
 * 23.12.2020
 **/
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ExportDashboardSurveyTask extends PeriodicTask<ExportDashboardSurveyTask.State> {
    @Value("${external.yt.service.arnold.root.default}/export/dashboard_survey_answers")
    private YtPath path;
    private final YtService ytService;
    private final DashboardSurveyAnswersYDao dashboardSurveyAnswersYDao;

    interface F {
        YtSchema SCHEMA = new YtSchema();
        YtColumn<Long> USER_ID = SCHEMA.addColumn("user_id", YtColumn.Type.INT_64);
        YtColumn<String> HOST_ID = SCHEMA.addColumn("host_id", YtColumn.Type.STRING);
        YtColumn<String> SURVEY_TYPE = SCHEMA.addColumn("survey_type", YtColumn.Type.STRING);
        YtColumn<String> ANSWER_PLACE = SCHEMA.addColumn("answer_place", YtColumn.Type.STRING);
        YtColumn<List<String>> CASES = SCHEMA.addColumn("cases", YtColumn.Type.any());
        YtColumn<Long> UPDATED = SCHEMA.addColumn("updated", YtColumn.Type.INT_64);
    }

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new State());


        YtTableData tableData = ytService.prepareTableData("export-dashboard-survey", tableWriter -> {
                    dashboardSurveyAnswersYDao.scan(e -> {
                        F.USER_ID.set(tableWriter, e.getUserId());
                        if (e.getHostId() != null) {
                            F.HOST_ID.set(tableWriter, e.getHostId().toString());
                        }
                        F.SURVEY_TYPE.set(tableWriter, e.getSurveyType());
                        F.ANSWER_PLACE.set(tableWriter, e.getAnswerPlace());
                        F.CASES.set(tableWriter, e.getCases());
                        F.UPDATED.set(tableWriter, e.getUpdated().getMillis());
                        tableWriter.rowEnd();
                    });
                }
        );
        ytService.inTransaction(path).execute(cService -> {
            YtNodeAttributes attributes = new YtNodeAttributes().setSchema(F.SCHEMA);
            if (cService.exists(path)) {
                cService.remove(path);
            }

            cService.create(path, YtNode.NodeType.TABLE, true, attributes);
            cService.writeTable(path, tableData);

            return true;
        });
        return Result.SUCCESS;
    }


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.EXPORT_DASHBOARD_SURVEY;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 43 2 * * *");
    }

    public static class State implements PeriodicTaskState {
    }
}
