package ru.yandex.webmaster3.worker.user;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.user.dao.UserLastVisitYDao;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.io.*;
import java.util.UUID;

/**
 * @author leonidrom
 */
@Slf4j
@Component("uploadUserLastVisitTask")
public class UploadUserLastVisitTask extends PeriodicTask<PeriodicTaskState> {
    private static final String[] YT_TABLE_COLUMN_NAMES = {"user_id", "last_visit" };

    private final YtPath hahnExportPath;
    private final YtPath arnoldExportPath;
    private final YtService ytService;
    private final UserLastVisitYDao userLastVisitYDao;

    private final File tmpFile;

    @Autowired
    public UploadUserLastVisitTask(
            @Value("${webmaster3.worker.uploadUserLastVisit.hahn.export.archive.userLastVisit.path}") YtPath hahnExportPath,
            @Value("${webmaster3.worker.uploadUserLastVisit.arnold.export.archive.userLastVisit.path}") YtPath arnoldExportPath,
            YtService ytService,
            UserLastVisitYDao userLastVisitYDao,
            @Value("${webmaster3.worker.uploadUserLastVisit.userLastVisit.tmp.file}") File tmpFile) {
        this.hahnExportPath = hahnExportPath;
        this.arnoldExportPath = arnoldExportPath;
        this.ytService = ytService;
        this.userLastVisitYDao = userLastVisitYDao;
        this.tmpFile = tmpFile;
    }

    @Override
    public Result run(UUID runId) throws Exception {
        downloadDataFromCassandra();
        writeYtTable(hahnExportPath);
        writeYtTable(arnoldExportPath);

        return Result.SUCCESS;
    }

    private void writeYtTable(YtPath tablePath) {
        log.info("Started writing Yt table: {}", tablePath.toYtPath());

        YtPath tmpPath = YtPath.path(tablePath.getParent(), tablePath.getName() + "-" + System.currentTimeMillis());

        ytService.inTransaction(tablePath).execute(cypressService -> {
            cypressService.create(tmpPath, YtNode.NodeType.TABLE, true);
            cypressService.writeTable(tmpPath, tableWriter -> {
                try (BufferedReader br = new BufferedReader(new FileReader(tmpFile))) {
                    String line;
                    while ((line = br.readLine()) != null) {
                        var arr = line.split(",");
                        tableWriter.column("user_id", arr[0]);
                        tableWriter.column("last_visit", arr[1]);
                        tableWriter.rowEnd();
                    }
                } catch (IOException e) {
                    log.error("Failed to write Yt table", e);
                    throw new YtException("Failed to write Yt table", e);
                }
            });

            if (cypressService.exists(tablePath)) {
                cypressService.remove(tablePath);
            }

            cypressService.move(tmpPath, tablePath, true);

            return true;
        });

        log.info("Done writing Yt table: {}", tablePath.toYtPath());
    }

    private void downloadDataFromCassandra() throws IOException {
        if (tmpFile.exists()) {
            tmpFile.delete();
        }

        log.info("Started downloading data from Cassandra");
        try (PrintWriter pw = new PrintWriter(tmpFile)) {
            userLastVisitYDao.forEach(p -> {
                long userId = p.getLeft();
                Instant lastVisit = p.getRight();
                pw.println(userId + "," + lastVisit.getMillis() / 1000L);
            });
        } catch (IOException | WebmasterYdbException e) {
            log.error("Error writing tmp file", e);
            throw e;
        }

        log.info("Done downloading data from Cassandra");
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_USER_LAST_VISIT;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0 * * * *");
    }
}
