package ru.yandex.webmaster3.worker.user;

import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.user.takeout.UserTakeoutRequestType;
import ru.yandex.webmaster3.core.util.json.JsonMapping;
import ru.yandex.webmaster3.core.worker.client.WorkerClient;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.UserTakeoutTaskData;
import ru.yandex.webmaster3.core.worker.task.WorkerTaskData;
import ru.yandex.webmaster3.storage.postpone.PostponeActionYDao;
import ru.yandex.webmaster3.storage.postpone.PostponeOperationType;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequest;
import ru.yandex.webmaster3.storage.user.UserTakeoutRequestStatus;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutQueryData;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

/**
 * ishalaru
 * 25.01.2021
 **/
@Slf4j
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
@Component("userTakeoutCheckTask")
public class UserTakeoutCheckTask extends PeriodicTask<UserTakeoutCheckTask.TaskState> {
    private final WorkerClient workerClient;
    private final PostponeActionYDao postponeActionYDao;
    private final UserTakeoutService userTakeoutService;

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new TaskState());
        postponeActionYDao.loadPostponedData(postponeData -> {
            if (postponeData.getType() == PostponeOperationType.USER_TAKEOUT_GET
            || postponeData.getType() == PostponeOperationType.USER_TAKEOUT_DELETE) {
                UserTakeoutQueryData userTakeoutQueryData = JsonMapping.readValue(postponeData.getData(), UserTakeoutQueryData.class);
                UserTakeoutRequest request = userTakeoutService.getRequest(userTakeoutQueryData.getReqId());
                if (request.getStatus() != UserTakeoutRequestStatus.IN_PROGRESS) {
                    postponeActionYDao.delete(postponeData);
                } else if (postponeData.getDate().isBeforeNow()) {
                    Duration retryPeriod = null;
                    UserTakeoutRequestType requestType = null;
                    if (postponeData.getType() == PostponeOperationType.USER_TAKEOUT_GET) {
                        requestType = UserTakeoutRequestType.GET;
                        retryPeriod = UserTakeoutService.GET_RETRY_PERIOD;
                    // защита на случай появления других типов тейкаута
                    } else if (postponeData.getType() == PostponeOperationType.USER_TAKEOUT_DELETE) {
                        requestType = UserTakeoutRequestType.DELETE;
                        retryPeriod = UserTakeoutService.DELETE_RETRY_PERIOD;
                    }

                    if (retryPeriod != null) {
                        var taskData = new UserTakeoutTaskData(userTakeoutQueryData.getUid(), userTakeoutQueryData.getReqId(), requestType);
                        workerClient.checkedEnqueueTask(taskData);
                        postponeActionYDao.insert(DateTime.now().plus(retryPeriod), postponeData.getType(), postponeData.getData());
                        postponeActionYDao.delete(postponeData);
                    }
                }
            }
        });

        return Result.SUCCESS;
    }

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.USER_TAKEOUT_CHECK;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 0/25 * * * *");
    }

    public class TaskState implements PeriodicTaskState {
    }
}
