package ru.yandex.webmaster3.worker.user.takeout;

import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.scheme.SchemeOperationProtos;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.description.ListDirectoryResult;
import com.yandex.ydb.table.description.TableColumn;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.rpc.grpc.GrpcSchemeRpc;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.storage.user.service.UserTakeoutService;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import javax.annotation.PostConstruct;
import java.util.*;

/**
 * @author leonidrom
 *
 * Вычитывает схемы всех наших таблиц в Ydb и находит те, где есть поле user_id
 * и нет поддержки удаления пользовтельских данных.
 * В случае нахождения таких таблиц таска завершиться с ошибкой, а в unknownTables
 * будет список найденных таблиц.
 */
@Component
@Slf4j
public class ScrapeForTakeoutYdbTablesPeriodicTask extends PeriodicTask<ScrapeForTakeoutYdbTablesPeriodicTask.TaskState> {
    private static final Set<String> SKIP_TABLES = Set.of(
        "webmaster3/internal/user_takeout_requests",
        "webmaster3/internal/yt_user_data_delete_queue",

        // поля начинаются с "user", но не про user id
        "webmaster3/turbo/turbo_domain_settings",
        "webmaster3/region/host_regions_mod_reqs"
    );

    @Autowired
    private GrpcTransport ydbRpcTransport;

    @Autowired
    private SessionRetryContext ydbSessionRetryContext;

    @Autowired
    private UserTakeoutService userTakeoutService;

    @Value("${webmaster3.storage.ydb.dbname}")
    private String ydbDatabase;

    private SchemeClient schemeClient;
    private List<String> takeoutTables;


    @PostConstruct
    public void init() {
        schemeClient = SchemeClient.newClient(GrpcSchemeRpc.useTransport(ydbRpcTransport)).build();
        takeoutTables = userTakeoutService.getTakeoutTables();
        log.info("Takeout tables: {}", Arrays.toString(takeoutTables.toArray()));
    }

    @Override
    public Result run(UUID runId) throws Exception {
        setState(new TaskState());
        scrapeDir(ydbDatabase + "/webmaster3");
        if (state.unknownTables.isEmpty()) {
            return Result.SUCCESS;
        } else {
            return Result.FAIL;
        }
    }

    private void scrapeDir(String dirPath) {
        log.info("Scraping dir: " + dirPath);
        ListDirectoryResult dirList = schemeClient.listDirectory(dirPath)
                .join()
                .expect("cannot describe directory: " + dirPath);
        for (var e : dirList.getChildren()) {
            String path = dirPath + "/" + e.getName();
            if (e.getType() == SchemeOperationProtos.Entry.Type.TABLE) {
                scrapeTable(path);
            } else if (e.getType() == SchemeOperationProtos.Entry.Type.DIRECTORY) {
                scrapeDir(path);
            }
        }
    }

    private void scrapeTable(String tablePath) {
        log.info("Scraping table: " + tablePath);

        String tableName = tablePath.replaceFirst(ydbDatabase + "/", "");
        if (SKIP_TABLES.contains(tableName)) {
            log.info("Skipping table: " + tablePath);
            return;
        }

        TableDescription tableDesc = ydbSessionRetryContext.supplyResult(session -> session.describeTable(tablePath))
                .join()
                .expect("cannot describe table: " + tablePath);

        boolean isTakeoutTable = tableDesc.getColumns().stream().anyMatch(this::isTakeoutTable);
        if (!isTakeoutTable) {
            return;
        }

        log.info("Found takeout table: {}", tablePath);
        if (!takeoutTables.contains(tablePath)) {
            log.error("Found unknown takeout table: {}", tablePath);
            state.unknownTables.add(tablePath);
        }
    }

    private boolean isTakeoutTable(TableColumn col) {
        String colName = col.getName().toLowerCase();
        return colName.contains("user_id") || colName.contains("userid") || colName.contains("user");
    }


    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.SCRAPE_FOR_TAKEOUT_YDB_TABLES;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.never();
    }

    public static class TaskState implements PeriodicTaskState {
        public List<String> unknownTables = new ArrayList<>();
    }
}
