package ru.yandex.webmaster3.worker.util;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.data.HttpResponsePart;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.core.worker.task.util.ServerResponseUtilTaskData;
import ru.yandex.webmaster3.storage.async.model.HttpAsyncRequestInfoResultType;
import ru.yandex.webmaster3.storage.async.AsyncRequestTaskService;
import ru.yandex.webmaster3.storage.util.serverresponseutil.ServerResponseUtilService;
import ru.yandex.webmaster3.storage.util.serverresponseutil.exception.ServerResponseHttpUtilException;
import ru.yandex.webmaster3.storage.util.serverresponseutil.exception.ServerResponseUtilException;
import ru.yandex.webmaster3.storage.util.serverresponseutil.model.ServerResponseAnalysisErrorTransfer;
import ru.yandex.webmaster3.storage.util.serverresponseutil.model.ServerResponseAnalysisResponseType;
import ru.yandex.webmaster3.storage.util.serverresponseutil.model.ServerResponseUtilInputData;
import ru.yandex.webmaster3.worker.Task;
import ru.yandex.webmaster3.worker.queue.TaskQueueMetrics;

/**
 * @author: ishalaru
 * DATE: 20.05.2019
 */
@Slf4j
@Service
@RequiredArgsConstructor(onConstructor_ = {@Autowired})
public class ServerResponseAnalysisTask extends Task<ServerResponseUtilTaskData> {
    private static final int MAX_CONCURRENT_TASKS = 32;

    private final ServerResponseUtilService serverResponseUtilService;
    private final AsyncRequestTaskService asyncRequestTaskService;

    @Override
    public Result run(ServerResponseUtilTaskData data) throws Exception {
        asyncRequestTaskService.startWorkingTask(data.getRequestId());
        try {
            ServerResponseUtilInputData work = new ServerResponseUtilInputData(data.getUrl(),
                    data.getUserIp(), data.getUserAgent(), data.getIfModifiedSince());
            HttpResponsePart result = serverResponseUtilService.process(work);
            asyncRequestTaskService.finishWorkingTask(data.getRequestId(), ServerResponseAnalysisResponseType.NORMAL.getValue(), result);
        } catch (ServerResponseHttpUtilException exp) {
            ServerResponseAnalysisErrorTransfer serverResponseAnalysisErrorTransfer = new ServerResponseAnalysisErrorTransfer(exp.getError(), exp.getHttpCodeInfo());
            asyncRequestTaskService.finishWorkingTask(data.getRequestId(), ServerResponseAnalysisResponseType.HTTP_PROBLEM.getValue(), serverResponseAnalysisErrorTransfer);
        } catch (ServerResponseUtilException exp) {
            asyncRequestTaskService.finishWorkingTask(data.getRequestId(), ServerResponseAnalysisResponseType.COMMON_PROBLEM.getValue(), exp.getError());
        } catch (Exception exp) {
            log.error("ServerResponseAnalysis task failed.", exp);
            asyncRequestTaskService.finishWorkingTask(data.getRequestId(), HttpAsyncRequestInfoResultType.INTERNAL_ERROR);
            return new Result(TaskResult.FAIL);
        }
        return new Result(TaskResult.SUCCESS);

    }

    @Override
    public Class<ServerResponseUtilTaskData> getDataClass() {
        return ServerResponseUtilTaskData.class;
    }

    @Override
    public float getPriority(TaskQueueMetrics.TaskStatistics stats) {
        if (stats.getProcessing() >= MAX_CONCURRENT_TASKS) {
            return 0;
        } else {
            return super.getPriority(stats);
        }
    }

}
