package ru.yandex.webmaster3.worker.video;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.util.ydb.exception.WebmasterYdbException;
import ru.yandex.webmaster3.storage.util.yt.*;
import ru.yandex.webmaster3.storage.video.VideohostOffer;
import ru.yandex.webmaster3.storage.video.VideohostOffersCurrentYDao;
import ru.yandex.webmaster3.storage.video.VideohostOffersHistoryYDao;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import java.util.UUID;

/**
 * @author vsedaikina
 * 18.10.21
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component
public class UploadVideohostOffersPeriodicTask extends PeriodicTask<UploadVideohostOffersPeriodicTask.TaskState> {

    @Value("${webmaster3.worker.video.upload.current.offers.path}")
    private final YtPath tableCurrentOffersPath;
    @Value("${webmaster3.worker.video.upload.history.offers.path}")
    private final YtPath tableHistoryOffersPath;

    private final YtService ytService;
    private final VideohostOffersCurrentYDao videohostOffersCurrentYDao;
    private final VideohostOffersHistoryYDao videohostOffersHistoryYDao;

    @Override
    public PeriodicTaskType getType() {
        return PeriodicTaskType.UPLOAD_VIDEOHOST_OFFER;
    }

    @Override
    public TaskSchedule getSchedule() {
        return TaskSchedule.startByCron("0 15 * * * *");
    }

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Upload current table to Yt");

        ytService.inTransaction(tableCurrentOffersPath).execute(cypressService -> {
            cypressService.create(tableCurrentOffersPath, YtNode.NodeType.TABLE, true, new YtNodeAttributes().setSchema(FC.YT_TABLE_SCHEMA), true);
            cypressService.writeTable(tableCurrentOffersPath, ytService.prepareTableData(tableCurrentOffersPath.getName(), tableWriter -> {
                try {
                    videohostOffersCurrentYDao.forEach(offer -> writeCurrent(offer, tableWriter));
                } catch (WebmasterYdbException e) {
                    throw new WebmasterException("Ydb error",
                            new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
                } catch (YtException e) {
                    throw new WebmasterException("YT error",
                            new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                }
            }));
            return true;
        });

        log.info("Upload history table to Yt");

        ytService.inTransaction(tableHistoryOffersPath).execute(cypressService -> {
            cypressService.create(tableHistoryOffersPath, YtNode.NodeType.TABLE, true, new YtNodeAttributes().setSchema(FH.YT_TABLE_SCHEMA), true);
            cypressService.writeTable(tableHistoryOffersPath, ytService.prepareTableData(tableHistoryOffersPath.getName(), tableWriter -> {
                try {
                    videohostOffersHistoryYDao.forEach(offer -> writeHistory(offer, tableWriter));
                } catch (WebmasterYdbException e) {
                    throw new WebmasterException("Ydb error",
                            new WebmasterErrorResponse.YDBErrorResponse(getClass(), e), e);
                } catch (YtException e) {
                    throw new WebmasterException("YT error",
                            new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                }
            }));
            cypressService.waitFor(cypressService.sort(tableHistoryOffersPath, tableHistoryOffersPath, FH.HOST_URL.getName(), FH.UPDATE_DATE.getName()));
            return true;
        });

        return new Result(TaskResult.SUCCESS);
    }

    private static void writeCurrent(VideohostOffer offer, TableWriter tableWriter) {
        FC.HOST_URL.set(tableWriter, offer.getHostUrl());
        FC.OFFER_URL.set(tableWriter, offer.getOfferUrl());
        FC.STATUS.set(tableWriter, offer.getStatus().name());
        FC.ADD_DATE.set(tableWriter, offer.getAddDate().getMillis());
        FC.ERROR.set(tableWriter, offer.getError().name());
        FC.COMMENTS.set(tableWriter, offer.getComments());
        tableWriter.rowEnd();
    }

    private interface FC {
        YtSchema YT_TABLE_SCHEMA = new YtSchema();
        YtColumn<String> HOST_URL = YT_TABLE_SCHEMA.addColumn("HostUrl", YtColumn.Type.STRING);
        YtColumn<String> OFFER_URL = YT_TABLE_SCHEMA.addColumn("OfferUrl", YtColumn.Type.STRING);
        YtColumn<String> STATUS = YT_TABLE_SCHEMA.addColumn("Status", YtColumn.Type.STRING);
        YtColumn<Long> ADD_DATE = YT_TABLE_SCHEMA.addColumn("AddDate", YtColumn.Type.INT_64);
        YtColumn<String> ERROR = YT_TABLE_SCHEMA.addColumn("Error", YtColumn.Type.STRING);
        YtColumn<String> COMMENTS = YT_TABLE_SCHEMA.addColumn("Comments", YtColumn.Type.STRING);
    }

    private static void writeHistory(VideohostOffer offer, TableWriter tableWriter) {
        FH.HOST_URL.set(tableWriter, offer.getHostUrl());
        FH.OFFER_URL.set(tableWriter, offer.getOfferUrl());
        FH.STATUS.set(tableWriter, offer.getStatus().name());
        FH.ADD_DATE.set(tableWriter, offer.getAddDate().getMillis());
        FH.ERROR.set(tableWriter, offer.getError().name());
        FH.COMMENTS.set(tableWriter, offer.getComments());
        FH.UPDATE_DATE.set(tableWriter, offer.getUpdateDate().getMillis());
        tableWriter.rowEnd();
    }

    private interface FH {
        YtSchema YT_TABLE_SCHEMA = new YtSchema();
        YtColumn<String> HOST_URL = YT_TABLE_SCHEMA.addColumn("HostUrl", YtColumn.Type.STRING);
        YtColumn<String> OFFER_URL = YT_TABLE_SCHEMA.addColumn("OfferUrl", YtColumn.Type.STRING);
        YtColumn<String> STATUS = YT_TABLE_SCHEMA.addColumn("Status", YtColumn.Type.STRING);
        YtColumn<Long> ADD_DATE = YT_TABLE_SCHEMA.addColumn("AddDate", YtColumn.Type.INT_64);
        YtColumn<String> ERROR = YT_TABLE_SCHEMA.addColumn("Error", YtColumn.Type.STRING);
        YtColumn<String> COMMENTS = YT_TABLE_SCHEMA.addColumn("Comments", YtColumn.Type.STRING);
        YtColumn<Long> UPDATE_DATE = YT_TABLE_SCHEMA.addColumn("UpdateDate", YtColumn.Type.INT_64);
    }

    public static class TaskState implements PeriodicTaskState {
    }
}
