package ru.yandex.webmaster3.worker.ytimport;

import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.Setter;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.joda.time.Instant;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.worker.task.PeriodicTaskState;
import ru.yandex.webmaster3.core.worker.task.PeriodicTaskType;
import ru.yandex.webmaster3.core.worker.task.TaskResult;
import ru.yandex.webmaster3.storage.clickhouse.replication.MdbClickhouseReplicationManager;
import ru.yandex.webmaster3.storage.clickhouse.replication.data.ClickhouseReplicationTaskGroup;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.searchquery.importing.dao.YtClickhouseDataLoadRepository;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseServer;
import ru.yandex.webmaster3.storage.util.yt.YtException;
import ru.yandex.webmaster3.storage.util.yt.YtNode;
import ru.yandex.webmaster3.storage.util.yt.YtPath;
import ru.yandex.webmaster3.storage.util.yt.YtService;
import ru.yandex.webmaster3.storage.ytimport.MdbYtClickhouseImportManager;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoad;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadType;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportState;
import ru.yandex.webmaster3.storage.ytimport.YtClickhouseImportStateEnum;
import ru.yandex.webmaster3.worker.PeriodicTask;
import ru.yandex.webmaster3.worker.TaskSchedule;

import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState.DONE;
import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState.FAILED;
import static ru.yandex.webmaster3.storage.ytimport.YtClickhouseDataLoadState.INITIALIZING;

/**
 * Задача автоматического импорта поисковых запросов
 * Created by Oleg Bazdyrev on 11/04/2017.
 */
public abstract class AbstractYtClickhouseDataLoadTask extends PeriodicTask<AbstractYtClickhouseDataLoadTask.TaskState> {

    protected static final Logger log = LoggerFactory.getLogger(AbstractYtClickhouseDataLoadTask.class);
    protected static final Pattern IN_TABLE_NAME_PATTERN = Pattern.compile("clicks_shows_(\\d{8})_(\\d{8})_for_wmc_web");
    protected static final Pattern IN_TABLE_NAME_PATTERN2 = Pattern.compile("\\d{4}-\\d{2}-\\d{2}");
    protected static final DateTimeFormatter IN_TABLE_NAME_DATE_FORMATTER = DateTimeFormat.forPattern("yyyyMMdd");
    protected static final DateTimeFormatter IN_TABLE_NAME_DATE_FORMATTER2 = DateTimeFormat.forPattern("yyyy-MM-dd");
    private static final String MOST_RECENT_SOURCE_ATTR = "most_recent_source";
    private static final String LEAST_RECENT_SOURCE_ATTR = "least_recent_source";

    @Setter
    protected ClickhouseServer clickhouseServer;
    @Setter
    protected ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;
    @Setter
    protected MdbClickhouseReplicationManager clickhouseReplicationManager;
    @Setter
    protected YtClickhouseDataLoadRepository ytClickhouseDataLoadYDao;
    @Setter
    protected MdbYtClickhouseImportManager ytClickhouseImportManager;
    @Setter
    protected YtService ytService;
    @Setter
    protected YtPath tablePath;
    @Setter
    private PeriodicTaskType type;
    @Setter
    private String schedule;
    @Setter
    private YtClickhouseDataLoadType importType;

    @Override
    public Result run(UUID runId) throws Exception {
        log.info("Import search queries of type {} periodic task started", getImportType());
        // загрузим текущий статус
        YtClickhouseDataLoad latestImport = ytClickhouseDataLoadYDao.load(getImportType());
        log.info("Latest import: {}", latestImport);

        if (latestImport == null) {
            log.info("Starting new import");
            latestImport = YtClickhouseDataLoad.createDefault(getImportType());
        } else if (latestImport.getState() == FAILED) {
            log.info("Restarting import because of failure");
            // рестартуем, если прошлое выполнение завершилось ошибкой
            latestImport = latestImport.restarted();
        } else if (latestImport.getState() == DONE) {
            log.info("Done importing, starting new import");
            // начинаем заново
            latestImport = latestImport.withState(INITIALIZING);
        }

        TaskState ts = new TaskState(latestImport);
        setState(ts);

        // в зависимости от текущей стадии будем выполнять соответствующие действия
        while (!latestImport.getState().isTerminal()) {
            log.info("Import to do: {}", latestImport);

            switch (latestImport.getState()) {
                case INITIALIZING:
                    latestImport = init(latestImport);
                    break;
                case PREPARING:
                    latestImport = prepare(latestImport);
                    break;
                case IMPORTING:
                    latestImport = doImport(latestImport);
                    break;
                case WAITING_FOR_IMPORT:
                    latestImport = waitForImport(latestImport);
                    break;
                case REPLICATING:
                    latestImport = replicate(latestImport);
                    break;
                case WAITING_FOR_REPLICATION:
                    latestImport = waitForReplication(latestImport);
                    break;
                case CLEANING:
                    latestImport = clean(latestImport);
                    break;
                case RENAMING:
                    latestImport = rename(latestImport);
                    break;
                case CREATING_DISTRIBUTED:
                    latestImport = createDistributedTables(latestImport);
                    break;
                default:
                    throw new IllegalStateException("Unknown non-terminal state " + latestImport.getState());
            }

            if (latestImport.getState() == YtClickhouseDataLoadState.CREATING_DISTRIBUTED) {
                latestImport = latestImport.withLastSuccess(Instant.now());
            }
            getState().imprt = latestImport;
            ytClickhouseDataLoadYDao.save(latestImport);

            log.info("Finished import: {}", latestImport);
        }

        return new Result(latestImport.getState() == FAILED ? TaskResult.FAIL : TaskResult.SUCCESS);
    }

    /**
     * Инициализирует импорт, находит подходящую таблицу в YT и вычисляет период обновления поисковых запросов
     */
    protected YtClickhouseDataLoad init(YtClickhouseDataLoad latestImport) throws Exception {
        return initBySourceAttr(latestImport);
    }

    /**
     * Инициализирует импорт, находит подходящую таблицу в YT и вычисляет период обновления поисковых запросов
     */
    protected YtClickhouseDataLoad initBySourceAttr(YtClickhouseDataLoad latestImport) throws Exception {
        // берем таблику favorite_queries, у нее смотрим атрибут most_recent_source и сравниваем с датами
        // из сохраненного импорта
        return ytService.withoutTransactionQuery(cypressService -> {
            YtNode node = cypressService.getNode(tablePath);
            LocalDate from = dateFromSource(node, LEAST_RECENT_SOURCE_ATTR);
            LocalDate to = dateFromSource(node, MOST_RECENT_SOURCE_ATTR);
            if (from != null && to != null) {
                if (latestImport.getDateTo() == null || latestImport.getDateTo().isBefore(to)) {
                    LocalDate prevFrom = latestImport.getDateTo() == null ? null : latestImport.getDateTo().plusDays(1);
                    return latestImport.withSourceTable(tablePath, ObjectUtils.max(from, prevFrom), to);
                }
            }
            return latestImport.withState(YtClickhouseDataLoadState.DONE);
        });
    }

    /**
     *  Инициализация по значенияю атрибуту
     */
    protected YtClickhouseDataLoad initBySourceAttr(YtClickhouseDataLoad latestImport, String attrName) throws Exception {
        return ytService.withoutTransactionQuery(cypressService -> {
            YtNode node = cypressService.getNode(tablePath);
            String attrValue = node.getNodeMeta().get(attrName).asText();
            if (latestImport.getData() == null || latestImport.getData().compareTo(attrValue) < 0) {
                LocalDate today = LocalDate.now();
                return latestImport.withSourceTable(tablePath, today, today).withData(attrValue);
            }
            return latestImport.withState(YtClickhouseDataLoadState.DONE);
        });
    }

    /**
     * Инициализирует импорт, находит подходящую таблицу в YT и вычисляет период обновления поисковых запросов
     * TODO  выделить общий код с
     *
     * @param latestImport
     * @return
     */
    protected YtClickhouseDataLoad initGroupsByTableName(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // поищем в папке groupDir первый необработанный файл с данными
        return ytService.withoutTransactionQuery(cypressService -> {
            List<YtPath> tables = cypressService.list(tablePath);
            tables.sort(YtPath::compareTo);
            LocalDate expectDate = Optional.ofNullable(latestImport.getDateTo()).map(d -> d.plusDays(1)).orElse(null);
            log.info("Expecting table for " + expectDate);
            for (YtPath table : tables) {
                Matcher matcher = IN_TABLE_NAME_PATTERN.matcher(table.getName());
                if (!matcher.matches()) {
                    continue;
                }
                // получаем период
                LocalDate from = IN_TABLE_NAME_DATE_FORMATTER.parseLocalDate(matcher.group(1));
                LocalDate to = IN_TABLE_NAME_DATE_FORMATTER.parseLocalDate(matcher.group(2));
                if (expectDate == null || expectDate.equals(from)) {
                    log.info("Found expected table: " + table);
                    return latestImport.withSourceTable(table, from, to);
                } else {
                    if (from.isAfter(expectDate)) {
                        final String m = "Found gap in source data: expected data for date" + expectDate + " but found " + from;
                        log.error(m);
                        throw new RuntimeException(m);
                    }
                }
            }

            // ничего не нашли, сразу завершимся
            log.info("Expected table not found, nothing to import");
            return latestImport.withState(DONE);
        });
    }

    /**
     * Инициализирует импорт находя первую необработанную таблицу, имена таблиц формата YYYY-MM-DD
     */
    protected YtClickhouseDataLoad initByTableName(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // поищем в папке groupDir первый необработанный файл с данными
        return ytService.withoutTransactionQuery(cypressService -> {
            String prevData = latestImport.getSourceTable() == null ? "" : latestImport.getSourceTable().getName();
            Optional<YtPath> tableToProcess = cypressService.list(tablePath).stream().filter(table -> table.getName().compareTo(prevData) > 0)
                    .min(Comparator.naturalOrder());
            if (tableToProcess.isPresent()) {
                LocalDate importingDate = new LocalDate(tableToProcess.get().getName());
                return latestImport.withSourceTable(tableToProcess.get(), importingDate, importingDate)
                        .withData(tableToProcess.get().getName().replace("-", ""));
            }
            // ничего не нашли, сразу завершимся
            log.info("Expected table not found, nothing to import");
            return latestImport.withState(DONE);
        });
    }

    /**
     * Инициализирует импорт находя первую необработанную таблицу с именем вида yyyy-mm-dd
     */
    protected YtClickhouseDataLoad initByTableDateName(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // поищем в папке groupDir первый необработанный файл с данными
        return ytService.withoutTransactionQuery(cypressService -> {
            String prevData = latestImport.getSourceTable() == null ? "" : latestImport.getSourceTable().getName();
            Optional<YtPath> tableToProcess = cypressService.list(tablePath).stream().filter(table -> table.getName().compareTo(prevData) > 0)
                    .min(Comparator.naturalOrder());
            if (tableToProcess.isPresent()) {
                LocalDate date = LocalDate.parse(tableToProcess.get().getName());
                return latestImport.withSourceTable(tableToProcess.get(), date, date).withData(tableToProcess.get().getName().replace("-", ""));
            }
            // ничего не нашли, сразу завершимся
            log.info("Expected table not found, nothing to import");
            return latestImport.withState(DONE);
        });
    }

    /**
     * Инициализирует импорт находя последнюю таблицу
     */
    protected YtClickhouseDataLoad initByMaxTableName(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // поищем в папке groupDir первый необработанный файл с данными
        return ytService.withoutTransactionQuery(cypressService -> {
            String prevData = latestImport.getData() == null ? "" : latestImport.getData();
            Optional<YtPath> tableToProcess = cypressService.list(tablePath).stream().max(Comparator.naturalOrder());
            if (tableToProcess.isPresent() && tableToProcess.get().getName().replace("-", "").compareTo(prevData) > 0) {
                return latestImport.withSourceTable(tableToProcess.get(), LocalDate.now(), LocalDate.now())
                        .withData(tableToProcess.get().getName().replace("-", ""));
            }
            // ничего не нашли, сразу завершимся
            log.info("Expected table not found, nothing to import");
            return latestImport.withState(DONE);
        });
    }

    /**
     * По дате обновления исходной таблички
     *
     * @param latestImport
     * @return
     * @throws InterruptedException
     * @throws YtException
     */
    protected YtClickhouseDataLoad initByUpdateDate(YtClickhouseDataLoad latestImport) throws InterruptedException, YtException {
        // проверим дату обновления таблички
        return ytService.inTransaction(tablePath).query(cypressService -> {
            YtNode node = cypressService.getNode(tablePath);
            if (node != null) {
                long millis = node.getUpdateTime().getMillis();
                long oldMillis = latestImport.getData() == null ? 0L : Long.valueOf(latestImport.getData());
                if (millis > oldMillis) {
                    LocalDate dateFrom = node.getUpdateTime().toLocalDate();
                    return latestImport.withData(String.valueOf(millis))
                            .withSourceTable(tablePath, dateFrom, dateFrom);
                }
            }
            // ничего нового
            return latestImport.withState(YtClickhouseDataLoadState.DONE);
        });
    }

    public static LocalDate dateFromSource(YtNode node, String attrName) {
        JsonNode attrNode = node.getNodeMeta().get(attrName);
        if (attrNode != null) {
            String source = attrNode.asText("");
            int slashIndex = source.lastIndexOf("/");
            if (slashIndex < source.length() - 1) {
                Matcher matcher = IN_TABLE_NAME_PATTERN.matcher(source.substring(slashIndex + 1));
                if (matcher.matches()) {
                    return IN_TABLE_NAME_DATE_FORMATTER.parseLocalDate(matcher.group(1)); // always same date;
                } else {
                    matcher = IN_TABLE_NAME_PATTERN2.matcher(source.substring(slashIndex + 1));
                    if (matcher.matches()) {
                        return IN_TABLE_NAME_DATE_FORMATTER2.parseLocalDate(matcher.group());
                    }
                }
            }
        }
        return null;
    }

    /**
     * Подготавливает данные для импорта
     */
    protected abstract YtClickhouseDataLoad prepare(YtClickhouseDataLoad imprt) throws Exception;

    /**
     * Импортирует данные из YT в Clickhouse
     */
    protected abstract YtClickhouseDataLoad doImport(YtClickhouseDataLoad imprt) throws Exception;

    /**
     * Ожидает завершения импорта
     */
    protected YtClickhouseDataLoad waitForImport(YtClickhouseDataLoad imprt) throws Exception {
        log.info("Waiting for import to complete: {}", imprt);

        // ждем, пока импорт не завершится
        List<UUID> taskIds = imprt.getImportTaskIds();
        if (taskIds != null) {
            for (UUID taskId : taskIds) {
                log.info("Waiting for import task {} to complete", taskId);
                YtClickhouseImportState importState = ytClickhouseImportManager.waitForCompletionAndGetState(taskId);
                if (importState == null || importState.getState() != YtClickhouseImportStateEnum.DONE) {
                    String message = importState == null
                            ? "Import task " + taskId + " was lost"
                            : "Import task " + taskId + " was failed";
                    log.error(message);
                    return imprt.withState(FAILED);
                }

                log.info("Finished waiting for import task {} to complete", taskId);
            }
        }
        return imprt.withNextState();
    }

    /**
     * Реплицирует данные между ДЦ ClickHouse
     */
    protected abstract YtClickhouseDataLoad replicate(YtClickhouseDataLoad imprt) throws Exception;

    /**
     * Ожидает завершения репликации
     */
    protected YtClickhouseDataLoad waitForReplication(YtClickhouseDataLoad imprt) throws Exception {
        log.info("Waiting for replication to complete: {}", imprt);

        // ждем, пока репликация не завершится (если репликация конечно нужна)
        List<UUID> taskIds = imprt.getReplicationTaskIds();
        if (!CollectionUtils.isEmpty(taskIds)) {
            for (UUID taskId : taskIds) {
                log.info("Waiting for replication task {} to complete", taskId);

                ClickhouseReplicationTaskGroup state = clickhouseReplicationManager.waitForCompletionAndGetState(taskId);
                if (state == null) {
                    String message = "Replication task " + taskId + " was lost";
                    log.error(message);
                    return imprt.withState(FAILED);
                }

                log.info("Finished waiting for replication task {} to complete", taskId);
            }
        }
        return imprt.withNextState();
    }

    /**
     * Чистит промежуточные данные в YT
     */
    protected YtClickhouseDataLoad clean(YtClickhouseDataLoad imprt) throws Exception {
        try {
            if (imprt.getPreparedTables() != null) {
                ytService.withoutTransaction(cypressService -> {
                    for (YtPath table : imprt.getPreparedTables()) {
                        if (cypressService.exists(table)) {
                            cypressService.remove(table);
                        }
                    }
                    return true;
                });
            }
        } catch (YtException e) {
            // ошибки при чистке не так важны
            log.error("Error when cleaning temporary tables in YT", e);
            throw e;
        }
        return imprt.withNextState();
    }

    protected YtClickhouseDataLoad rename(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    protected YtClickhouseDataLoad createDistributedTables(YtClickhouseDataLoad imprt) throws Exception {
        return imprt.withNextState();
    }

    public class TaskState implements PeriodicTaskState {
        public TaskState(YtClickhouseDataLoad imprt) {
            this.imprt = imprt;
        }

        public YtClickhouseDataLoad imprt;
    }

    @Override
    public PeriodicTaskType getType() {
        return type;
    }

    @Override
    public TaskSchedule getSchedule() {
        return schedule == null ? TaskSchedule.never() : TaskSchedule.startByCron(schedule);
    }

    protected YtClickhouseDataLoadType getImportType() {
        return importType;
    }

}
