package ru.yandex.wmtools.common.data.partition;

import java.util.Map;

/**
 * This class simplifies use of multiple databases and partitions.
 * Anywhere hostname is passed as a parameter it should be in a 'canonical hostname' format.
 *
 * @see ru.yandex.wmtools.common.SupportedProtocols#getCanonicalHostname(java.net.URL)
 * @author baton
 */
abstract public class AbstractPartition implements IPartition {
    protected void partitionById(Long partId, Map<String, Integer> partMap, StringBuilder sb) {
        for (Map.Entry<String, Integer> entry : partMap.entrySet()) {
            int fromIndex = 0;
            while (true) {
                fromIndex = sb.indexOf(entry.getKey(), fromIndex);
                if (fromIndex < 0) {
                    break;
                }
                int finishIndex = fromIndex + entry.getKey().length();
                if (checkWholeWord(sb, fromIndex, finishIndex)) {
                    sb.replace(fromIndex, finishIndex, entry.getKey() + partitioning(partId, entry.getValue()));
                }
                fromIndex++;
            }
        }
    }

    /**
     * @param cs char sequence where to find symbols.
     * @param fromIndex inclusive.
     * @param finishIndex exclusive.
     * @return returns, if the specified word is bounded by start/end of line or non-word characters.
     */
    private boolean checkWholeWord(CharSequence cs, int fromIndex, int finishIndex) {
        return checkNonWordOrDoesNotExist(cs, fromIndex - 1) && checkNonWordOrDoesNotExist(cs, finishIndex);
    }

    private boolean checkNonWordOrDoesNotExist(CharSequence cs, int idx) {
        return ((idx < 0) || (idx >= cs.length()) || new String(new char[]{cs.charAt(idx)}).matches("\\W"));
    }

    /**
     * Method for defining the number of part to address when partitioning table in db.
     *
     * @param partId Key for partitioning.
     * @param partsCnt Total number of parts. Must be a degree of 2.
     * @return Number of part to address with leading zeroes to have length of 3 symbols.
     */
    private String partitioning(long partId, long partsCnt) {
        if ((partsCnt & (partsCnt - 1)) != 0) {
            throw new IllegalArgumentException("partsCnt must be a degree of 2.");
        }

        // Count number of digits in partitioned table names.
        return String.format("%1$0" + getDigitsCount() + "d", partId % partsCnt);
    }

    private int getDigitsCount() {
        int digits = 0;
        int count = getPartitionsCount();
        while (count > 0) {
            digits++;
            count /= 10;
        }

        return digits;
    }

    public static int getStringHash(String hostname, int baseCount) {
        String loHostname = hostname.toLowerCase();
        int res = 0;
        for (int i = 0; i < loHostname.length(); i++) {
            res = (res + loHostname.charAt(i)) % baseCount;
        }

        return res;
    }

    abstract protected int getPartitionsCount();
}
