package ru.yandex.wmtools.common.data.plot;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import org.apache.commons.lang3.StringEscapeUtils;

import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * @author avhaliullin
 */
public class HighchartPlotDataWrapper<K, V> extends XmlDataWrapper<List<NavigableMap<K, V>>> {
    private PlotHelper<K, V> plotHelper;
    private List<String> labels;
    private List<Long> ids;
    private boolean outputIfEmpty;

    public HighchartPlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, PlotHelper<K, V> plotHelper) {
        this(name, graphs, labels, null, plotHelper,false);
    }

    public HighchartPlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, PlotHelper<K, V> plotHelper, boolean outputIfEmpty) {
        super(graphs, "chart", "name", name);
        this.outputIfEmpty = outputIfEmpty;
        this.plotHelper = plotHelper;
        this.labels = labels;
        this.ids = null;
    }

    public HighchartPlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, List<Long> ids, PlotHelper<K, V> plotHelper) {
        this(name, graphs, labels, ids, plotHelper,false);
    }

    public HighchartPlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, List<Long> ids, PlotHelper<K, V> plotHelper, boolean outputIfEmpty) {
        super(graphs, "chart", "name", name);
        this.outputIfEmpty = outputIfEmpty;
        this.plotHelper = plotHelper;
        this.labels = labels;
        this.ids = ids;
    }

    private Iterable<K> allPossibleKeys(NavigableMap<K, V> graph) {
        final K min = graph.isEmpty() ? null : graph.firstKey();
        final K max = graph.isEmpty() ? null : graph.lastKey();
        return new Iterable<K>() {
            @Override
            public Iterator<K> iterator() {
                return new Iterator<K>() {
                    private K current;

                    @Override
                    public boolean hasNext() {
                        if (min == null) {
                            return false;
                        }
                        return current == null || plotHelper.compare(current, max) < 0;
                    }

                    @Override
                    public K next() {
                        if (current == null) {
                            current = min;
                        } else {
                            current = plotHelper.nextKey(current);
                        }
                        return current;
                    }

                    @Override
                    public void remove() {
                        throw new UnsupportedOperationException();
                    }
                };
            }
        };
    }

    public HighchartPlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, PlotHelper<K, V> plotHelper) {
        this(name, graphs, null, null, plotHelper);
    }

    @Override
    protected void doToXml(StringBuilder result) {
        List<NavigableMap<K, V>> correctedGraphs = new ArrayList<NavigableMap<K, V>>();
        for (NavigableMap<K, V> graph : data) {
            NavigableMap<K, V> correctedGraph = new TreeMap<K, V>();
            correctedGraphs.add(correctedGraph);
            if (graph.isEmpty()) {
                continue;
            }
            if (graph.size() == 1) {
                correctedGraph.putAll(graph);
                K key = graph.lastKey();
                K next = plotHelper.nextKey(key);
                K prev = plotHelper.prevKey(key);
                correctedGraph.put(prev, null);
                correctedGraph.put(next, null);
                continue;
            }
            K lastExistingKey = null;
            for (K key : allPossibleKeys(graph)) {
                V value = graph.get(key);
                if (value != null) { // all good
                    correctedGraph.put(key, value);
                    lastExistingKey = key;
                    continue;
                }

                K nextExistingKey = graph.higherKey(key);
                if (nextExistingKey == null) {
                    break;
                }

                if (lastExistingKey == null) {
                    continue;
                }

                value = plotHelper.interpolate(lastExistingKey, graph.get(lastExistingKey),
                        nextExistingKey, graph.get(nextExistingKey),
                        key);
                if (value != null) { // interpolated in real value, all good
                    correctedGraph.put(key, value);
                    continue;
                }
                if (!plotHelper.isNeedGaps()) { // get null, but doesn't use gaps
                    continue;
                }
                if (correctedGraph.isEmpty() || correctedGraph.lastEntry().getValue() == null) { // drawing gap, but already put down null
                    continue;
                }
                correctedGraph.put(key, null);
            }
        }

        //optimizing 0 values:
        if (plotHelper instanceof DateNumberPlotHelper) {
            DateNumberPlotHelper numberPlotHelper = (DateNumberPlotHelper) plotHelper;
            boolean first = true;
            Set<K> allKeysToRemove = new HashSet<K>();
            for (NavigableMap<K, V> graph : correctedGraphs) {
                Set<K> keysToRemove = new HashSet<K>();
                for (Iterator<K> it = graph.navigableKeySet().iterator(); it.hasNext(); ) {
                    K key = it.next();
                    K prev = graph.lowerKey(key);
                    K next = graph.higherKey(key);
                    if (numberPlotHelper.getZero().equals(graph.get(key)) && // Current value == 0
                            (prev != null && numberPlotHelper.getZero().equals(graph.get(prev))) && // Prev value == 0
                            (next != null && numberPlotHelper.getZero().equals(graph.get(next)))) { // Next value == 0
                        //it.remove();
                        keysToRemove.add(key);
                    }
                }
                if (first) {
                    allKeysToRemove = keysToRemove;
                    first = false;
                } else {
                    allKeysToRemove.retainAll(keysToRemove);
                }
            }
            if (!allKeysToRemove.isEmpty()) {
                for (NavigableMap<K, V> graph : correctedGraphs) {
                    graph.keySet().removeAll(allKeysToRemove);
                }
            }
        }

        TreeSet<K> series = new TreeSet<K>(plotHelper);
        for (NavigableMap<K, V> graph : correctedGraphs) {
            series.addAll(graph.keySet());
        }

        if (series.isEmpty() && outputIfEmpty) {
            result.append("<series/><graphs>");
            int i = 1;
            if (ids != null && !ids.isEmpty()) {
                for (String label : labels) {
                    result.append("<graph gid=\"" + (i) + "\" title=\"" + StringEscapeUtils.escapeXml10(label) + "\"");
                    result.append(" id=\"").append(ids.get(i)).append("\" />");
                    i++;
                }
            } else {
                for (String label : labels) {
                    result.append("<graph gid=\"" + (i++) + "\" title=\"" + StringEscapeUtils.escapeXml10(label) + "\"/>");
                }
            }
            result.append("</graphs>");
            return;
        }
        result.append("<series>");
        int index = 0;
        Map<K, Integer> key2index = new HashMap<K, Integer>();
        for (K key : series) {
            result.append("<value xid=\"").append(index).append("\">");
            result.append(StringEscapeUtils.escapeXml10(plotHelper.formatKey(key)));
            result.append("</value>");
            key2index.put(key, index++);
        }
        result.append("</series>");
        result.append("<graphs>");
        int graphIndex = 1;
        for (NavigableMap<K, V> graph : correctedGraphs) {
            result.append("<graph gid=\"").append(graphIndex).append("\"");
            if ((labels != null) && (!labels.isEmpty())) {
                result.append(" title=\"").append(StringEscapeUtils.escapeXml10(labels.get(graphIndex - 1))).append("\"");
            }
            if (ids != null && !ids.isEmpty()) {
                result.append(" id=\"").append(ids.get(graphIndex - 1)).append("\"");
            }
            if (graph.isEmpty() && outputIfEmpty) {
                result.append("/>");
                graphIndex++;
                continue;
            }
            result.append(">");
            graphIndex++;
            for (K key : graph.navigableKeySet()) {
                V value = graph.get(key);
                if (value != null) {
                    result.append("<value xid=\"").append(key2index.get(key)).append("\">");
                    result.append(StringEscapeUtils.escapeXml10(plotHelper.formatValue(value)));
                    result.append("</value>");
                } else {
                    result.append("<value xid=\"").append(key2index.get(key)).append("\">null</value>");
                }
            }
            result.append("</graph>");
        }
        result.append("</graphs>");
    }
}
