package ru.yandex.wmtools.common.data.plot;

import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeSet;

import org.apache.commons.lang3.StringEscapeUtils;

import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 29.10.2008
 * Time: 20:09:16
 * <p/>
 * Used to return data for amchart (www.amcharts.com) flash plots.
 * Takes list of maps : X --> Y and PlotHelper as input.
 * Important: all x in X MUST be on some grid, i.e. any x_i in X must be in G : x_0, PlotHelper.nextKey(x_0), PlotHelper.nextKey(PlotHelper.nextKey(x_0)), ...
 */
public class PlotDataWrapper<K, V> extends XmlDataWrapper<List<NavigableMap<K, V>>> {
    private TreeSet<K> series;
    private PlotHelper<K, V> plotHelper;
    private List<String> labels;
    private boolean outputIfEmpty;

    public PlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, PlotHelper<K, V> plotHelper) {
        this(name, graphs, labels, plotHelper, false);
    }

    public PlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, List<String> labels, PlotHelper<K, V> plotHelper, boolean outputIfEmpty) {
        super(graphs, "chart", "name", name);
        this.outputIfEmpty = outputIfEmpty;
        this.plotHelper = plotHelper;
        this.labels = labels;

        Comparator<? super K> keyComporator = new Comparator<K>() {
            @Override
            public int compare(K key1, K key2) {
                return PlotDataWrapper.this.plotHelper.compare(key1, key2);
            }
        };
        series = new TreeSet<K>(keyComporator);
        TreeSet<K> initialSeries = new TreeSet<K>(keyComporator);
        for (Map<K, V> graph : graphs) {
            initialSeries.addAll(graph.keySet());
        }

        if (!initialSeries.isEmpty()) {
            K first = initialSeries.first();
            K last = initialSeries.last();
            K key = first;
            while (plotHelper.compare(key, last) <= 0) {
                series.add(key);
                key = plotHelper.nextKey(key);
            }
        }
    }

    public PlotDataWrapper(String name, List<NavigableMap<K, V>> graphs, PlotHelper<K, V> plotHelper) {
        this(name, graphs, null, plotHelper);
    }

    @Override
    protected void doToXml(StringBuilder result) {
        if (series.isEmpty() && outputIfEmpty) {
            result.append("<series/><graphs>");
            int i = 1;
            for (String label : labels) {
                result.append("<graph gid=\"" + (i++) + "\" title=\"" + StringEscapeUtils.escapeXml10(label) + "\"/>");
            }
            result.append("</graphs>");
            return;
        }
        result.append("<series>");
        int index = 0;
        Map<K, Integer> key2index = new HashMap<K, Integer>();
        for (K key : series) {
            result.append("<value xid=\"").append(index).append("\">");
            result.append(StringEscapeUtils.escapeXml10(plotHelper.formatKey(key)));
            result.append("</value>");
            key2index.put(key, index++);
        }
        result.append("</series>");
        result.append("<graphs>");
        int graphIndex = 1;
        K lastKey = series.last();
        for (Map<K, V> graph : data) {
            result.append("<graph gid=\"").append(graphIndex).append("\"");
            if ((labels != null) && (!labels.isEmpty())) {
                result.append(" title=\"").append(StringEscapeUtils.escapeXml10(labels.get(graphIndex - 1))).append("\"");
            }
            if (graph.isEmpty() && outputIfEmpty) {
                result.append("/>");
                graphIndex++;
                continue;
            }
            /////////////////
            result.append(">");
            graphIndex++;
            K prevKey = null;
            K nextKey = null;
            boolean afterGraph = false;
            boolean inGap = false;
            for (K key : series) {
                if (graph.containsKey(key)) {
                    prevKey = key;
                    nextKey = key;
                    if (plotHelper.isNeedGaps() && inGap) {
                        result.append("<graph gid=\"").append(graphIndex - 1).append("\">");
                    }
                    inGap = false;
                } else {
//                    Logger.getLogger(PlotDataWrapper.class).debug("Date is " + key + " need gaps = " + plotHelper.isNeedGaps() + " must contain = " + plotHelper.mustContainKey(key) + " in gap = " + inGap);
                    if (plotHelper.isNeedGaps() && plotHelper.mustContainKey(key) && !inGap) {
                        result.append("</graph>");
                        inGap = true;
                    }

                    if (!afterGraph && (nextKey == null || plotHelper.compare(nextKey, key) < 0)) {
                        nextKey = plotHelper.nextKey(key);
                        do {
                            if (plotHelper.compare(nextKey, key) >= 0 && graph.containsKey(nextKey)) {
                                break;
                            }
                            nextKey = plotHelper.nextKey(nextKey);
                        }
                        while (plotHelper.compare(nextKey, lastKey) <= 0);
                        if (!(plotHelper.compare(nextKey, key) >= 0 && graph.containsKey(nextKey))) {
                            afterGraph = true;
                            nextKey = null;
                        }
                    }
                }
                V value;
                if (graph.containsKey(key)) {
                    value = graph.get(key);
                } else {
                    V prevValue = (prevKey == null) ? null : graph.get(prevKey);
                    V nextValue = (nextKey == null) ? null : graph.get(nextKey);
                    value = plotHelper.interpolate(prevKey, prevValue, nextKey, nextValue, key);
                }
                if (value != null) {
                    result.append("<value xid=\"").append(key2index.get(key)).append("\">");
                    result.append(StringEscapeUtils.escapeXml10(plotHelper.formatValue(value)));
                    result.append("</value>");
                }
            }
            if (!inGap) {
                result.append("</graph>");
            }
        }
        result.append("</graphs>");
    }
}
