package ru.yandex.wmtools.common.error;

import java.util.Collection;

import ru.yandex.common.framework.core.ErrorInfo;
import ru.yandex.common.util.xml.XmlConvertable;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * Created by IntelliJ IDEA.
 * User: senin
 * Date: 22.03.2007
 * Time: 20:48:18
 */
public class CodeErrorInfo extends XmlDataWrapper<Object> implements ErrorInfo {
    private static final String TAG_USER_TEXT = "user-text";
    private static final String TAG_CODE = "code";
    private static final String TAG_MESSAGE = "message";
    private static final String TAG_EXCEPTION = "exception";

    private final String message;
    private final String userText;
    private final Throwable throwable;
    private final Collection<ExtraTagInfo> params;

    public CodeErrorInfo(Object code, Integer id, String message, String userText, Throwable throwable,
            Collection<ExtraTagInfo> extraParams) {
        super(code, "error", "code", (id == null) ? "" : Integer.toString(id));
        this.message = message;
        this.userText = userText;
        this.throwable = throwable;
        this.params = extraParams;
    }

    @Override
    protected void doToXml(StringBuilder result) {
        if (data instanceof XmlConvertable) {
            ((XmlConvertable) data).toXml(result);
        } else {
            putTag(result, TAG_CODE, data.toString());
        }

        if (message != null) {
            putTag(result, TAG_MESSAGE, message);
        }

        if (userText != null) {
            String parsedUserText = parseUserText(userText, params);

            putTag(result, TAG_USER_TEXT, parsedUserText);
        }

        if (params != null) {
            for (ExtraTagInfo param : params) {
                putTag(result, param.getName(), param.getValue());
            }
        }

        if (throwable != null) {
            putTag(result, TAG_EXCEPTION, throwable.getClass().getName() + ": " + throwable.getMessage());
        }
    }

    /**
     * User text can include extra tag values, and that's why supports special language.
     *
     * Text, that will be produced without any conditions is written in property value and should not
     *      contain any '$', '[', '|' and ']' marks.
     * '[' is a start of block, and ']' is the end of block.
     * '|' is a start of alternative part inside a block.
     * You can assume that either text is a block too (as if it were surrounded by '[' and ']').
     * Block will be produced into output if and only if no conditions were failed.
     * If some conditions in block were failed, but an alternative exist - it will be output.
     * Alternative part should contain no conditions.
     * Condition - is a '$abc' string. It will be changed into value of extra-param with name 'abc'.
     * If no extra-param with such name exists, or it is empty - condition is failed and the
     *      entire block will not be produced into output (if it is no alternative part).
     * Blocks may be inserted one into another unlimited times.
     *
     * For example.
     *
     * Input:
     *      User [with IP-address $ip|unknown] is not allowed here.
     * Possible output:
     *      User unknown is not allowed here.
     *      User with IP-address 1.2.3.4 is not allowed here.
     *
     * Input:
     *      My heart will go on[ if you tell me a $story_type[ about $elephant_name]| always!]
     * Possible output:
     *      My heart will go on always!
     *      My heart will go on if you tell me a fairy tale
     *      My heart will go on if you tell me a fairy tale about Vova
     *
     * @param userText Text for user, written on a special language.
     * @param params Extra params that will be used to replace '$abc' conditions.
     * @return Returns final text for user.
     */
    protected String parseUserText(String userText, Collection<ExtraTagInfo> params) {
        StringBuilder sb = new StringBuilder();
        resolveBrackets(sb, 0, userText + "]", params);
        return sb.toString();
    }

    private int resolveBrackets(StringBuilder sb, int startIndex, String s, Collection<ExtraTagInfo> params) {
        boolean ok = true;

        StringBuilder alt = new StringBuilder();
        boolean altMode = false;

        for (int i = startIndex; i < s.length(); i++) {
            if (s.charAt(i) == '[') {
                StringBuilder toAppend = new StringBuilder();
                i = resolveBrackets(toAppend, i + 1, s, params);
                if (altMode) {
                    alt.append(toAppend);
                } else {
                    sb.append(toAppend);
                }
            } else if (s.charAt(i) == ']') {
                if (!ok) {
                    sb.delete(0, sb.length());
                    if (altMode) {
                        sb.append(alt);
                    }
                }
                return i;
            } else if (s.charAt(i) == '|') {
                altMode = true;
            } else if (s.charAt(i) == '$') {
                if (altMode) {
                    throw new AssertionError("Symbol '$' is not allowed in alternative (right) part of construction '[...$...|...]");
                }

                int cur = i + 1;
                StringBuilder tagName = new StringBuilder();

                while (true) {
                    if (cur >= s.length()) {
                        sb.delete(0, sb.length());
                        return s.length() - 1;
                    }
                    String nextChar = new String(new char[] {s.charAt(cur)}).toLowerCase();
                    if (nextChar.matches("[a-zA-Z_]")) {
                        tagName.append(nextChar);
                    } else {
                        break;
                    }
                    cur++;
                }

                ok = false;
                for (ExtraTagInfo param : params) {
                    if (param.getName().toLowerCase().equals(tagName.toString())) {
                        String value = param.getValue();
                        if ((value != null) &&  (!value.trim().equals(""))) {
                            ok = true;
                            sb.append(param.getValue());
                        }
                        break;
                    }
                }

                i = cur - 1;
            } else {
                if (altMode) {
                    alt.append(s.charAt(i));
                } else {
                    sb.append(s.charAt(i));
                }
            }
        }

        sb.delete(0, sb.length());
        return s.length() - 1;
    }
}
