package ru.yandex.wmtools.common.servantlet;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.IDN;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.InputStreamSource;

import ru.yandex.common.framework.core.RemoteFile;
import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.common.framework.core.Servantlet;
import ru.yandex.wmtools.common.Constants;
import ru.yandex.wmtools.common.SupportedProtocols;
import ru.yandex.wmtools.common.error.AbstractWMToolsException;
import ru.yandex.wmtools.common.error.CodeErrorInfo;
import ru.yandex.wmtools.common.error.ExtraTagInfo;
import ru.yandex.wmtools.common.error.ExtraTagNameEnum;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.framework.http.ServResponseStatus;
import ru.yandex.wmtools.common.framework.http.WMHttpServResponse;
import ru.yandex.wmtools.common.service.IService;
import ru.yandex.wmtools.common.util.URLUtil;
import ru.yandex.wmtools.common.util.XmlDataWrapper;

/**
 * This class has to be a superclass for all servantlets, since it contains
 * error handling common for all servantlets.
 *
 * @author ailyin
 */
public abstract class AbstractServantlet implements Servantlet, Constants {
    private static final Logger log = LoggerFactory.getLogger(AbstractServantlet.class);
    private static final Logger actionLog = LoggerFactory.getLogger("ActionResult");

    private static final Pattern COLOR_REGEX = Pattern.compile(
            "[#][0-9a-fA-F][0-9a-fA-F][0-9a-fA-F][0-9a-fA-F][0-9a-fA-F][0-9a-fA-F]");
    private static final int MAX_URL_LENGTH = 1024;
    private static final int MAX_HOST_LENGTH = 255;

    private static final String THROWABLE_ERROR_CODE = "UNSPECIFIED_ERROR";
    private static final String PARAM_YA_DOMAIN = "_ya_domain";

    private static final String ACTION_RESULT_LOGGING = "Action Result: status={} action={} timeMs={}";

    @Override
    public void process(ServRequest req, ServResponse res) {
        long startedAt = System.currentTimeMillis();
        try {
            doProcess(req, res);
        } catch (UserException e) {
            log.info(getErrorMessage(e));
            addErrorInfo(res, e);
        } catch (InternalException e) {
            log.error(getErrorMessage(e), e);
            addErrorInfo(res, e);
        } catch (RuntimeException e) {
            log.error(getErrorMessage(e), e);
            addErrorInfo(res, e);
        }
        long executionTime = System.currentTimeMillis() - startedAt;
        if (res instanceof WMHttpServResponse) {
            actionLog.info(ACTION_RESULT_LOGGING, ((WMHttpServResponse) res).getStatus().name(), req.getName(), executionTime);
        } else {
            log.error("ServResponse should be instance of WMHttpServResponse");
            actionLog.info(ACTION_RESULT_LOGGING, ServResponseStatus.SUCCESS.name(), req.getName(), executionTime);
        }
    }

    protected void setResponseErrorStatus(ServResponse resp, ServResponseStatus status) {
        if (resp instanceof WMHttpServResponse) {
            ((WMHttpServResponse) resp).errorHappened(status);
        }
    }

    /**
     * May be overridden
     */
    public void addErrorInfo(ServResponse res, UserException e) {
        res.addErrorInfo(new CodeErrorInfo(e.getProblem(), e.getProblem().getId(), e.getMessage(), null, e, e.getExtraParams()));
        setResponseErrorStatus(res, ServResponseStatus.USER_ERROR);
    }

    /**
     * May be overridden
     */
    public void addErrorInfo(ServResponse res, InternalException e) {
        res.addErrorInfo(new CodeErrorInfo(e.getProblem(), null, e.getMessage(), null, e, e.getExtraParams()));
        setResponseErrorStatus(res, ServResponseStatus.FAIL);
    }

    /**
     * May be overridden
     */
    public void addErrorInfo(ServResponse res, Throwable e) {
        res.addErrorInfo(new CodeErrorInfo(THROWABLE_ERROR_CODE, null, e.getMessage(), null, e, null));
        setResponseErrorStatus(res, ServResponseStatus.FAIL);
    }

    public static String getDbIndexMessageFromExtraParam(AbstractWMToolsException e) {
        ExtraTagInfo extraTagInfo = e.getExtraParam(ExtraTagNameEnum.DB_INFO);
        if (extraTagInfo != null) {
            return getDatabaseInfoLine(extraTagInfo);
        }
        return "";
    }

    public static String getDatabaseInfoLine(ExtraTagInfo extraTagInfo) {
        return "Database info: '" + extraTagInfo.getValue() + "'; ";
    }

    protected abstract void doProcess(ServRequest req, ServResponse res) throws UserException, InternalException;

    protected String getErrorMessage(Throwable e) {
        StringBuilder errorMessage = new StringBuilder();
        if (e instanceof AbstractWMToolsException) {
            errorMessage.append(getDbIndexMessageFromExtraParam((AbstractWMToolsException) e));
        }
        errorMessage.append("Exception in ").append(getClass().getName()).append(": ").append(e.getMessage());
        return errorMessage.toString();
    }

    public static void checkService(IService service, Class clazz) {
        if (service == null) {
            throw new IllegalStateException(clazz.getName() + " service is missed! (SERVICE_MISSED)");
        }
    }

    public static String getRequiredStringParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = req.getParam(paramName, true);
        if (stringValue == null) {
            throw new UserException(UserProblem.REQUIRED_PARAM_MISSED, "Required param missed: " + paramName, paramName);
        }
        return stringValue;
    }

    public static long getRequiredLongParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = getRequiredStringParam(req, paramName);

        try {
            return Long.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
    }

    public static Long getLongParam(final ServRequest req, final String paramName) throws UserException {
        return getLongParam(req, paramName, null);
    }

    public static Long getLongParam(final ServRequest req, final String paramName, Long defaultValue) throws UserException {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return defaultValue;
        }

        try {
            return Long.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
    }

    public static Boolean getBooleanParam(final ServRequest req, final String paramName) {
        return getBooleanParam(req, paramName, null);
    }

    public static Boolean getBooleanParam(final ServRequest req, final String paramName, Boolean defaultValue) {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return defaultValue;
        }
        return Boolean.parseBoolean(stringValue);
    }

    public static boolean getRequiredBooleanParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = getRequiredStringParam(req, paramName);
        return Boolean.parseBoolean(stringValue);
    }

    public static boolean getRequiredCheckboxBooleanParam(final ServRequest req, final String paramName) {
        return Boolean.parseBoolean(req.getParam(paramName, true));
    }

    public static int getRequiredIntParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = getRequiredStringParam(req, paramName);

        try {
            return Integer.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
    }

    public static byte getRequiredByteParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = getRequiredStringParam(req, paramName);

        try {
            return Byte.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
    }

    public static long getRequiredUserId(final ServRequest req) throws UserException {
        Long userId = req.getUserId();
        if (userId == null) {
            throw new UserException(UserProblem.USER_NOT_SIGNED_IN, "User not signed in!");
        }
        return userId;
    }

    public static Long[] getMultiParamLong(ServRequest req, String paramName) {
        List<String> params = req.getMultiParams(paramName);

        List<Long> res = new ArrayList<Long>();
        Set<Long> cache = new HashSet<Long>();
        for (String stringParam : params) {
            try {
                Long longParam = Long.decode(stringParam);

                if (!cache.contains(longParam)) {
                    cache.add(longParam);
                    res.add(longParam);
                }
            } catch (NumberFormatException e) {
                // skip
            }
        }

        return res.toArray(new Long[res.size()]);
    }

    public static Integer[] getMultiParamInt(ServRequest req, String paramName) {
        List<String> params = req.getMultiParams(paramName);

        List<Integer> res = new ArrayList<Integer>();
        Set<Integer> cache = new HashSet<Integer>();
        for (String stringParam : params) {
            try {
                Integer intParam = Integer.decode(stringParam);

                if (!cache.contains(intParam)) {
                    cache.add(intParam);
                    res.add(intParam);
                }
            } catch (NumberFormatException e) {
                // skip
            }
        }

        return res.toArray(new Integer[res.size()]);
    }

    public static Integer getIntParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return null;
        }

        int value;
        try {
            value = Integer.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
        return value;
    }

    public static Byte getByteParam(final ServRequest req, final String paramName) throws UserException {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return null;
        }

        byte value;
        try {
            value = Byte.decode(stringValue);
        } catch (NumberFormatException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
        return value;
    }

    public static String getStringParam(final ServRequest req, final String paramName) {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return null;
        }

        return stringValue;
    }

    public static String getStringParam(final ServRequest req, final String paramName, final String defaultValue) throws InternalException {
        if (defaultValue == null) {
            throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Default string param <" + paramName + "> cannot be null!");
        }
        String value = getStringParam(req, paramName);
        if (value == null) {
            return defaultValue;
        } else {
            return value;
        }
    }

    public static Date getDateParam(ServRequest req, String paramName) throws UserException {
        String stringValue = req.getParam(paramName, true);
        if (StringUtils.isEmpty(stringValue)) {
            return null;
        }
        Date value;
        try {
            value = XmlDataWrapper.getDateFormat().parse(stringValue);
        } catch (ParseException e) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Invalid param: " + paramName, e, paramName, stringValue);
        }
        return value;
    }

    public static Date getRequiredDateParam(ServRequest req, String paramName) throws UserException {
        Date date = getDateParam(req, paramName);
        if (date == null) {
            throw new UserException(UserProblem.REQUIRED_PARAM_MISSED, "Required param missed: " + paramName, paramName);
        }
        return date;
    }

    public static URL getUrlParam(ServRequest req, String param) throws UserException {
        String urlString = req.getParam(param, true);
        URL rpHeaderLink = null;
        if (urlString != null) {
            rpHeaderLink = prepareUrl(urlString);
        }
        return rpHeaderLink;
    }

    protected String getYaDomainParam(ServRequest req) {
        return getStringParam(req, PARAM_YA_DOMAIN);
    }

    public static String getColorParam(ServRequest req, String paramName, String defaultColor) throws UserException {
        defaultColor = processColorParam(paramName, defaultColor);
        String color = getStringParam(req, paramName);
        if (color == null || color.isEmpty()) {
            return defaultColor;
        } else {
            return processColorParam(paramName, color);
        }
    }

    public static String getRequiredColorParam(ServRequest req, String paramName) throws UserException {
        String color = getRequiredStringParam(req, paramName);
        return processColorParam(paramName, color);
    }

    public static String getRequiredColorParam(ServRequest req, String paramName, boolean allowTransparent) throws UserException {
        String color = getRequiredStringParam(req, paramName);
        return processColorParam(paramName, color, allowTransparent);
    }

    public static int getRequiredFontSizeParam(ServRequest req, String paramName) throws UserException {
        int size = getRequiredIntParam(req, paramName);
        if ((size <= 0) || (size > 128)) {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Wrong font size param value! " + size, paramName, Integer.toString(size));
        }
        return size;
    }

    /**
     * Возвращает параметр без изменений (с лидирующими пробелами и пр.)
     *
     * @return значение параметра или null
     */
    public static String getPreciseParam(ServRequest req, String name) {
        Map<String, String> params = req.getParams();
        if (params == null) {
            return null;
        }
        return params.get(name);
    }

    /**
     * Normalizes url and checks its correctness. Doesn't encode url using punycode.
     *
     * @param url url to be checked
     * @return normalized url
     * @throws UserException if url isn't correct
     */
    @Deprecated
    public static URL prepareUrl(String url) throws UserException {
        return doPrepareUrl(url, false, false);
    }

    /**
     * Normalizes url and checks its correctness.
     *
     * @param url         url to be checked
     * @param usePunycode indicates whether or not url should be encoded using punycode
     * @return normalized url
     * @throws UserException if url isn't correct
     */
    public static URL prepareUrl(String url, boolean usePunycode) throws UserException {
        return doPrepareUrl(url, usePunycode, false);
    }

    /**
     * Normalizes hostname and checks its correctness. Doesn't encode hostname using punycode.
     *
     * @param hostname hostname to be checked
     * @return normalized hostname
     * @throws UserException if hostname isn't correct
     */
    @Deprecated
    public static URL prepareHostname(String hostname) throws UserException {
        return doPrepareUrl(hostname, false, true);
    }

    /**
     * Normalizes hostname and checks its correctness. Doesn't encode hostname using punycode.
     *
     * @param hostname    hostname to be checked
     * @param usePunycode indicates whether or not hostname should be encoded using punycode
     * @return normalized hostname
     * @throws UserException if hostname isn't correct
     */
    protected URL prepareHostname(String hostname, boolean usePunycode) throws UserException {
        return doPrepareUrl(hostname, usePunycode, true);
    }

    public static URL doPrepareUrl(String urlString, boolean usePunycode, boolean isHost) throws UserException {
        return doPrepareUrl(urlString, usePunycode, isHost, true);
    }

    public static URL doPrepareUrl(String urlString, boolean usePunycode, boolean isHost, boolean validate)
            throws UserException {
        //todo check host length
        if (isHost) {
            checkHostLength(urlString);
        } else {
            checkUrlLength(urlString);
        }

        URL url = null;     // make compiler happy
        try {
            url = SupportedProtocols.getURL(urlString);
        } catch (MalformedURLException e) {
            throwInvalidURLException(urlString, e);
        } catch (URISyntaxException e) {
            throwInvalidURLException(urlString, e);
        } catch (SupportedProtocols.UnsupportedProtocolException e) {
            throw new UserException(UserProblem.WRONG_PROTOCOL_USED,
                    "unexpected protocol was found in url: " + urlString,
                    new ExtraTagInfo(ExtraTagNameEnum.GIVEN, e.getProtocol()));
        }

        if (usePunycode) {
            try {
                String encodedHost = IDN.toASCII(url.getHost());
                if (!encodedHost.equals(url.getHost())) {
                    urlString = url.toString().replace(url.getHost(), encodedHost);
                    url = new URL(urlString);
                }
            } catch (IllegalArgumentException e) {
                /*
                 * IDN.toASCII() method performs some checks on initial and encoded urls and throws
                 * IllegalArgumentException in case url is not valid.
                 */
                throwInvalidURLException(url.getHost(), e);
            } catch (MalformedURLException e) {
                throwInvalidURLException(urlString, e);
            }
        }

        if (!validate) {
            return url;
        }

        if (!isValid(url)) {
            try {
                URI uri = new URI(url.getProtocol(), url.getAuthority(), url.getPath(), url.getQuery(), null);
                url = new URL(uri.toASCIIString());
            } catch (MalformedURLException e) {
                throwInvalidURLException(urlString, e);
            } catch (URISyntaxException e) {
                throwInvalidURLException(urlString, e);
            }
        }
        if (!isValid(url)) {
            throwInvalidURLException(url.toString(), null);
        }
        if (url.getHost().matches("\\d+\\.\\d+\\.\\d+\\.\\d+")) {
            throw new UserException(UserProblem.IP_ADDRESSES_FORBIDDEN,
                    "IP address was used instead of hostname: " + url.getHost());
        }

        return url;
    }

    private static void checkHostLength(String hostname) throws UserException {
        if (hostname.length() > MAX_HOST_LENGTH) {
            throw new UserException(UserProblem.HOST_NAME_TOO_LONG, "Hostname too long: " + hostname);
        }
    }

    private static void checkUrlLength(String url) throws UserException {
        if (url.length() > MAX_URL_LENGTH) {
            throw new UserException(UserProblem.URL_TOO_LONG, "Url too long: " + url);
        }
    }

    private static void throwInvalidURLException(String url, Exception e) throws UserException {
        throw new UserException(UserProblem.INVALID_URL, "Invalid url: " + url, e);
    }

    private static String processColorParam(String paramName, String color) throws UserException {
        return processColorParam(paramName, color, false);
    }

    private static String processColorParam(String paramName, String color, boolean allowTransparent) throws UserException {
        if (allowTransparent && "transparent".equalsIgnoreCase(color)) {
            return color.toLowerCase();
        }
        if (color == null || color.isEmpty()) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Color param is null or empty!", paramName, color);
        }
        if (!color.startsWith("#")) {
            color = "#" + color;
        }
        if (!COLOR_REGEX.matcher(color).matches()) {
            throw new UserException(UserProblem.ILLEGAL_VALUE_TYPE, "Wrong color param format! " + color, paramName, color);
        }
        return color;
    }

    protected String safeGetRemoteIp(final ServRequest request) {
        return request.getParam("remote-ip", request.getRemoteIp());
    }

    protected String[] splitInput(String input) {
        if (StringUtils.isEmpty(input)) {
            return new String[0];
        }
        return input.split("[ \\t\\n\\f\\r]+");
    }

    protected boolean isIDN(String hostname) {
        return hostname.startsWith(ACE_PREFIX) || hostname.contains('.' + ACE_PREFIX);
    }

    /**
     * Checks if url syntax is valid.
     *
     * @param url url which syntax is to be checked
     * @return true, if url syntax is valid
     */
    public static boolean isValid(String url) {
        return URLUtil.isURLValid(url);
    }

    /**
     * Checks if url syntax is valid.
     *
     * @param url url which syntax is to be checked
     * @return true, if url syntax is valid
     */
    public static boolean isValid(URL url) {
        return isValid(url.toString());
    }

    /**
     * Tryes to convert URL to URI.
     * May be useful to prevent URISyntaxException in HttpConnector.
     *
     * @param url url to check
     * @return true if URL successfully converted to URI, false otherwise
     */
    public static boolean tryConvertURLtoURI(URL url) {
        try {
            url.toURI();
            return true;
        } catch (URISyntaxException e) {
            return false;
        }
    }

    protected InputStreamSource getBufferedFileContent(final RemoteFile file, long fileSizeLimit)
            throws UserException, InternalException {
        final int blockSize = 1024 * 45;
        if (file.getSize() > fileSizeLimit) {
            throw new UserException(UserProblem.TOO_BIG_ATTACHMENT, "Too big attachment!");
        }

        final long fileSize = file.getSize();

        final List<byte[]> is = new ArrayList<byte[]>();
        InputStream fileInputStream;
        try {
            fileInputStream = file.getInputStream();
        } catch (IOException e) {
            throw new InternalException(InternalProblem.READ_FILE_ERROR, "Actual file size differs from provided by getSize() method");
        }
        for (long position = 0; position < fileSize; position += blockSize) {
            //Safe cast to int, because result less than "blockSize" and "blockSize" is int
            int toRead = (fileSize - position) > blockSize ? blockSize : (int) (fileSize - position);
            byte[] buff = new byte[toRead];
            log.debug("Buffering " + toRead + " bytes");
            try {
                int pos = 0;
                while (pos < toRead) {
                    int returnValue = fileInputStream.read(buff, pos, toRead - pos);
                    if (returnValue < 0) {
                        throw new InternalException(InternalProblem.READ_FILE_ERROR, "Actual file size differs from provided by getSize() method");
                    }
                    pos += returnValue;
                }
            } catch (IOException e) {
                throw new InternalException(InternalProblem.INTERNAL_PROBLEM, "Can't read from uploaded file", e);
            }
            is.add(buff);
        }
        try {
            fileInputStream.close();
        } catch (IOException e) {
            log.warn("Failed to close FileInputStream", e);
        }
        return new InputStreamSource() {
            @Override
            public InputStream getInputStream() throws IOException {
                return new InputStream() {
                    int bufferNumber = 0;
                    InputStream current;

                    @Override
                    public int read() throws IOException {
                        if (bufferNumber >= is.size()) {
                            return -1;
                        }
                        if (current == null) {
                            current = new ByteArrayInputStream(is.get(bufferNumber));
                        }
                        int res = current.read();
                        if (res < 0) {
                            bufferNumber++;
                            current = null;
                            return read();
                        }
                        return res;
                    }
                };
            }
        };
    }

    public static Map<String, String> getHeaders(final ServRequest req) {
        final Map<String, String> headers = new LinkedHashMap<String, String>();
        for (String header : req.getHttpHeaders().split("\n")) {
            String[] hs = header.split(":", 2);
            if (hs.length >= 2) {
                headers.put(hs[0].trim(), hs[1].trim());
            }
        }
        return Collections.unmodifiableMap(headers);
    }

    /**
     * Получить имя хоста с указанием схемы или порта, если они отличаются от HTTP и порта по умолчанию для схемы
     * Используется для проверки главного зеркала.
     *
     * @param url URL, подготовленный с помощью prepareURL
     * @return имя хоста
     */
    public static String getHostName(URL url) {
        // не показываем протокол HTTP
        return URLUtil.getHostName(url, false);
    }
}
