package ru.yandex.wmtools.common.servantlet;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmtools.common.authorization.AuthorizationScheme;
import ru.yandex.wmtools.common.authorization.PassportAuthorizationScheme;
import ru.yandex.wmtools.common.data.strategy.OutputStrategy;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.service.IUserIdService;
import ru.yandex.wmtools.common.service.IVisitingStatService;

/**
 * This is a base class for all servantlets, which require authentication.
 * It supports such common features as:
 * 1) checking user permissions to visit this page
 * 2) outputting server date and time
 * 3) outputting all hosts added by user
 * 4) checking and substituting (if needed) virtual user id for support users
 * 5) updating visiting statistics for user
 * 6) initializing services used in most servantlets
 *
 * @author baton
 * @author ailyin
 */
public abstract class AuthenticationServantlet extends AbstractServantlet {
    private static final Logger log = LoggerFactory.getLogger(AuthenticationServantlet.class);

    private AuthorizationScheme authorizationScheme;
    private OutputStrategy outputStrategy;
    private Boolean countInVisitingStatService = true;
    private IVisitingStatService visitingStatService;

    @Override
    protected final void doProcess(ServRequest req, ServResponse res)
            throws UserException, InternalException {
        checkService(visitingStatService, IVisitingStatService.class);
        outputStrategy = new OutputStrategy(req);

        Long realUserId = req.getUserId();
        log.debug("Real user id = " + realUserId);
        String userIp = req.getParam("ip", true);
        if (userIp != null) {
            log.debug("UID_IP_NAME_LOG user_id=" + realUserId + ", user_ip=" + userIp + ", name=" + req.getName());
        }
        if (countInVisitingStatService && (realUserId != null)) {
            visitingStatService.updateVisitingStat(realUserId, req.getName()); // write history for real user id
        }

        long userId = authorizationScheme.getUserId(req, res);
        log.debug("Virtual user id = " + userId);
        doProcess(req, res, userId);
    }

    protected abstract void doProcess(ServRequest req, ServResponse res, long userId)
            throws UserException, InternalException;

    /**
     * Deprecated - use {@link
     * AuthenticationServantlet#createOutputStrategy(ru.yandex.common.framework.core.ServRequest)}
     *
     * @return
     */
    @Deprecated
    public OutputStrategy getOutputStrategy() {
        return outputStrategy;
    }

    protected OutputStrategy createOutputStrategy(ServRequest req) throws UserException {
        return new OutputStrategy(req);
    }

    protected IVisitingStatService getVisitingStatService() {
        return visitingStatService;
    }

    public void setAuthorizationScheme(AuthorizationScheme authorizationScheme) {
        this.authorizationScheme = authorizationScheme;
    }

    public AuthorizationScheme getAuthorizationScheme() {
        return authorizationScheme;
    }

    public void setUserIdService(IUserIdService userIdService) {
        setAuthorizationScheme(new PassportAuthorizationScheme(userIdService));
    }

    @Required
    public void setVisitingStatService(IVisitingStatService visitingStatService) {
        this.visitingStatService = visitingStatService;
    }

    public void setCountInVisitingStatService(Boolean countInVisitingStatService) {
        this.countInVisitingStatService = countInVisitingStatService;
    }
}
