package ru.yandex.wmtools.common.servantlet.top;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.framework.core.ServRequest;
import ru.yandex.common.framework.core.ServResponse;
import ru.yandex.wmtools.common.data.TopQueryHistoryTypeEnum;
import ru.yandex.wmtools.common.data.info.RegionInfo;
import ru.yandex.wmtools.common.data.info.TopInfo;
import ru.yandex.wmtools.common.data.info.TopInfoItem;
import ru.yandex.wmtools.common.data.info.TopQueryHistoryInfo;
import ru.yandex.wmtools.common.data.info.TotalsInfo;
import ru.yandex.wmtools.common.data.plot.DateDoublePlotHelper;
import ru.yandex.wmtools.common.data.plot.DateIntegerPlotHelper;
import ru.yandex.wmtools.common.data.plot.DatePlotHelper;
import ru.yandex.wmtools.common.data.plot.PlotHelper;
import ru.yandex.wmtools.common.data.plot.UnixTimestampDateFormat;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.error.UserProblem;
import ru.yandex.wmtools.common.servantlet.AuthenticationServantlet;
import ru.yandex.wmtools.common.service.CustomRegionService;
import ru.yandex.wmtools.common.service.TopInfoService;
import ru.yandex.wmtools.common.util.SqlUtil;

abstract public class
        WMToolsTopQueryHistoryPlotDataServantlet<HostInfoType, TI extends TopInfo, TII extends TopInfoItem, TTL extends TotalsInfo>
        extends AuthenticationServantlet {

    private static final Logger log = LoggerFactory.getLogger(WMToolsTopQueryHistoryPlotDataServantlet.class);

    abstract protected TopInfoService<HostInfoType, TI, TII, TTL> getTopInfoService();
    abstract protected CustomRegionService<HostInfoType> getCustomRegionService();

    private DatePlotHelper<Integer> getIntegerPlotHelper(Integer region, HostInfoType hostInfo) throws InternalException {
        DatePlotHelper<Integer> helper = new DateIntegerPlotHelper(PlotHelper.InterpolateMode.NULL);
        helper.useGaps(getTopInfoService().getTopInfoUpdateDatesForRegionByHost(region, hostInfo));
        helper.setDateFormat(new UnixTimestampDateFormat(true));
        helper.setPeriodDays(7);
        return helper;
    }

    private DatePlotHelper<Double> getDoublePlotHelper(Integer region, HostInfoType hostInfo) throws InternalException {
        DatePlotHelper<Double> helper = new DateDoublePlotHelper(PlotHelper.InterpolateMode.NULL);
        helper.useGaps(getTopInfoService().getTopInfoUpdateDatesForRegionByHost(region, hostInfo));
        helper.setDateFormat(new UnixTimestampDateFormat(true));
        helper.setPeriodDays(7);
        return helper;
    }

    abstract protected HostInfoType getHostInfoType(ServRequest req, ServResponse res, long userId) throws UserException, InternalException;

    @Override
    public void doProcess(ServRequest req, ServResponse res, long userId) throws InternalException, UserException {
        HostInfoType hostInfoType = getHostInfoType(req, res, userId);

        List<Long> selectedQueryIds = Arrays.asList(getParamQueryId(req));
        Integer region = getParamRegion(req);
        region = region == null ? 0 : region;
        if (region < 0) {
            region = 0;
        }

        Boolean includeSubregions = getParamIncludeSubregions(req);
        includeSubregions = includeSubregions == null ? WMToolsQueriesInfoServantlet.INCLUDE_SUBREGIONS_BY_DEFAULT : includeSubregions;

        Date date = getTopInfoService().getLatestTopOrCustomInfoDateIfDateIsNull(
                region, hostInfoType, getParamGraphDate(req));
        log.debug("selected date = " + date);

        List<RegionInfo> keyRegions = getCustomRegionService().getKeyRegionsForHostAndDate(hostInfoType, date);
        if (!getTopInfoService().existsInKeyRegions(region, keyRegions)) {
            // Если заданный в параметрах регион отсутствует в списке регионов за указанный период, то
            // по-умолчанию используется регион "Общая статистика"
            region = 0;
        }
        log.debug("selected region = " + region);

        TopQueryHistoryTypeEnum type = getParamQueryHistoryType(req);
        if (type == null) {
            throw new UserException(UserProblem.ILLEGAL_PARAM_VALUE, "Value of parameter 'type' was not recognized.");
        }

        // get all history queries
        Date latestDate = getTopInfoService().getLatestInfoDateIfDateIsNull(region, hostInfoType, null, includeSubregions);
        Map<Long, String> queryId2name = new HashMap<Long, String>(); // TODO awfully unoptimal realization of getting quryId2name correspondense!!!
        log.debug("History length is " + WMToolsQueriesInfoServantlet.HISTORY_LENGTH);
        List<TopQueryHistoryInfo> historyInfoList = getTopInfoService().getQueriesHistoryInfo(region, type, keyRegions, hostInfoType, WMToolsQueriesInfoServantlet.HISTORY_LENGTH, includeSubregions, null, null, latestDate);
        for (TopQueryHistoryInfo queryHistoryInfo : historyInfoList) {
            queryId2name.put(queryHistoryInfo.getQueryId(), queryHistoryInfo.getQuery());
        }

        switch (type) {
            case CLICKS:
                processHistory(
                        res, region, keyRegions, hostInfoType, selectedQueryIds, queryId2name, getIntegerPlotHelper(region, hostInfoType), includeSubregions,
                        new HistoryPlotDataProvider<Integer, HostInfoType>() {
                            @Override
                            public NavigableMap<Date, Integer> getHistoryPlotData(Integer region, List<RegionInfo> keyRegions, HostInfoType hostInfoType, long queryId, boolean includeSubregions) throws InternalException {
                                return getTopInfoService().getQueryClicksHistory(region, keyRegions, hostInfoType, queryId, includeSubregions);
                            }
                        }
                );
                break;
            case POSITION:
                processHistory(
                        res, region, keyRegions, hostInfoType, selectedQueryIds, queryId2name, getIntegerPlotHelper(region, hostInfoType), includeSubregions,
                        new HistoryPlotDataProvider<Integer, HostInfoType>() {
                            @Override
                            public NavigableMap<Date, Integer> getHistoryPlotData(Integer region, List<RegionInfo> keyRegions, HostInfoType hostInfoType, long queryId, boolean includeSubregions) throws InternalException {
                                return getTopInfoService().getQueryPositionHistory(region, keyRegions, hostInfoType, queryId, includeSubregions);
                            }
                        }
                );
                break;
            case SHOWS:
                processHistory(
                        res, region, keyRegions, hostInfoType, selectedQueryIds, queryId2name, getIntegerPlotHelper(region, hostInfoType), includeSubregions,
                        new HistoryPlotDataProvider<Integer, HostInfoType>() {
                            @Override
                            public NavigableMap<Date, Integer> getHistoryPlotData(Integer region, List<RegionInfo> keyRegions, HostInfoType hostInfoType, long queryId, boolean includeSubregions) throws InternalException {
                                return getTopInfoService().getQueryShowsHistory(region, keyRegions, hostInfoType, queryId, includeSubregions);
                            }
                        }
                );
                break;
            case CTR:
                processHistory(
                        res, region, keyRegions, hostInfoType, selectedQueryIds, queryId2name, getDoublePlotHelper(region, hostInfoType), includeSubregions,
                        new HistoryPlotDataProvider<Double, HostInfoType>() {
                            @Override
                            public NavigableMap<Date, Double> getHistoryPlotData(Integer region, List<RegionInfo> keyRegions, HostInfoType hostInfoType, long queryId, boolean includeSubregions) throws InternalException {
                                return getTopInfoService().getQueryCTRHistory(region, keyRegions, hostInfoType, queryId, includeSubregions);
                            }
                        }
                );
                break;
            default:
                processHistory(
                        res, region, keyRegions, hostInfoType, selectedQueryIds, queryId2name, getDoublePlotHelper(region, hostInfoType), includeSubregions,
                        new HistoryPlotDataProvider<Double, HostInfoType>() {
                            @Override
                            public NavigableMap<Date, Double> getHistoryPlotData(Integer region, List<RegionInfo> keyRegions, HostInfoType hostInfoType, long queryId, boolean includeSubregions) throws InternalException {
                                return getTopInfoService().getQueryPartHistory(region, keyRegions, hostInfoType, queryId, includeSubregions);
                            }
                        }
                );
                break;
        }
    }

    interface HistoryPlotDataProvider<T, HostInfoType> {
        NavigableMap<Date, T> getHistoryPlotData(final Integer region, final List<RegionInfo> keyRegions, final HostInfoType hostInfoType, long queryId, boolean includeSubregions)
                throws InternalException;
    }

    private <T> void processHistory(ServResponse res,
                                    Integer region,
                                    List<RegionInfo> keyRegions,
                                    HostInfoType hostInfoType,
                                    List<Long> queryIds,
                                    Map<Long, String> queryId2name,
                                    DatePlotHelper<T> plotHelper,
                                    boolean includeSubregions,
                                    HistoryPlotDataProvider<T, HostInfoType> historyPlotDataProvider
    ) throws InternalException {
        List<NavigableMap<Date, T>> data = new ArrayList<NavigableMap<Date, T>>();
        List<String> labels = new ArrayList<String>();
        List<Long> selectedQueryIds = new ArrayList<Long>();
        for (Long queryId : queryIds) {
            if (queryId2name.get(queryId) != null) {
                NavigableMap<Date, T> map = historyPlotDataProvider.getHistoryPlotData(region, keyRegions, hostInfoType, queryId, includeSubregions);
                data.add(map);
                labels.add(queryId2name.get(queryId));
                selectedQueryIds.add(queryId);
            }
        }

        if (data.isEmpty()) {
            NavigableMap<Date, T> dummyMap = new TreeMap<Date, T>();

            Calendar cal = Calendar.getInstance();
            cal.setTime(SqlUtil.getMidnightTimestamp(null));

            dummyMap.put(cal.getTime(), null);
            cal.add(Calendar.DATE, -7);
            dummyMap.put(cal.getTime(), null);
            data.add(dummyMap);
        }

        outputData(res, plotHelper, data, labels, selectedQueryIds);
    }

    abstract protected Long[] getParamQueryId(ServRequest req);

    abstract protected Integer getParamRegion(ServRequest req) throws UserException;

    abstract protected Date getParamGraphDate(ServRequest req) throws UserException;

    abstract protected Boolean getParamIncludeSubregions(ServRequest req);

    abstract protected TopQueryHistoryTypeEnum getParamQueryHistoryType(ServRequest req);

    abstract protected <T> void outputData(ServResponse res, DatePlotHelper<T> plotHelper, List<NavigableMap<Date, T>> data, List<String> labels, List<Long> queryIds);
}
