package ru.yandex.wmtools.common.service;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.text.DateFormat;
import java.text.ParseException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.mail.internet.MailDateFormat;

import org.apache.http.entity.ByteArrayEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.InternalProblem;
import ru.yandex.wmtools.common.util.HttpConnector;
import ru.yandex.wmtools.common.util.HttpResponse;

/**
 * @author avhaliullin
 */
public class AbstractExternalHttpService implements IService {
    private static final Logger log = LoggerFactory.getLogger(AbstractExternalHttpService.class);

    protected static final int CONNECTION_TIMEOUT = 10000;
    protected static final String HEADER_LAST_MODIFIED = "Last-Modified";
    protected static final String HEADER_CONTENT_TYPE = "Content-Type";
    protected static final DateFormat DATE_FORMAT = new MailDateFormat();

    protected String externalServantUrl = null;

    public void setExternalServantUrl(String externalServantUrl) {
        this.externalServantUrl = externalServantUrl;
    }

    protected String urlEncode(String s) {
        try {
            return URLEncoder.encode(s, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new AssertionError(e);
        }
    }

    protected InternalException assertServantIsUnavailable(Throwable t) throws InternalException {
        return new InternalException(InternalProblem.SERVANT_FAILED, "External servant is unavailable", t);
    }

    protected InternalException assertServantIsUnavailable() throws InternalException {
        return new InternalException(InternalProblem.SERVANT_FAILED, "External servant is unavailable");
    }

    protected String getContentCharset(HttpResponse res) {
        String contentType = res.getContentType();
        if (contentType != null) {
            String[] tokens = contentType.split(";");
            for (String token : tokens) {
                token = token.trim();
                if (token.toLowerCase().startsWith("charset")) {
                    int pos = token.indexOf("=");
                    if (pos < 0 || token.length() < pos + 2) {
                        return null;
                    }
                    String charset = token.substring(pos + 1).trim();
                    if (Charset.isSupported(charset)) {
                        return charset;
                    } else {
                        return null;
                    }
                }
            }
        }
        return null;
    }

    protected HttpResponse httpPostResponse(String command, Map<String, Object> params, Integer socketTimeout, boolean okStatusRequired) throws InternalException {
        String urlAddress = externalServantUrl + "/" + command;
        try {
            StringBuilder stringEntity = new StringBuilder();
            for (Map.Entry<String, Object> param : params.entrySet()) {
                if (param.getValue() instanceof Collection) {
                    for (Object o : (Collection) param.getValue()) {
                        stringEntity
                                .append(param.getKey())
                                .append("=")
                                .append(URLEncoder.encode(o.toString(), "UTF-8"))
                                .append("&");
                    }
                } else {
                    stringEntity
                            .append(param.getKey())
                            .append("=")
                            .append(URLEncoder.encode(param.getValue().toString(), "UTF-8"))
                            .append("&");
                }
            }

            HttpResponse res = (socketTimeout == null ? new HttpConnector.RequestBuilder(new URL(urlAddress))
                    : new HttpConnector.RequestBuilder(new URL(urlAddress)).socketTimeout(socketTimeout))
                    .connectionTimeout(CONNECTION_TIMEOUT)
                    .method(HttpConnector.HttpMethod.POST)
                    .header(HEADER_CONTENT_TYPE, "application/x-www-form-urlencoded")
                    .entity(stringEntity.toString(), "windows-1251")
                    .allowRedirect(true)
                    .okStatusRequired(okStatusRequired)
                    .execute();
            logRequestTime(urlAddress, res);
            return res;
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
    }

    protected InputStream httpPostByteArrayResponse(
            String command,
            byte data[],
            int socketTimeout,
            boolean okStatusRequired) throws InternalException {
        String urlAddress = externalServantUrl + "/" + command;
        try {
            HttpResponse res = new HttpConnector.RequestBuilder(new URL(urlAddress))
                    .socketTimeout(socketTimeout)
                    .connectionTimeout(CONNECTION_TIMEOUT)
                    .method(HttpConnector.HttpMethod.POST)
                    .header(HEADER_CONTENT_TYPE, "application/x-www-form-urlencoded")
                    .entity(new ByteArrayEntity(data))
                    .allowRedirect(true)
                    .okStatusRequired(okStatusRequired)
                    .execute();
            logRequestTime(urlAddress, res);
            return res.getContent();

        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
    }

    protected HttpResponse httpPostBinaryResponse(String command, Map<String, Object> params, String charset, Integer socketTimeout, boolean okStatusRequired) throws InternalException {
        String urlAddress = externalServantUrl + "/" + command;
        try {
            StringBuilder stringEntity = new StringBuilder();
            String delimiter = "";
            for (Map.Entry<String, Object> param : params.entrySet()) {
                if (param.getValue() instanceof Collection) {
                    for (Object o : (Collection) param.getValue()) {
                        stringEntity
                                .append(delimiter)
                                .append(param.getKey())
                                .append("=")
                                .append(o.toString());
                    }
                } else {
                    stringEntity
                            .append(delimiter)
                            .append(param.getKey())
                            .append("=")
                            .append(param.getValue().toString());
                }
                delimiter = "&";
            }

            HttpResponse res = (socketTimeout == null ? new HttpConnector.RequestBuilder(new URL(urlAddress))
                    : new HttpConnector.RequestBuilder(new URL(urlAddress)).socketTimeout(socketTimeout))
                    .connectionTimeout(CONNECTION_TIMEOUT)
                    .method(HttpConnector.HttpMethod.POST)
                    .header(HEADER_CONTENT_TYPE, "application/x-www-form-urlencoded")
                    .entity(stringEntity.toString(), charset)
                    .allowRedirect(true)
                    .okStatusRequired(okStatusRequired)
                    .execute();
            logRequestTime(urlAddress, res);
            return res;
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
    }

    protected HttpResponse httpGetResponse(String command, Map<String, Object> params, Integer socketTimeout, boolean okStatusRequired) throws InternalException {
        return httpGetResponse(command, params, socketTimeout, okStatusRequired, null);
    }

    protected HttpResponse httpGetResponse(String command, Map<String, Object> params, Integer socketTimeout, boolean okStatusRequired, Map<String, Object> headers) throws InternalException {
        StringBuilder sb = new StringBuilder(externalServantUrl).append("/").append(command);
        boolean first = true;
        if (params != null) {
            sb.append("?");
            for (Map.Entry<String, Object> param : params.entrySet()) {
                sb.append(first ? "" : "&")
                        .append(urlEncode(param.getKey()))
                        .append("=")
                        .append(urlEncode(param.getValue().toString()));
                first = false;
            }
        }
        String urlAddress = sb.toString();
        try {
            HttpConnector.RequestBuilder builder = (socketTimeout == null ? new HttpConnector.RequestBuilder(new URL(sb.toString()))
                    : new HttpConnector.RequestBuilder(new URL(urlAddress)).socketTimeout(socketTimeout))
                    .connectionTimeout(CONNECTION_TIMEOUT)
                    .allowRedirect(true)
                    .okStatusRequired(okStatusRequired);
            if (headers != null) {
                for (Map.Entry<String, Object> header : headers.entrySet()) {
                    builder.header(header.getKey(), String.valueOf(header.getValue()));
                }
            }
            HttpResponse res = builder.execute();
            logRequestTime(urlAddress, res);
            return res;
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
    }

    protected HttpResponse httpPostXmlResponse(String command,
                                               Map<String, Object> params,
                                               Integer socketTimeout,
                                               boolean okStatusRequired,
                                               String xmlFile,
                                               String charset) throws InternalException {
        StringBuilder sb = new StringBuilder(externalServantUrl).append("/").append(command).append("?");
        for (Map.Entry<String, Object> param : params.entrySet()) {
            sb.append(param.getKey())
                    .append("=")
                    .append(param.getValue().toString())
                    .append("&");
        }
        String urlAddress = sb.toString();
        try {
            HttpResponse res = (socketTimeout == null ? new HttpConnector.RequestBuilder(new URL(sb.toString()))
                    : new HttpConnector.RequestBuilder(new URL(urlAddress)).socketTimeout(socketTimeout))
                    .connectionTimeout(CONNECTION_TIMEOUT)
                    .method(HttpConnector.HttpMethod.POST)
                    .header(HEADER_CONTENT_TYPE, "text/xml")
                    .entity(xmlFile, charset)
                    .allowRedirect(true)
                    .okStatusRequired(okStatusRequired)
                    .execute();
            logRequestTime(urlAddress, res);
            return res;
        } catch (IOException e) {
            throw assertServantIsUnavailable(e);
        }
    }

    protected InputStream httpGetResponse(String command, Map<String, Object> params, Integer socketTimeout) throws InternalException {
        return httpGetResponse(command, params, socketTimeout, null);
    }

    protected InputStream httpGetResponse(String command, Map<String, Object> params, Integer socketTimeout, Map<String, Object> headers) throws InternalException {
        return httpGetResponse(command, params, socketTimeout, true, headers).getContent();
    }

    protected Reader httpGetResponseReader(String command, Map<String, Object> params, Integer socketTimeout, Map<String, Object> headers) throws InternalException {
        HttpResponse resp = httpGetResponse(command, params, socketTimeout, true, headers);
        try {
            String charset = getContentCharset(resp);
            return new InputStreamReader(resp.getContent(), charset == null ? "utf-8" : charset);
        } catch (UnsupportedEncodingException e) {
            throw new InternalException(InternalProblem.CONNECTION_PROBLEM, "Unsupported encoding", e);
        }
    }

    protected Date getLastModifiedDate(HttpResponse resp) {
        Collection<String> headers = resp.getHeaders().get(HEADER_LAST_MODIFIED);
        if (headers == null) {
            return null;
        }
        for (String header : headers) {
            try {
                Date result = DATE_FORMAT.parse(header);
                if (result != null) {
                    return result;
                }
            } catch (ParseException e) {
                continue;
            }
        }
        return null;
    }

    protected InputStream httpPostResponse(String command, Map<String, Object> params, Integer socketTimeout) throws InternalException {
        return httpPostResponse(command, params, socketTimeout, true).getContent();
    }

    protected InputStream httpPostBinaryResponse(String command, Map<String, Object> params, String charset, Integer socketTimeout) throws InternalException {
        return httpPostBinaryResponse(command, params, charset, socketTimeout, true).getContent();
    }

    protected class Params extends HashMap<String, Object> {
        public Params addParam(String name, Object value) {
            put(name, value);
            return this;
        }
    }

    protected Params createParams() {
        return new Params();
    }

    protected String getDelimitedWith(List<String> arg, String delimiter) {
        StringBuilder sb = new StringBuilder();

        for (String s : arg) {
            sb.append(s).append(delimiter);
        }
        if (sb.length() > 0) {
            sb.setLength(sb.length() - delimiter.length());
        }
        return sb.toString();
    }


    protected String getDelimitedWith(String[] arg, String delimiter) {
        return getDelimitedWith(Arrays.asList(arg), delimiter);
    }

    private void logRequestTime(String reqUrl, HttpResponse res) {
        log.debug(getClass().getSimpleName() + ": Request time for URL = \"" + reqUrl + "\" was " + res.getRequestTime() + " ms");
    }
}
