package ru.yandex.wmtools.common.service;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.simple.ParameterizedRowMapper;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.common.util.db.BooleanRowMapper;
import ru.yandex.wmtools.common.data.partition.IPartition;
import ru.yandex.wmtools.common.error.InternalException;
import ru.yandex.wmtools.common.error.UserException;
import ru.yandex.wmtools.common.util.ServiceTransactionCallback;

/**
 * User: azakharov
 * Date: 03.10.12
 * Time: 11:43
 */
public class AbstractLockableDbService extends AbstractDbService {
    private static final Logger log = LoggerFactory.getLogger(AbstractLockableDbService.class);

    private static final String GET_LOCK_QUERY =
            "SELECT GET_LOCK(?, 0)";

    private static final String RELEASE_LOCK_QUERY =
            "SELECT RELEASE_LOCK(?)";

    private static final String LOCK_TABLE_NAME = "tbl_locks";

    private static final String SELECT_LOCK_QUERY =
            "SELECT id, name, locked, time FROM " + LOCK_TABLE_NAME + " WHERE name like ? FOR UPDATE";

    private static final String INSERT_LOCK_QUERY =
            "INSERT INTO " + LOCK_TABLE_NAME + " (name, locked, time) VALUES (?, true, NOW())";

    private static final String UPDATE_LOCK_QUERY =
            "UPDATE " + LOCK_TABLE_NAME + " SET locked = ?, time = NOW() WHERE name LIKE ?";

    private JdbcTemplate lockTemplate;

    public static class LockRecord {
        private final long id;
        private final String name;
        private final boolean locked;
        private final Date timestamp;

        public LockRecord(long id, String name, boolean locked, Date timestamp) {
            this.id = id;
            this.name = name;
            this.locked = locked;
            this.timestamp = timestamp;
        }

        public boolean isLocked() {
            return locked;
        }

        public Date getTimestamp() {
            return timestamp;
        }
    }

    private static final ParameterizedRowMapper<LockRecord> mapper = new ParameterizedRowMapper<LockRecord>() {
        @Override
        public LockRecord mapRow(ResultSet resultSet, int i) throws SQLException {
            return new LockRecord(resultSet.getLong("id"), resultSet.getString("name"),
                    resultSet.getBoolean("locked"),
                    resultSet.getTimestamp("time"));
        }
    };

    protected boolean getLock(final IPartition partition, final String lockName, final Integer autoReleaseAfterMinutes) {
        try {
            return getServiceTransactionTemplate(partition).executeInService(
                    new ServiceTransactionCallback<Boolean>() {
                        @Override
                        public Boolean doInTransaction(TransactionStatus transactionStatus) throws UserException, InternalException {
                            List<LockRecord> res = getJdbcTemplate(partition).query(
                                    SELECT_LOCK_QUERY,
                                    mapper, lockName);
                            if (res.isEmpty()) {
                                getJdbcTemplate(partition).update(
                                        INSERT_LOCK_QUERY, lockName);
                                return true;
                            }
                            LockRecord lock = res.iterator().next();
                            if (lock.isLocked()) {
                                if (autoReleaseAfterMinutes != null) {
                                    Calendar c = Calendar.getInstance();
                                    c.add(Calendar.MINUTE, -autoReleaseAfterMinutes);
                                    if (lock.getTimestamp() == null || c.getTime().after(lock.getTimestamp())) {
                                        log.warn("releasing lock " + lockName + " because of timeout: now is " +
                                                new Date() + " cal.getTime = " +
                                                c.getTime() + " lock time is " + lock.getTimestamp());
                                        getJdbcTemplate(partition).update(
                                                UPDATE_LOCK_QUERY, true, lockName);
                                        return true;
                                    }
                                }
                                return false;
                            } else {
                                getJdbcTemplate(partition).update(
                                        UPDATE_LOCK_QUERY, true, lockName);
                            }
                            return true;
                        }
                    });
        } catch (InternalException e) {
            log.warn("Can't get lock " + lockName, e);
            return false;
        } catch (UserException e) {
            log.warn("Can't get lock " + lockName, e);
            return false;
        }
    }

    protected boolean refreshLock(final IPartition partition, final String lockName) {
        log.warn("refreshing lock " + lockName);
        try {
            getJdbcTemplate(partition).update(UPDATE_LOCK_QUERY, true, lockName);
            return true;
        } catch (InternalException e) {
            log.warn("Can't refresh lock " + lockName, e);
            return false;
        }
    }

    protected boolean releaseLock(final IPartition partition, final String lockName) {
        try {
            return getServiceTransactionTemplate(partition).executeInService(
                    new ServiceTransactionCallback<Boolean>() {
                        @Override
                        public Boolean doInTransaction(TransactionStatus transactionStatus) throws UserException, InternalException {
                            List<LockRecord> res = getJdbcTemplate(partition).query(
                                    SELECT_LOCK_QUERY,
                                    mapper, lockName);
                            if (res.isEmpty()) {
                                return true;
                            }
                            LockRecord lock = res.iterator().next();
                            if (lock.isLocked()) {
                                getJdbcTemplate(partition).update(
                                        UPDATE_LOCK_QUERY, false, lockName);
                            }
                            return true;
                        }
                    });
        } catch (InternalException e) {
            log.warn("Can't release lock " + lockName, e);
            return false;
        } catch (UserException e) {
            log.warn("Can't release lock " + lockName, e);
            return false;
        }
    }

    protected boolean getLock(final String lockName) throws InternalException {
        final Boolean res;
        synchronized (lockTemplate) {
            try {
                res = lockTemplate.queryForObject(GET_LOCK_QUERY, new BooleanRowMapper(), lockName);
            } catch (DataAccessException e) {
                log.warn("Can't get lock " + lockName + " due to exception ", e);
                return false;
            }
        }
        return res == null ? false : res;
    }

    protected Boolean releaseLock(final String lockName) throws InternalException{
        synchronized (lockTemplate) {
            try {
                return lockTemplate.queryForObject(RELEASE_LOCK_QUERY, new BooleanRowMapper(), lockName);
            } catch (DataAccessException e) {
                log.warn("Exception while releasing lock " + lockName, e);
                return false;
            }
        }
    }

    public void setLockTemplate(JdbcTemplate lockTemplate) {
        this.lockTemplate = lockTemplate;
    }
}
