package ru.yandex.wmtools.common.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.jdom.Document;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.common.framework.user.UserInfo;
import ru.yandex.common.framework.user.UserInfoField;
import ru.yandex.common.framework.user.blackbox.BlackBoxField;
import ru.yandex.common.framework.user.blackbox.BlackBoxUserInfo;
import ru.yandex.common.util.collections.Pair;
import ru.yandex.wmtools.common.error.BlackboxProblem;
import ru.yandex.wmtools.common.error.InternalException;

/**
 * @author avhaliullin
 */
public class BlackboxUserService extends AbstractBlackboxService implements UserService {
    private static final Logger log = LoggerFactory.getLogger(BlackboxUserService.class);

    private UserInfoRequestHandler userInfoRequestHandler;

    private List<String> dbFields = Arrays.asList("account_info.fio.uid",
            "account_info.sex.uid",
            "account_info.country.uid",
            "account_info.city.uid",
            "accounts.login.uid");

    private Map<String, String> getCheckAndParseResponse(Pair<IdentificationMethod, String> id, List<String> dbFields) {
        try {
            Pair<BlackboxProblem, Document> resp = getResponse(id, Method.USER_INFO,
                    //TODO: workaround for https://st.yandex-team.ru/PASSP-9297
                    createParams().addParam("sid", "passport"),
                    2, dbFields);

            if (!BlackboxProblem.OK.equals(resp.first)) {
                throw new RuntimeException("Blackbox error: " + resp.first.name());
            }
            return parseResponse(resp.second);
        } catch (InternalException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public UserInfo getUserInfo(long uid) {
        Map<String, String> fields = getCheckAndParseResponse(
                Pair.of(IdentificationMethod.UID, String.valueOf(uid)),
                dbFields);

        String uidString = fields.get(customTagKey("uid"));
        BlackBoxUserInfo info = null;

        if (!uidString.isEmpty()) {
            info = new BlackBoxUserInfo(Long.parseLong(uidString));
            for (Map.Entry<String, String> field : fields.entrySet()) {
                BlackBoxField bbField = BlackBoxField.byCode(field.getKey());
                if (bbField != null) {
                    info.addField(bbField.getUserInfoField(), field.getValue());
                }
            }
        }

        if (userInfoRequestHandler != null) {
            try {
                userInfoRequestHandler.handleRequest(uid, info);
            } catch (RuntimeException e) {
                log.warn("User info handler failed with error", e);
            }
        }

        return info;
    }

    @Override
    public Long getUidByLogin(String login) {
        Map<String, String> fields = getCheckAndParseResponse(
                Pair.of(IdentificationMethod.LOGIN, login), new ArrayList<String>(0));
        String uidString = fields.get(customTagKey("uid"));
        return uidString.isEmpty() ? null : Long.parseLong(uidString);
    }

    public static UserInfo createUserInfoStub(long userId, String login, String fio) {
        BlackBoxUserInfo userInfo = new BlackBoxUserInfo(userId);
        userInfo.addField(UserInfoField.LOGIN, login);
        userInfo.addField(UserInfoField.FIO, fio);
        return userInfo;
    }

    public void setDbFields(List<String> dbFields) {
        if (dbFields != null) {
            this.dbFields = dbFields;
        }
    }

    public interface UserInfoRequestHandler {
        public void handleRequest(Long userId, UserInfo response);
    }

    public void setUserInfoRequestHandler(UserInfoRequestHandler userInfoRequestHandler) {
        this.userInfoRequestHandler = userInfoRequestHandler;
    }

}
